%% Analyze Fluorescent-Force-Gliding Assay
    %% 1) Import positions from TrackMate and plot (Direction of Mic movement is important!)
    clear all; fclose all; close all;
    
    % Parameters
    UseGaussianFit = 0;     % 1 to apply GaussianFit, 0 if just using TrackMate
    
    % Define all suffixes
    KinFileSuffix = '(Kin).tif';
    MicFileSuffix = '(Mic).tif';
    TrackMateFileSuffix = '(KinTrackMateSpots).csv';
    
    TextSize = 16;
    InputName = dir(['*' TrackMateFileSuffix]);
    InputName = InputName.name;
    disp('Downloading csv file...');
    Input = csvread(InputName,1,3);
    disp('Csv file downloaded...');
    Input = Input(:,1:6);
    Input = sortrows(Input,6);
    
    % Ask for pixel size
    PixelTime = inputdlg({'Pixel size (nm):','Exposure Time (s):'},'Parameters Input',1,{'160','0.2'});
    PixelSize = str2num(PixelTime{1}); ExposureTime = str2num(PixelTime{2});
    %PixelSize = 106.7; ExposureTime = 0.1;
    
    % Assign TrackID, xPos, yPos and tPos
    TrackID = Input(:,1);
    xPos = Input(:,3);
    yPos = Input(:,4);
    tPos = Input(:,6)*ExposureTime;

    % Find files with RedFileInfix, then find corresponding green files
    KinFile = strrep(InputName,TrackMateFileSuffix,KinFileSuffix);
    MicFile = strrep(InputName,TrackMateFileSuffix,MicFileSuffix);

    KinInputinfo = imfinfo(KinFile);
    MicInputinfo = imfinfo(MicFile);
    dim = [KinInputinfo(1).Height KinInputinfo(1).Width length(KinInputinfo)];

    KinInput = uint16(zeros(dim));
    MicInput = uint16(zeros(dim));
    
    % Get arrays
    for i = 1:dim(3)
        KinInput(:,:,i)=imread(KinFile, i, 'Info', KinInputinfo);
        MicInput(:,:,i)=imread(MicFile, i, 'Info', MicInputinfo);
    end
    KinInputMean = imadjust(uint16(round(mean(KinInput,3))));
    MicInputMean = imadjust(uint16(round(mean(MicInput,3))));

    % Plot figure with all Kin coordinates and Mic and Kin images
    screenSize = get(0,'screensize');
    screenWidth = screenSize(3);
    screenHeight = screenSize(4);
    hFig = figure('units','pixels','outerposition',[0.2*screenWidth 0*screenHeight 0.8*screenWidth 1*screenHeight]);
    fpos = get(hFig,'Position');
    Scaling = min([fpos(3)/size(KinInputMean,2) fpos(4)/size(KinInputMean,1)])*0.90;
    axOffset = (fpos(3:4)-[size(KinInputMean,2)*Scaling size(KinInputMean,1)*Scaling])/2;
    ha = axes('Parent',hFig,'Units','pixels',...
            'Position',[axOffset size(KinInputMean,2)*Scaling size(KinInputMean,1)*Scaling]);
    imshow(cat(3,KinInputMean+MicInputMean,KinInputMean+MicInputMean,KinInputMean+MicInputMean),'Parent',ha); hold on;
    TrackNo = length(unique(TrackID));
    
    % Specify the number of colors to plot to let us get a sense of time
    NoOfPlots = 4;      % How many colors we want to have
    PointSpacing = floor(length(xPos)/NoOfPlots);

    % Plot figure with point positions
    disp('Plotting points...');
    cmap = colormap(hsv(NoOfPlots));
    ColorID = zeros(length(xPos),1);
    for i = 1:NoOfPlots
        if i == NoOfPlots
            plot(xPos((i-1)*PointSpacing+1:length(xPos)),yPos((i-1)*PointSpacing+1:length(xPos)),'.','Color',cmap(i,:),'MarkerSize',5,'Parent',ha); hold on;
            ColorID((i-1)*PointSpacing+1:length(xPos)) = i;
        else
            plot(xPos((i-1)*PointSpacing+1:i*PointSpacing),yPos((i-1)*PointSpacing+1:i*PointSpacing),'.','Color',cmap(i,:),'MarkerSize',5,'Parent',ha); hold on;
            ColorID((i-1)*PointSpacing+1:length(xPos)) = i;
        end
    end
    colorbar;
    set(ha,'Position',[axOffset size(KinInputMean,2)*Scaling size(KinInputMean,1)*Scaling]);
    disp('Done plotting points...');

    % Initialize TraceMatrix, TraceIndex and figure handles
    TraceMatrix = cell(40,1);
    TraceIndex = 1;
    hBox = zeros(40,1);
    hText = zeros(40,1);
    
    % Start loop to choose points
    StopLoop = 0;
    while StopLoop == 0
        % Get polyline
        pan on;
        disp(['Please draw region ' num2str(TraceIndex) '. Double click when done to close polygon.']);
        h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK'), 'Zoom/ Pan');
        set(h, 'Position', [50 250 250 60]);
        uiwait(h); 
        choice = questdlg('Continue choosing regions?.','Continue?', ...
                'Yes','No','Redo','Yes');

        switch choice
            case 'Yes'
                % Get polyline
                disp(['Please draw region ' num2str(TraceIndex) '. Double click when done to close polygon.']);
                [xBorder,yBorder] = getline(gca);

                % Draw polyline
                hBox(TraceIndex) = plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'r'); 

                % Find points within polyline
                BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                % Combine all points to one trace
                Trace = [TrackID(BoundPoints) xPos(BoundPoints) yPos(BoundPoints) tPos(BoundPoints) ColorID(BoundPoints)];
                Trace = sortrows(Trace,4);
                
                % Find duplicated values in x, y and time, then delete
                [~,index,~] = unique(Trace(:,2:4),'rows');
                Trace = Trace(index,:);
                Trace = sortrows(Trace,4);

                % Find duplicated values in time
                UniqueTime = unique(Trace(:,4));
                [n, bin] = histc(Trace(:,4), UniqueTime);
                multiple = find(n > 1);     % Indexes of UniqueTime that are repeated
                
                % Initialize UndoMatrix and UndoIndex
                UndoMatrix = cell(40,1);
                UndoIndex = 1;
                
                % Delete repeated time points if there are multiple ones
                if ~isempty(multiple)
                    % If there is duplicated value, proceed to plot x vs t and
                    % y vs t to delete duplicates
                    disp('Duplicate times detected. Choose points to keep.');
                    
                    % Plot traces with duplicated values marked
                    hDuplicate = figure('units','normalized','position',[0.2 0.01 0.8 0.9]); 
                    subplot(2,1,1); 
                    for i = 1:NoOfPlots
                        plot(Trace(Trace(:,5)==i,4),Trace(Trace(:,5)==i,2),'.','Color',cmap(i,:),'MarkerSize',5); hold on;
                    end
                    xlim([-max(Trace(:,4))*0.05 max(Trace(:,4))*1.05]);
                    xDiff = (max(Trace(:,2))-min(Trace(:,2)))*0.15;
                    ylim([min(Trace(:,2))-xDiff max(Trace(:,2))+xDiff]);
                    for i = 1:length(multiple)
                        index = find(ismember(bin, multiple(i)));
                        xLine = [min(Trace(index,2)) max(Trace(index,2))];
                        plot([Trace(index(1),4) Trace(index(1),4)],[xLine(1) xLine(2)],'r');
                        for j = 1:length(index)
                            text(Trace(index(1),4), Trace(index(j),2),num2str(j));
                        end
                        text(Trace(index(1),4), max(Trace(:,2))+xDiff/2,['(' num2str(i) ')'],'FontWeight','b');
                    end
                    title('x vs time (Duplicate times detected)');
                    xlabel('Time (s)');
                    ylabel('x (pixel)');

                    subplot(2,1,2); 
                    for i = 1:NoOfPlots
                        plot(Trace(Trace(:,5)==i,4),-Trace(Trace(:,5)==i,3),'.','Color',cmap(i,:),'MarkerSize',5); hold on;
                    end
                    xlim([-max(Trace(:,4))*0.05 max(Trace(:,4))*1.05]);
                    yDiff = (max(Trace(:,3))-min(Trace(:,3)))*0.15;
                    ylim([min(-Trace(:,3))-yDiff max(-Trace(:,3))+yDiff]);
                    for i = 1:length(multiple)
                        index = find(ismember(bin, multiple(i)));
                        yLine = [min(-Trace(index,3)) max(-Trace(index,3))];
                        plot([Trace(index(1),4) Trace(index(1),4)],[yLine(1) yLine(2)],'r');
                        for j = 1:length(index)
                            text(Trace(index(1),4), -Trace(index(j),3),num2str(j));
                        end
                        text(Trace(index(1),4), max(-Trace(:,3))+yDiff/2,['(' num2str(i) ')'],'FontWeight','b');
                    end
                    title('y vs time (Duplicate times detected)');
                    xlabel('Time (s)');
                    ylabel('y (pixel)');

                    % Ask for Duplicates to keep
                    DuplicateNo = length(multiple);
                    TitleDialog = cell(1,DuplicateNo);
                    StartingValues = cell(1,DuplicateNo);
                    for i = 1:DuplicateNo
                        TitleDialog(i) = {['Time (' num2str(i) '): Point to keep:']};
                        StartingValues(i) = {num2str([])};
                    end

                    % Specify options for inputdlg
                    options.Resize = 'off'; options.WindowStyle = 'normal'; options.Interpreter = 'none';
                    
                    if DuplicateNo > 10
                        % Determine how many times we need to loop
                        PointsToKeep = cell(1,DuplicateNo);
                        for i = 1:ceil(DuplicateNo/10)
                            if i == ceil(DuplicateNo/10)
                                PointsToKeep((i-1)*10+1:DuplicateNo) = inputdlg(TitleDialog((i-1)*10+1:DuplicateNo),'Parameters Input',1,StartingValues((i-1)*10+1:DuplicateNo),options);
                            else
                                PointsToKeep((i-1)*10+1:i*10) = inputdlg(TitleDialog((i-1)*10+1:i*10),'Parameters Input',1,StartingValues((i-1)*10+1:i*10),options);
                            end
                        end
                    else
                        PointsToKeep = inputdlg(TitleDialog,'Parameters Input',1,StartingValues,options);
                    end

                    % Delete points not chosen in PointsToKeep
                    for i = length(multiple):-1:1
                        index = find(ismember(bin, multiple(i)));
                        index(str2num(PointsToKeep{i})) = [];
                        UndoMatrix{UndoIndex} = [UndoMatrix{UndoIndex}; Trace(index,:)];
                        Trace(index,:) = [];
                    end
                    
                    close(hDuplicate);
                    
                    % Increase UndoIndex
                    UndoIndex = UndoIndex + 1;
                end
                
                % Delete points if necessary
                disp('Delete points if necessary. Draw around points to delete');
                StopLoop2 = 0;
                while StopLoop2 == 0
                    % Plot traces with duplicated values marked
                    hDelete = figure('units','normalized','position',[0.2 0.01 0.8 0.9]); 
                    subplot(4,2,[1,3]);
                    for i = 1:NoOfPlots
                        plot3(Trace(Trace(:,5)==i,2),-Trace(Trace(:,5)==i,3),Trace(Trace(:,5)==i,4),'.','Color',cmap(i,:),'MarkerSize',5); hold on;
                    end
                    view(0,90)
                    title('y vs x (3D plot - Rotatable)');
                    xlabel('x (pixel)');
                    ylabel('y (pixel)');
                    
                    subplot(4,2,[2,4]);
                    for i = 1:NoOfPlots
                        plot(Trace(Trace(:,5)==i,2),-Trace(Trace(:,5)==i,3),'.','Color',cmap(i,:),'MarkerSize',5); hold on;
                    end
                    view(0,90)
                    title('y vs x (Deletable)');
                    xlabel('x (pixel)');
                    ylabel('y (pixel)');
                    
                    subplot(4,2,5:6);
                    for i = 1:NoOfPlots
                        plot(Trace(Trace(:,5)==i,4),Trace(Trace(:,5)==i,2),'.','Color',cmap(i,:),'MarkerSize',5); hold on;
                    end
                    xlim([-max(Trace(:,4))*0.05 max(Trace(:,4))*1.05]);
                    xDiff = (max(Trace(:,2))-min(Trace(:,2)))*0.15;
                    ylim([min(Trace(:,2))-xDiff max(Trace(:,2))+xDiff]);
                    title('x vs time (Deletable)');
%                     xlabel('Time (s)');
                    ylabel('x (pixel)');

                    subplot(4,2,7:8); 
                    for i = 1:NoOfPlots
                        plot(Trace(Trace(:,5)==i,4),-Trace(Trace(:,5)==i,3),'.','Color',cmap(i,:),'MarkerSize',5); hold on;
                    end
                    xlim([-max(Trace(:,4))*0.05 max(Trace(:,4))*1.05]);
                    yDiff = (max(Trace(:,3))-min(Trace(:,3)))*0.15;
                    ylim([min(-Trace(:,3))-yDiff max(-Trace(:,3))+yDiff]);
                    title('y vs time (Deletable)');
                    xlabel('Time (s)');
                    ylabel('y (pixel)');
                    
                    h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK.\nClick on the subplot to delete before clicking OK'), 'Zoom/ Pan');
                    set(h, 'Position', [50 250 250 60]);
                    uiwait(h); 
                    choice2 = questdlg('Continue deleting points?.','Continue?', ...
                            'Yes','No','Redo','Yes');

                    switch choice2
                        case 'Yes'
                            % Delete other points
                            figure(hDelete);
                            h = imfreehand;
                            Borders = getPosition(h);
                            delete(h);
                            
                            ActiveAxes = get(gca,'Position');
                            if ActiveAxes(1) > 0.5
                                if ActiveAxes(2) > 0.5
                                    % Find points within border
                                    BoundPoints = inpolygon(Trace(:,2),-Trace(:,3),Borders(:,1),Borders(:,2));
                                end
                            else
                                if ActiveAxes(2) > 0.5
                                    % Find points within border
                                    BoundPoints = inpolygon(Trace(:,4),Trace(:,2),Borders(:,1),Borders(:,2));
                                else
                                    % Find points within border
                                    BoundPoints = inpolygon(Trace(:,4),-Trace(:,3),Borders(:,1),Borders(:,2));
                                end 
                            end

                            % Delete all BoundPoints and store in
                            % UndoMatrix
                            UndoMatrix{UndoIndex} = [UndoMatrix{UndoIndex}; Trace(BoundPoints,:)];
                            Trace(BoundPoints,:) = [];
                            
                            % Increase UndoIndex
                            UndoIndex = UndoIndex + 1;
                        case 'No'
                            StopLoop2 = 1;
                        case 'Redo'
                            % Decrease UndoIndex
                            UndoIndex = UndoIndex - 1;
                            
                            % Restore Trace
                            Trace = [Trace; UndoMatrix{UndoIndex}];
                            Trace = sortrows(Trace,4);
                    end
                    
                    % Close hDelete
                    close(hDelete);
                end
                    
%                 % If there is duplicated value, ask for a reference point
%                 % for which points further from this point will be deleted
%                 if ~isempty(multiple)
%                     dcm_obj=datacursormode;
%                     h = msgbox('Overlapping points detected. Choose one data point closest to the data points to be kept. Other points at the same time will be deleted. Click OK when data point is selected.','Overlapping points detected');
%                     set(h, 'Position', [50 250 280 70]);
%                     uiwait(h);
%                     Cursor=getCursorInfo(dcm_obj);
%                     
%                     while isempty(Cursor)
%                         h = msgbox('Overlapping points detected. Choose one data point closest to the data points to be kept. Other points at the same time will be deleted. Click OK when data point is selected.','Overlapping points detected');
%                         set(h, 'Position', [50 250 280 70]);
%                         uiwait(h);
%                         Cursor=getCursorInfo(dcm_obj);
%                     end
%                     
%                     Cursor = [Cursor.Position(1) Cursor.Position(2)];
%                     
%                     % Delete duplicated data other than the first one
%                     for i = length(multiple):-1:1
%                         index = find(ismember(bin, multiple(i)));
%                         
%                         % Initialize Dist Matrix to find distances from
%                         % reference point
%                         Dist = zeros(length(index),1);
%                         
%                         % Loop to find Dist
%                         for ii = 1:length(index)
%                             Dist(ii) = (Trace(index(ii),2)-Cursor(1))^2 + (Trace(index(ii),3)-Cursor(2))^2;
%                         end
%                         
%                         [~,minInd] = min(Dist);
%                         
%                         % Delete all except index with smallest distance
%                         Trace(index(minInd+1:end),:)=[];
%                         Trace(index(1:minInd-1),:)=[];
%                     end
%                     
%                     % Remove datatip
%                     dcm_obj.removeAllDataCursors();
%                 end
% 
                hText(TraceIndex) = text(max(xBorder),min(yBorder),num2str(TraceIndex),'Color','r','FontSize',20,'fontweight','b');

                TraceMatrix{TraceIndex} = Trace(:,1:4);        
                TraceIndex = TraceIndex + 1;
            case 'No'
                StopLoop = 1;
            case 'Redo'
                % Undo record on TraceIndex and TraceMatrix
                TraceIndex = TraceIndex - 1;
                TraceMatrix{TraceIndex} = [];
                
                % Delete previous text and trace plot
                delete(hBox(TraceIndex));
                delete(hText(TraceIndex));
        end
    end

    % Zoom out, save image with spot regions, then close
    zoom out;

    % Omit empty TraceMatrix elements
    TraceMatrix = TraceMatrix(~cellfun(@isempty,TraceMatrix));
    RegionNo = size(TraceMatrix,1);
    
    % Fit with 2d Gaussian if asked
    if UseGaussianFit == 1
        % Initialize RegionMatrix and TiffMatrix
        RegionMatrix = cell(length(TraceMatrix),1);
        TiffMatrix = cell(length(TraceMatrix),1);
        CroppedTraceMatrix = TraceMatrix;
        MaxFrame = 1;
        xytLimits = zeros(length(TraceMatrix),3);

        % Find regions to crop
        Radius = 6;
        for i = 1:size(TraceMatrix,1)
            Trace = TraceMatrix{i}(:,2:4);
            xLimits = [max(floor(min(Trace(:,1)))-Radius,1) min(ceil(max(Trace(:,1)))+Radius,dim(2))];
            yLimits = [max(floor(min(Trace(:,2)))-Radius,1) min(ceil(max(Trace(:,2)))+Radius,dim(1))];
            tLimits = round([min(Trace(:,3)) max(Trace(:,3))]/ExposureTime) + 1;
            TiffMatrix{i} = uint16(zeros(diff(yLimits)+1,diff(xLimits)+1,diff(tLimits)+1));
            RegionMatrix{i} = [yLimits; xLimits; tLimits];

            CroppedTraceMatrix{i}(:,2:3) = [TraceMatrix{i}(:,2)-xLimits(1)+1 TraceMatrix{i}(:,3)-yLimits(1)+1];

            % Interpolate to fill every frames with points
            t = Trace(:,3);
            tq = (min(t):ExposureTime:max(t))';
            xq = interp1(t,CroppedTraceMatrix{i}(:,2),tq);
            yq = interp1(t,CroppedTraceMatrix{i}(:,3),tq);

            CroppedTraceMatrix{i} = [TraceMatrix{i}(1,1)*ones(length(xq),1) xq yq tq];
            MaxFrame = max(MaxFrame,max(tq)/ExposureTime);
            xytLimits(i,:) = [xLimits(1) yLimits(1) tLimits(1)];
        end

%         % Find out the maximum frame index from each file
%         FileLengthsSum = cumsum(FileLengths);

%         FrameIndex = ones(size(TraceMatrix,1),1);
% 
%         % Get regions from tiff files
%         tic;
%         for i = 1:max(FileLengthsSum)
%             % Identify which file to use
%             ind = min(find(FileLengthsSum>=i));
%             if ind > 1
%                 idx = i-FileLengthsSum(ind-1);
%             else
%                 idx = i;
%             end
%             data = imread(FileInput{ind},idx, 'Info', info{ind});
% 
%             for ii = 1:size(TraceMatrix,1)
%                 if i >= RegionMatrix{ii}(3,1) && i <= RegionMatrix{ii}(3,2)
%                     TiffMatrix{ii}(:,:,FrameIndex(ii)) = data(RegionMatrix{ii}(1,1):RegionMatrix{ii}(1,2),RegionMatrix{ii}(2,1):RegionMatrix{ii}(2,2));
%                     FrameIndex(ii) = FrameIndex(ii)+1;
%                 end
%             end
% 
%             if mod(i,100)==0
%                 disp(['Frame ' num2str(i) '/' num2str(max(FileLengthsSum)) ' cropped. Elapsed time: ' num2str(floor(toc)) ' s']);
%             end
%         end
        
        for ii = 1:size(TraceMatrix,1)
            TiffMatrix{ii} = KinInput(RegionMatrix{ii}(1,1):RegionMatrix{ii}(1,2),RegionMatrix{ii}(2,1):RegionMatrix{ii}(2,2),RegionMatrix{ii}(3,1):RegionMatrix{ii}(3,2));
        end
        
        % Add paths to initialize ImageJ
        DataPath = pwd;
        javaaddpath('F:\MATLAB\R2016b\java\jar\mij.jar');       % Path where mij.jar is
        addpath('C:\Users\tjioe2\Desktop\Unused Shortcuts\Fiji\Fiji.app\scripts');   % Path for matlab scripts in FIJI
        Miji;       % Start ImageJ through Matlab. See http://javadoc.imagej.net/Fiji/MIJ.html for command details
        cd(DataPath);

        % Create new folder if none exists
        TiffName = 'Cropped Kin Tiff';
        if exist(TiffName,'dir')~=7; mkdir(TiffName); end
        xytOffset = zeros(size(TraceMatrix,1),3);

        % Edit and save all cropped files
        for i = 1:size(TiffMatrix,1)
            MIJ.createImage('Original',TiffMatrix{i},1);
            for ii = 1:8
                MIJ.run('In [+]');
            end

            prompt = {['Click <Cancel> to discard file.' newline 'Draw a rectangle to crop image.' newline 'Start Frame:'],'End Frame:'};
            dlg_title = 'Save File?';
            num_lines = 1;
            defaultans = {'1',num2str(size(TiffMatrix{i},3))};
            answer = inputdlg(prompt,dlg_title,num_lines,defaultans);

            if ~isempty(answer)
                try
                    roi = MIJ.getRoi(1);
                    CroppedTraceMatrix{i}(:,2:3) = [CroppedTraceMatrix{i}(:,2)-roi(1,1) CroppedTraceMatrix{i}(:,3)-roi(2,1)];
                    xytOffset(i,1:2) = [roi(1,1) roi(2,1)];
                    MIJ.setRoi(roi,2);      % Set ROI
                catch
                    roi = [];
                end

                Start = str2num(answer{1}); End = str2num(answer{2});
                xytOffset(i,3) = Start;
                CroppedTraceMatrix{i} = CroppedTraceMatrix{i}(Start:End,:);

                MIJ.run('Duplicate...', ['title=[' TiffName '-' num2str(i) '] duplicate range=' num2str(Start) '-' num2str(End)]);
                for ii = 1:8
                    MIJ.run('In [+]');
                end
                MIJ.selectWindow('Original');
                MIJ.run('Close');
                MIJ.selectWindow([TiffName '-' num2str(i)]);

                % Give a warning dialog to allow preview before saving
                h=warndlg('OK to save cropped file');
                %set(h, 'Position', [50 500 140 60]);
                uiwait(h);
                MIJ.selectWindow([TiffName '-' num2str(i)]);

                % Save image and xlsx file
                MIJ.run('Save', ['save=[' TiffName '\' TiffName '-' num2str(i) '.tif]']);
                xlswrite([TiffName '\' TiffName '-' num2str(i) '.xlsx'],CroppedTraceMatrix{i});

                % Close file
                MIJ.selectWindow([TiffName '-' num2str(i) '.tif']);
                MIJ.run('Close');
            else
                MIJ.selectWindow('Original');
                MIJ.run('Close');
            end
        end

        MIJ.closeAllWindows;    % Close all windows
        MIJ.exit;               % Exit ImageJ
        
        % Apply FIONA
        FIONAPositions = cell(size(TraceMatrix,1),1);
        
        % Generate warning and ask for code path if we cannot find code in the active
        % folder and a few folder above
        if isempty(dir('gauss2dfunct.m'))
            cd ..;      % Go to one folder up and look for gauss2dfunct.m there   
            if isempty(dir('gauss2dfunct.m'))
                cd ..;      % Go to one folder up and look for gauss2dfunct.m there
                if isempty(dir('gauss2dfunct.m'))
                    cd ..       % Go to one folder up and look for gauss2dfunct.m there
                    if isempty(dir('gauss2dfunct.m'))
                        % Generate warning and ask for the code path
                        h=warndlg('Cannot find gauss2dfunct.m in active folder. Locate folder with gauss2dfunct.m in the next window');
                        uiwait(h);
                        CodePath = uigetdir(DataPath,'Select folder containing gauss2dfunct');
                        cd(CodePath);
                    else; CodePath = pwd; 
                    end;
                else; CodePath = pwd; 
                end;
            else; CodePath = pwd; 
            end;
        else; CodePath = pwd; 
        end;

        % Sum frames to lengthen exposure time
        SumFactor = 1;      % Number of frames to be added together. More frames added together is akin to doing experiment at longer exposure time
        UsePos = 0;         % 1 to use available position information. 0 to skip
        Use1dFit = 1;       % 1 to use gaussian 1d to estimate initial parameter (~2x slower, but slightly more accurate). 0 to skip 1d fit
        batchSize = 500;     % The amount of files to be put in group to speed up calculation
        DisplayMultiple = 1000;     % Number of frames analyzed before reporting to command window

        % List of parameters
        ActPix = 1;             % Actual pixel size in nanometer
        ObjMag = 1;               % Objective magnification
        AddMag = 1;               % Additional magnification

        CCDsens = 12.13;             % CCD sensitivity of the camera at specific readout rate and pre-amp setting. See note below for values
        EMgain = 40;                % Electron multiplying (EM) gain setting of camera during acquisition

        AskPath = 0;                % (0 or 1) Prompt Matlab to start a dialog box asking directory
%         DataPath = 'C:\Users\';     % Path used if AskPath is zero, for automation purposes
        FileType = '.tif';          % Input file type. Can be changed to other file type

        % Get directories
        if AskPath == 1
            DataPath = uigetdir(CodePath,'Select folder with tiff files to be analyzed');
        end
        cd([DataPath '\' TiffName]);

        % Get all the file names and assign them under FileInput
        FileIn = dir(['*' FileType]);
        FileInput = cell(length(FileIn),1);
        FileInputName = cell(length(FileIn),1);
        for ind=1:length(FileIn)
            FileInput{ind}=FileIn(ind).name;
            FileInputName{ind}=strrep(FileIn(ind).name,FileType,'');
        end

        % Applying FIONA to each file and generate output file
        PixSize = ActPix/(ObjMag*AddMag);     % Pixel size in nanometer
        CountToPhoton = CCDsens/EMgain;         % Count to photon conversion
        a2 = PixSize*PixSize;               % Square of PixelSize

        % Looping over all files
        for ind=1:length(FileIn)
            % Getting info from image to make it faster to upload multiple frames
            cd([DataPath '\' TiffName]);
            info = imfinfo(FileInput{ind});
            FIONAdim = [info(1).Height info(1).Width];
            num_images = numel(info);
            FinalFrameNo = floor(num_images/SumFactor);

            if UsePos == 1
                % Input *.xlsx position data
                Position = xlsread([FileInputName{ind} '.xlsx']);
            end

            % Allocating memory to variables
            Output = zeros(FinalFrameNo,8);

            % Initialize DisplayCount
            DisplayCount = 1;

            tic;
            % Loop over all frames
            for k = 1:ceil((FinalFrameNo-1)/batchSize)
                if k==ceil((FinalFrameNo-1)/batchSize)
                    dataSize = rem(FinalFrameNo,batchSize);
                    if dataSize == 0
                        dataSize = batchSize;
                    end
                else
                    dataSize = batchSize;
                end
                data = double(zeros(FIONAdim(1),FIONAdim(2),dataSize));

                for kk = 1:dataSize
                    FrameIndex = (k-1)*batchSize + kk;
                    dataTemp = double(zeros(FIONAdim(1),FIONAdim(2)));
                    for j = 1:SumFactor
                        dataTemp = dataTemp+double(imread(FileInput{ind}, (SumFactor*(FrameIndex-1))+j, 'Info', info));
                    end
                    data(:,:,kk) = dataTemp/SumFactor;
                end

                [ny,nx,nz]=size(data);                 % Find out the dimensions
                %grid = [nx ny 1:nx 1:ny];         % Gridding input for gauss2dfunct and gauss2dfit
                tilt=0;

                if UsePos == 1
                    Pos = Position(SumFactor*(k-1)*batchSize+1:SumFactor*((k-1)*batchSize+dataSize),1:2);
                    if SumFactor > 1
                        Pos = [mean(reshape(Pos(:,1),SumFactor,[]))' mean(reshape(Pos(:,2),SumFactor,[]))'];
                    end
                else
                    Pos = [];
                end

                % Change to code directory
                cd(CodePath);

                % Parameters: p(1): z-offset, p(2): amplitude, p(3): xStdev, p(4): yStdev, p(5): xCenter, p(6): yCenter, p(7): tilt.
                [popt,resnorm,residual,ret]=Batchgauss2dfit(data,tilt,Use1dFit,Pos);

                % Getting center and precision
                xCenter = popt(:,5)*PixSize;        % Center of x in nanometer
                yCenter = popt(:,6)*PixSize;        % Center of y in nanometer

                sx2 = popt(:,3).*popt(:,3)*PixSize^2;   % Square of xStdev
                sy2 = popt(:,4).*popt(:,4)*PixSize^2;   % Square of yStdev
                sx4 = (popt(:,3)*PixSize).^4;   % xStdev^4
                sy4 = (popt(:,4)*PixSize).^4;   % yStdev^4

                % To estimate b, which is the standard deviation of the background,
                % we'll look at the z-offset (popt(1)) and calculate the standard
                % deviation based on anything below the z-offset. Before that we
                % would want to make everything 4 standard deviations away from
                % our spot to be (z-offset + 1) or NaN.

                % Find the limits of the data
                xmin = max(floor(popt(:,5)-4*popt(:,3)),1);
                xmax = min(ceil(popt(:,5)+4*popt(:,3)),nx);
                ymin = max(floor(popt(:,6)-4*popt(:,4)),1);
                ymax = min(ceil(popt(:,6)+4*popt(:,4)),ny);

                % Initialize xPrecision and yPrecision
                b = zeros(nz,1);
                RSquare = zeros(nz,1);

                for kk = 1:nz
                    z = reshape(data(:,:,kk),ny*nx,1); 
                    SumSquaresTotal = sum(z.*z);
                    RSquare(kk) = 1-(resnorm(kk)/SumSquaresTotal);

                    %residual = data - zfit;                % Compute residual
                    if ~isempty(residual{kk})
                        res = reshape(residual{kk},ny,nx);     % Extract residual
                        res = -res;                   % Invert such that residual = data - zfit
                        res(xmin(kk):xmax(kk),ymin(kk):ymax(kk))=0;        % Set values of residual around center to be 0
                        res = res(res<0);
                        b(kk) = sqrt(sum(res.*res)/(length(res)-1));
                    else
                        b(kk) = 0;
                    end
                end
                b = b*CountToPhoton;                    % Standard deviation of the background
                PhotonNo = abs(2*pi*popt(:,2).*popt(:,3).*popt(:,4)*CountToPhoton);   % Number of Photons calculated using volume under gaussian, which is 2*pi*A*stdev(x)*stdev(y)
                xPrecision = sqrt((sx2./PhotonNo) + (a2./(12*PhotonNo)) + (8*pi*sx4.*b.^2./(a2*PhotonNo.^2)));
                yPrecision = sqrt((sy2./PhotonNo) + (a2./(12*PhotonNo)) + (8*pi*sy4.*b.^2./(a2*PhotonNo.^2)));

                cd([DataPath '\' TiffName]);
                Output((k-1)*batchSize+1:(k-1)*batchSize+dataSize,:) = [PhotonNo xPrecision yPrecision xCenter yCenter RSquare popt(:,8) ((k-1)*batchSize+1:(k-1)*batchSize+dataSize)'];
                FIONAPositions{ind} = [xCenter yCenter];

                if (k-1)*batchSize+dataSize > DisplayCount
                    disp(['Finish analyzing ' num2str((k-1)*batchSize+dataSize) ' frames. Elapsed time: ' num2str(toc) ' s']);
                    DisplayCount = DisplayCount + DisplayMultiple;
                end
            end
            toc;

            % Save Output
            dlmwrite([FileInputName{ind} '(' num2str(SumFactor) 'x).txt'],Output,'\t');
        end

        clear i;
        
        % Account for offset after cropping in FIONAPositions
        for i = 1:size(TiffMatrix,1)
            CurrentPositions = FIONAPositions{i};
            CurrentPositions(CurrentPositions==0) = NaN;
            FIONAPositions{i} = [CurrentPositions(:,1)+xytOffset(i,1)+xytLimits(i,1)-2 CurrentPositions(:,2)+xytOffset(i,2)+xytLimits(i,2)-2];

            % Prepare FIONAPositions to have the same format as TraceMatrix
            FIONAPositions{i} = [TraceMatrix{i}(1,1)*ones(size(FIONAPositions{i},1),1) FIONAPositions{i} (TraceMatrix{i}(1,4):ExposureTime:TraceMatrix{i}(1,4)+(size(FIONAPositions{i},1)-1)*ExposureTime)'];

            % Remove NaN from FIONA Positions
            FIONAPositions{i}(isnan(CurrentPositions(:,1)),:) = [];
        end
        
        % Plot and delete outliers after FIONA fits
        hOutlierDelete = figure('units','pixels','outerposition',[0.2*screenWidth 0*screenHeight 0.8*screenWidth 1*screenHeight]);
        for i = 1:length(FIONAPositions)
            plot(FIONAPositions{i}(:,2),FIONAPositions{i}(:,3)); hold on;
            plot(FIONAPositions{i}(1,2),FIONAPositions{i}(1,3),'p','MarkerFaceColor','black','MarkerSize',15); 
            plot(FIONAPositions{i}(end,2),FIONAPositions{i}(end,3),'o','MarkerFaceColor','black','MarkerSize',10);
        end
        
        % Start loop to delete outliers
        StopLoop = 0;
        DeleteIndex = 1;
        DeletedPoints = cell(40,1);
        while StopLoop == 0
            % Redraw plot
            figure(hOutlierDelete); hold off;
            for i = 1:length(FIONAPositions)
                plot(FIONAPositions{i}(:,2),FIONAPositions{i}(:,3)); hold on;
                plot(FIONAPositions{i}(1,2),FIONAPositions{i}(1,3),'p','MarkerFaceColor','black','MarkerSize',15); 
                plot(FIONAPositions{i}(end,2),FIONAPositions{i}(end,3),'o','MarkerFaceColor','black','MarkerSize',10);
            end
            
            % Get polyline
            pan on;
            disp('Please delete region. Double click when done to close polygon.');
            h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK'), 'Zoom/ Pan');
            set(h, 'Position', [50 250 250 60]);
            uiwait(h); 
            choice = questdlg('Continue deleting regions?.','Continue?', ...
                    'Yes','No','Redo','Yes');

            switch choice
                case 'Yes'
                    % Get polyline
                    disp('Please delete region. Double click when done to close polygon.');
                    [xBorder,yBorder] = getline(gca);

                    % Draw polyline
%                     hBox(DeleteIndex) = plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'r'); 
                    
                    for i = 1:length(FIONAPositions)
                        BoundPoints = inpolygon(FIONAPositions{i}(:,2),FIONAPositions{i}(:,3),xBorder,yBorder);
                        DeletedPoints{DeleteIndex} = [DeletedPoints{DeleteIndex}; FIONAPositions{i}(BoundPoints,:)];
                        FIONAPositions{i}(BoundPoints,:) = [];
                    end
                           
                    DeleteIndex = DeleteIndex + 1;
                case 'No'
                    StopLoop = 1;
                case 'Redo'
                    % Undo record on DeleteIndex
                    DeleteIndex = DeleteIndex - 1;

                    % Recover DeletedPoints
                    for i = 1:length(FIONAPositions)
                        FIONAPositions{i} = [FIONAPositions{i}; DeletedPoints{DeleteIndex}(DeletedPoints{DeleteIndex}(:,1)==FIONAPositions{i}(1,1),:)];
                        % Delete repeated values
                        FIONAPositions{i} = unique(FIONAPositions{i},'rows','stable');
                        % Sort rows
                        FIONAPositions{i} = sortrows(FIONAPositions{i},4);
                    end
            end
        end 

        % Close figure
        close(hOutlierDelete);
        
        % Assign TraceMatrix to be FIONA Positions
        TraceMatrix = FIONAPositions;
        
        % Go to DataPath
        cd(DataPath);
    end
    
    % Ask for the number of stuck and dark spots
    TitleDialog = {'No. of stuck spots (Appear to have motor, but no displacement):','No. of dark spots (Appear to have motor, but no fluorescence):'};
    StuckDark = inputdlg(TitleDialog,'Number of Stuck and Dark Spots',1,{'0','0'});
    
    % Save StuckDark in EXCEL
    xlswrite(strrep(InputName,TrackMateFileSuffix,'(Stuck-Dark).xlsx'),{'No. Stuck', 'No. Dark'; StuckDark{1},StuckDark{2}});

    % Draw polyline for microtubule trace and rotate traces
    % Put warning dialog to ask user to be careful about the order
    h = warndlg(sprintf(' Please trace microtubule axis. Press <Enter> when done.\n Critical: Order is important! \n Trace Microtubule starting from tail to head.'), 'Trace microtubule');
    uiwait(h); 
    
    % Get polyline
    disp('Please trace microtubule axis. Press <Enter> when done.');
    [xAxis,yAxis] = getline(gca);
    
    % Interpolate lines
    Precision = 0.001;
    MaxDistance = max(max(xAxis)-min(xAxis),max(yAxis)-min(yAxis));
    NoOfPoints = MaxDistance/Precision;
    xyInit = 1:length(xAxis);
    xyFinal = 1:1/NoOfPoints:length(xAxis);
    
    xAxisFine = interp1(xyInit,xAxis,xyFinal);
    yAxisFine = interp1(xyInit,yAxis,xyFinal);
    plot(xAxisFine,yAxisFine,'black','LineWidth',1);
    
    % Calculate the distance of each point to the first point to get OnAxisDistance
    OnAxisDistance=sqrt((xAxisFine(2:end)-xAxisFine(1:end-1)).^2+(yAxisFine(2:end)-yAxisFine(1:end-1)).^2);
    OnAxisDistance=[0;OnAxisDistance']; OnAxisDistance=cumsum(OnAxisDistance);
    
    % Initialize RotatedTraceMatrix
    RotatedTraceMatrix = cell(RegionNo,1);
    
    % Loop traces and find closest distance to polyline for the OnAxis
    % distance, then find the OffAxis distance
    for i = 1:RegionNo
        [OnAxisIndex,OffAxisDistance] = knnsearch([xAxisFine' yAxisFine'],TraceMatrix{i}(:,2:3));
        RotatedTraceMatrix{i} = [OnAxisDistance(OnAxisIndex)*PixelSize OffAxisDistance*PixelSize];
    end
    
    % Save Point Locations figure
    if exist('Image Files','dir')~=7; mkdir('Image Files'); end
    print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Point Locations).png')]);
    savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Point Locations).fig')]);
    
    % Get coordinate graph
    cc=hsv(RegionNo)';
    figure('units','normalized','outerposition',[0.05 0 0.95 0.6]);
    for i = 1:RegionNo
        plot(RotatedTraceMatrix{i}(:,1),RotatedTraceMatrix{i}(:,2),'Color',cc(:,i)); hold on;
        text(max(RotatedTraceMatrix{i}(:,1)),max(RotatedTraceMatrix{i}(:,2)),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b');
    end
    title('Rotated Trace vs time');
    xlabel('X (nm)'); ylabel('Y (nm)');
    set(gca,'FontSize',TextSize,'FontWeight','b');
    
    % Save coordinate graph
    if exist('Image Files','dir')~=7; mkdir('Image Files'); end
    print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Coordinate-Graph).png')]);
    savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Coordinate-Graph).fig')]);
    
    % Issue dialog box to ask for frames to rotate microtubules
    if exist('FramesMic','var')
        FramesMic = inputdlg('Choose start frames when microtubule changes angle (comma separated). Include Frame 1.','Select Frames',1,{strjoin(arrayfun(@(x) num2str(x),FramesMic,'UniformOutput',false),',')});
    else
        FramesMic = inputdlg('Choose start frames when microtubule changes angle (comma separated). Include Frame 1.','Select Frames',1,{'1'});
    end
    FramesMic = strsplit(FramesMic{1},',');   % Separate the commas to generate cells of strings
    FramesMic = cellfun(@str2num,FramesMic);  % Change cells of strings to matrix

    % Get MicInputMeans
    MicInputMeans = uint16(zeros([dim(1) dim(2) length(FramesMic)]));
    for i = 1:length(FramesMic)
        if i == length(FramesMic)
            MicInputMeans(:,:,i) = imadjust(uint16(round(mean(MicInput(:,:,FramesMic(i):dim(3)),3))));
        else
            MicInputMeans(:,:,i) = imadjust(uint16(round(mean(MicInput(:,:,FramesMic(i):FramesMic(i+1)-1),3))));
        end
    end

    % Find points to rotate microtubules
    screenSize = get(0,'screensize');
    screenWidth = screenSize(3);
    screenHeight = screenSize(4);
    hFig = figure('units','pixels','outerposition',[0.2*screenWidth 0*screenHeight 0.8*screenWidth 1*screenHeight]);
    fpos = get(hFig,'Position');
    Scaling = min([fpos(3)/size(MicInputMean,2) fpos(4)/size(MicInputMean,1)])*0.90;
    axOffset = (fpos(3:4)-[size(MicInputMean,2)*Scaling size(MicInputMean,1)*Scaling])/2;
    ha = axes('Parent',hFig,'Units','pixels',...
            'Position',[axOffset size(MicInputMean,2)*Scaling size(MicInputMean,1)*Scaling]);
    % Store rotation points in Axes
    Axes = zeros(4,length(FramesMic));
    for i = 1:length(FramesMic)
        % Draw microtubule
        imshow(MicInputMeans(:,:,i),'Parent',ha);

        % Draw polyline for microtubule trace and rotate traces
        % Put warning dialog to ask user to be careful about the order
        h = warndlg(sprintf('Click two points at start & end of microtubule.\nPress <Enter> when done.\n\nCritical: Order is important! \nTrace Microtubule starting from tail to head.\nMake sure line is parallel to most of microtubule.'), 'Trace microtubule');
        uiwait(h); 

        % Get polyline
        disp('Click two points at start & end of microtubule. Press <Enter> when done.');
        [xAxis,yAxis] = getline(gca);

        Axes(:,i) = [xAxis(1); xAxis(end); yAxis(1); yAxis(end)];
    end

    % Rotate microtubules and draw rectangle to make kymograph
    ResizeFactor = 1;
    FrameOverlap = 15;       % Define how many frames will be overlapping
    LengthThreshold = 10*ResizeFactor;   % Determine lengths of kymographs below which lines are deleted
    UseSubPixelDetection = 0;
    StopLoop = 0;
    while StopLoop == 0
        % Change ResizeFactor to 1 if no sub-pixel detection is opted
        if UseSubPixelDetection == 0
            ResizeFactor = 1;
        end
        MicKymographs = cell(length(FramesMic),1);    % Initialize MicKymographs to store kymographs
        KymoDim = [0 0];

        for i = 1:length(FramesMic)
            % Calculate angle
            Angle = atan2d(Axes(4,i)-Axes(3,i),Axes(2,i)-Axes(1,i))-90;
            TransformedImage = imrotate(MicInputMeans(:,:,i),Angle);
    %         HalfWidth = round(dim(2)/2);
    %         MappingTFORM = fitgeotrans([[Axes(1,i);Axes(2,i)] [Axes(3,i);Axes(4,i)]], [[HalfWidth;HalfWidth] [0;OnAxisDistance(end)]],'nonreflectivesimilarity');
    %         TransformedImage = imwarp(MicInputMeans(:,:,i),MappingTFORM);
            h=figure;
            imshow(imadjust(TransformedImage));
            set(h,'units','normalized','outerposition',[0 0 1 1]);
            disp('Please draw rectangle along the rotated microtubule to make kymograph...');
            disp('Double click inside rectangle when done.');
            h = imrect;
            RectPosition = wait(h); RectPosition = round(RectPosition);
            delete(h);
            rectangle('Position',RectPosition,'EdgeColor','red');

            if i == length(FramesMic)
                StartFrame = max(FramesMic(i)-FrameOverlap,1);
                EndFrame = dim(3);
            else
                StartFrame = max(FramesMic(i)-FrameOverlap,1);
                EndFrame = min(FramesMic(i+1)+FrameOverlap,dim(3));
            end

            RotatedMic = uint16(zeros([RectPosition(4)+1 RectPosition(3)+1 EndFrame-StartFrame+1]));
            RotInd = 1;
            for j = StartFrame:EndFrame
                TransformedImage = imrotate(MicInput(:,:,j),Angle);
                RotatedMic(:,:,RotInd) = TransformedImage(RectPosition(2):RectPosition(2)+RectPosition(4),RectPosition(1):RectPosition(1)+RectPosition(3));
                RotInd = RotInd + 1;
            end

            MicKymograph = mean(RotatedMic,2);
            MicKymograph = permute(MicKymograph,[1 3 2]);
            MicKymographs(i) = {MicKymograph};

            % Get Kymograph Dimension
            KymoDim(1) = max(KymoDim(1),size(MicKymograph,1));
            KymoDim(2) = KymoDim(2) + size(MicKymograph,2) + 1;
        end

        % Combine MicKymographs to one Kymograph
        MicKymograph = zeros(KymoDim);
        FrameAssignment = zeros(KymoDim(2),1);
        ColumnStart = 1;
        for i = 1:length(FramesMic)
            MicKymograph(1:size(MicKymographs{i},1),ColumnStart:ColumnStart+size(MicKymographs{i},2)-1) = MicKymographs{i};

            if i == length(FramesMic)
                StartFrame = max(FramesMic(i)-FrameOverlap,1);
                EndFrame = dim(3);
            else
                StartFrame = max(FramesMic(i)-FrameOverlap,1);
                EndFrame = min(FramesMic(i+1)+FrameOverlap,dim(3));
            end

            FrameAssignment(ColumnStart:ColumnStart+size(MicKymographs{i},2)-1) = StartFrame:EndFrame;
            ColumnStart = ColumnStart + size(MicKymographs{i},2) + 1;
        end

        figure('units','normalized','outerposition',[0 0 1 1]);
        imshow(imadjust(uint16(round(MicKymograph)))); hold on;
        title('Kymograph of Microtubule Displacement');
        xlabel('Time (Frame)'); ylabel('D (pixel)');
        set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[],'yTick',[]); 

        % Plot red vertical lines for frame start and ends
        xPlot = find(ismember(FrameAssignment,FramesMic)); xPlot = xPlot(2:end);
        for ii = 1:length(xPlot)
            plot([xPlot(ii) xPlot(ii)],[0 size(MicKymograph,1)],'r');
        end

        % Save Kymograph
        print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Kymograph).png')]);
        savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Kymograph).fig')]);

        % Generate warning and ask for code path if we cannot find code in the active
        % folder and a few folder above
        DataPath = pwd;
        CodeToSearch = 'subpixelEdges.m';
        StopSearch = 0;
        SearchIndex = 0;
        while StopSearch == 0
            if isempty(dir(CodeToSearch))
                cd ..       % Go to one folder up and look for code there
                SearchIndex = SearchIndex + 1;
                if SearchIndex == 6
                    % Generate warning and ask for the code path
                    h=warndlg(['Cannot find ' CodeToSearch ' in active folder. Locate folder with ' CodeToSearch ' in the next window']);
                    uiwait(h);
                    CodePath = uigetdir(DataPath,['Select folder containing ' CodeToSearch]);
                    cd(CodePath);
                    StopSearch = 1;
                end
            else; CodePath = pwd; StopSearch = 1;
            end;
        end

        % Resize Kymograph
        KymoResized = imresize(MicKymograph,ResizeFactor);
        threshold = [5 10 15 20];

        % Loop to draw four edge detections for user to choose
        figure('units','normalized','outerposition',[0.05 0 0.95 0.95]);
        ha = tight_subplot(2,2,[.05 .01],[.1 .1],[.03 .03]);

        % Draw base image from KymoResized
        for j = 1:4
            axes(ha(j));
            imshow(imadjust(uint16(round(KymoResized))),'InitialMagnification','fit'); hold on;
        end

        % Do edge detection and plot
        for j = 1:4
            if UseSubPixelDetection == 1
                edges = subpixelEdges(MicKymograph, threshold(j), 'SmoothingIter', 1);
                BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
            else
                % Get velocity using edge detection
                if j == 1
                    [BW,cannyThreshold] = edge(uint16(round(MicKymograph)),'canny');
                elseif j == 2
                    BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.025 cannyThreshold(1)+0.05]);
                elseif j == 3
                    BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.05 cannyThreshold(1)+0.1]);
                elseif j == 4
                    BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.075 cannyThreshold(1)+0.15]);   
                end
            end

            % Make skeleton out of image and remove branchpoints
            BW = bwmorph(BW,'skel');
            BW2 = bwmorph(BW,'branchpoints');
            BW(BW2)=false;

            % Remove vertical lines more than 2 pixels
            for i = 1:size(BW,2)
                rp = regionprops(BW(:,i), 'PixelIdxList', 'Area');
                rp = rp(abs([rp.Area])>2);
                BW(vertcat(rp.PixelIdxList),i) = false;
            end

            % Remove branchpoints
            BW2 = bwmorph(BW,'branchpoints');
            BW(BW2)=false;

            % Remove regions with areas less than LengthThreshold
            rp = regionprops(BW, 'PixelIdxList', 'Area', 'Orientation');
            rp = rp(abs([rp.Area])<LengthThreshold);
            BW(vertcat(rp.PixelIdxList)) = false;

            % Remove regions which are roughly vertical
            rp = regionprops(BW, 'PixelIdxList', 'Orientation');
            rp = rp([rp.Orientation]>40 | [rp.Orientation]<-70);
            BW(vertcat(rp.PixelIdxList)) = false;

            % Remove regions that form loop
            rp = regionprops(BW, 'PixelList','PixelIdxList');
            for i = 1:length(rp)
                Pixels = rp(i).PixelList;
                PixelIDs = rp(i).PixelIdxList;
                [N,edges,bin]=histcounts(Pixels(:,1),'BinMethod','integers');
                Centers = (edges(2:end)+edges(1:end-1))/2;
                DoubledIndex = find(N>1);
                DiffMatrix = zeros(length(DoubledIndex),1);
                DiffCount = 1;
                for ii = DoubledIndex
                    yPix = Pixels(Pixels(:,1)==Centers(ii),2);
                    yDiff = max(yPix) - min(yPix);
                    DiffMatrix(DiffCount) = yDiff;
                    DiffCount = DiffCount + 1;
                end

                if ~isempty(DiffMatrix) && max(DiffMatrix)>1
                    DoubledIndex = DoubledIndex(DiffMatrix>1);
                    DiffMatrix = DiffMatrix(DiffMatrix>1);

                    if DoubledIndex(1) == 1
                        StartIndex = DoubledIndex(1);
                    else
                        StartIndex = DoubledIndex(1) - 1;
                    end
                    yPix1Real = Pixels(Pixels(:,1)==Centers(StartIndex),2);
                    yPix1 = Pixels(Pixels(:,1)==Centers(StartIndex+1),2);
                    yPix1Range = yPix1(1):yPix1(end);
                    yPix1 = yPix1Range(~ismember(yPix1Range,yPix1));
%                         yPix1 = yPix1(1)+1:yPix1(2)-1;

                    if length(Centers) == DoubledIndex(end)
                        EndIndex = DoubledIndex(end);
                    else
                        EndIndex = DoubledIndex(end) + 1;
                    end
                    yPix2Real = Pixels(Pixels(:,1)==Centers(EndIndex),2);
                    yPix2 = Pixels(Pixels(:,1)==Centers(EndIndex-1),2);
                    yPix2Range = yPix2(1):yPix2(end);
                    yPix2 = yPix2Range(~ismember(yPix2Range,yPix2));

                    if ismember(yPix1,yPix1Real)        % Delete DoubledIndex(1) side if yPix1 is a member of yPix1Real, which means it is the side forming loop
                        BW(PixelIDs(Pixels(:,1)==Centers(StartIndex) | Pixels(:,1)==Centers(StartIndex+1))) = false;
                    elseif ismember(yPix2,yPix2Real)
                        BW(PixelIDs(Pixels(:,1)==Centers(EndIndex) | Pixels(:,1)==Centers(EndIndex-1))) = false;
                    end
                end
            end

            % Remove regions with areas less than LengthThreshold
            rp = regionprops(BW, 'PixelIdxList', 'Area', 'Orientation');
            rp = rp(abs([rp.Area])<LengthThreshold);
            BW(vertcat(rp.PixelIdxList)) = false;

            % Plot green image then add alphadata to make only those with BW green
            axes(ha(j));
            green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
            h2 = imshow(green);
            set(h2, 'AlphaData', BW);
            title(num2str(j));
            set(gca,'FontSize',18,'FontWeight','b');

            % Plot red vertical lines for frame start and ends
            xPlot = find(ismember(FrameAssignment,FramesMic)); xPlot = xPlot(2:end);
            for ii = 1:length(xPlot)
                plot([xPlot(ii)*ResizeFactor xPlot(ii)*ResizeFactor],[0 size(MicKymograph,1)*ResizeFactor],'r');
            end

            % Mark unconnected regions with blue lines and dots
            rp = regionprops(BW, 'PixelIdxList', 'PixelList');
            for ii = 1:length(rp)
                Pixels = rp(ii).PixelList;
                PixelIDs = rp(ii).PixelIdxList;
                PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                rp(ii).PixelIdxList = PixelIDs;
                rp(ii).PixelList = Pixels;
            end
            BW2 = logical(zeros(size(BW)));
            BW2(vertcat(rp.PixelIdxList)) = true;
            MaxBW = max(BW2);
            FirstInd = find(MaxBW==1,1,'first');
            LastInd = find(MaxBW==1,1,'last');
            rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
            hPlots = zeros(length(rp2),4);
            for i = 1:length(rp2)
                hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            end
        end

        % Allow zooming before choosing 
        disp('Please choose which edge detection to use...');
        h = warndlg(sprintf('Please choose which edge detection to use...\nZoom in/out or pan to desired region before clicking OK.'), 'Zoom/ Pan');
%         set(h, 'Position', [400 500 250 60]);
        uiwait(h); 

        % Get a dialog box to ask which edge detection to choose
        [choice,v] = listdlg('PromptString','Which edge detection parameter to use?',...
                'SelectionMode','single', 'ListSize', [200 75], ...
                'ListString',{'1','2','3','4','Change Variables'});
        if UseSubPixelDetection == 1
            switch choice
                case 1
                    edges = subpixelEdges(MicKymograph, threshold(1), 'SmoothingIter', 4);
                    BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                    StopLoop = 1;
                case 2
                    edges = subpixelEdges(MicKymograph, threshold(2), 'SmoothingIter', 4);
                    BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                    StopLoop = 1;
                case 3
                    edges = subpixelEdges(MicKymograph, threshold(3), 'SmoothingIter', 4);
                    BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                    StopLoop = 1;
                case 4
                    edges = subpixelEdges(MicKymograph, threshold(4), 'SmoothingIter', 4);
                    BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                    StopLoop = 1;
                case 5
                    % Get user input for FrameOverlap, LengthThreshold and UseSubPixelDetection
                    NewVariables = inputdlg({['Frame Overlap (Last value = ' num2str(FrameOverlap) '):'], ['Length Threshold (Last value = ' num2str(LengthThreshold) '):'], ['Use SubPixel Detection (0 or 1)? (Last value = ' num2str(UseSubPixelDetection) '):'], ['Resize Factor (1 for No SubPixel Det.) (Last value = ' num2str(ResizeFactor) '):']},...
                                 'Enter new variable values', [1 65],{num2str(FrameOverlap),num2str(LengthThreshold),num2str(UseSubPixelDetection),num2str(ResizeFactor)});
                    NewVariables = cellfun(@str2num,NewVariables);
                    FrameOverlap = NewVariables(1);
                    LengthThreshold = NewVariables(2);
                    UseSubPixelDetection = NewVariables(3);
                    ResizeFactor = NewVariables(4);
            end
        else
            switch choice
                case 1
                    BW = edge(uint16(round(MicKymograph)),'canny');
                    StopLoop = 1;
                case 2
                    BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.025 cannyThreshold(1)+0.05]);
                    StopLoop = 1;
                case 3
                    BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.05 cannyThreshold(1)+0.1]);
                    StopLoop = 1;
                case 4
                    BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.075 cannyThreshold(1)+0.15]);
                    StopLoop = 1;
                case 5
                    % Get user input for FrameOverlap, LengthThreshold and UseSubPixelDetection
                    NewVariables = inputdlg({['Frame Overlap (Last value = ' num2str(FrameOverlap) '):'], ['Length Threshold (Last value = ' num2str(LengthThreshold) '):'], ['Use SubPixel Detection (0 or 1)? (Last value = ' num2str(UseSubPixelDetection) '):'], ['Resize Factor (1 for No SubPixel Det.) (Last value = ' num2str(ResizeFactor) '):']},...
                                 'Enter new variable values', [1 65],{num2str(FrameOverlap),num2str(LengthThreshold),num2str(UseSubPixelDetection),num2str(ResizeFactor)});
                    NewVariables = cellfun(@str2num,NewVariables);
                    FrameOverlap = NewVariables(1);
                    LengthThreshold = NewVariables(2);
                    UseSubPixelDetection = NewVariables(3);
                    ResizeFactor = NewVariables(4);
            end
        end
    end

    % Draw out chosen edge detection
    % Make skeleton out of image and remove branchpoints
    BW = bwmorph(BW,'skel');
    BW2 = bwmorph(BW,'branchpoints');
    BW(BW2)=false;
    % Remove vertical lines more than 2 pixels
    for i = 1:size(BW,2)
        rp = regionprops(BW(:,i), 'PixelIdxList', 'Area');
        rp = rp(abs([rp.Area])>2);
        BW(vertcat(rp.PixelIdxList),i) = false;
    end
    % Remove branchpoints
    BW2 = bwmorph(BW,'branchpoints');
    BW(BW2)=false;
    % Remove regions with areas less than LengthThreshold
    rp = regionprops(BW, 'PixelIdxList', 'Area');
    rp = rp(abs([rp.Area])<LengthThreshold);
    BW(vertcat(rp.PixelIdxList)) = false;
    % Remove regions which are roughly vertical
    rp = regionprops(BW, 'PixelIdxList', 'Orientation');
    rp = rp([rp.Orientation]>50 | [rp.Orientation]<-80);
    BW(vertcat(rp.PixelIdxList)) = false;
    % Remove regions that form loop
    rp = regionprops(BW, 'PixelList','PixelIdxList');
    for i = 1:length(rp)
        Pixels = rp(i).PixelList;
        PixelIDs = rp(i).PixelIdxList;
        [N,edges,bin]=histcounts(Pixels(:,1),'BinMethod','integers');
        Centers = (edges(2:end)+edges(1:end-1))/2;
        DoubledIndex = find(N>1);
        DiffMatrix = zeros(length(DoubledIndex),1);
        DiffCount = 1;
        for ii = DoubledIndex
            yPix = Pixels(Pixels(:,1)==Centers(ii),2);
            yDiff = max(yPix) - min(yPix);
            DiffMatrix(DiffCount) = yDiff;
            DiffCount = DiffCount + 1;
        end

        if ~isempty(DiffMatrix) && max(DiffMatrix)>1
            DoubledIndex = DoubledIndex(DiffMatrix>1);
            DiffMatrix = DiffMatrix(DiffMatrix>1);

            if DoubledIndex(1) == 1
                StartIndex = DoubledIndex(1);
            else
                StartIndex = DoubledIndex(1) - 1;
            end
            yPix1Real = Pixels(Pixels(:,1)==Centers(StartIndex),2);
            yPix1 = Pixels(Pixels(:,1)==Centers(StartIndex+1),2);
            yPix1Range = yPix1(1):yPix1(end);
            yPix1 = yPix1Range(~ismember(yPix1Range,yPix1));
%                         yPix1 = yPix1(1)+1:yPix1(2)-1;

            if length(Centers) == DoubledIndex(end)
                EndIndex = DoubledIndex(end);
            else
                EndIndex = DoubledIndex(end) + 1;
            end
            yPix2Real = Pixels(Pixels(:,1)==Centers(EndIndex),2);
            yPix2 = Pixels(Pixels(:,1)==Centers(EndIndex-1),2);
            yPix2Range = yPix2(1):yPix2(end);
            yPix2 = yPix2Range(~ismember(yPix2Range,yPix2));

            if ismember(yPix1,yPix1Real)        % Delete DoubledIndex(1) side if yPix1 is a member of yPix1Real, which means it is the side forming loop
                BW(PixelIDs(Pixels(:,1)==Centers(StartIndex) | Pixels(:,1)==Centers(StartIndex+1))) = false;
            elseif ismember(yPix2,yPix2Real)
                BW(PixelIDs(Pixels(:,1)==Centers(EndIndex) | Pixels(:,1)==Centers(EndIndex-1))) = false;
            end
        end
    end
    % Remove regions with areas less than LengthThreshold
    rp = regionprops(BW, 'PixelIdxList', 'Area', 'Orientation');
    rp = rp(abs([rp.Area])<LengthThreshold);
    BW(vertcat(rp.PixelIdxList)) = false;
    % Plot processed BW image
    screenSize = get(0,'screensize');
    screenWidth = screenSize(3);
    screenHeight = screenSize(4);
%     hFig = figure('units','pixels','outerposition',[0.05*screenWidth 0*screenHeight 0.95*screenWidth 0.9*screenHeight]);
    hFig = figure('units','pixels','outerposition',[0.1*screenWidth 0*screenHeight 0.9*screenWidth 1*screenHeight]);
    fpos = get(hFig,'Position');
    Scaling = min([fpos(3)/size(BW,2) fpos(4)/size(BW,1)])*0.90;
    axOffset = (fpos(3:4)-[size(BW,2)*Scaling size(BW,1)*Scaling])/2;
    ha = axes('Parent',hFig,'Units','pixels',...
            'Position',[axOffset size(BW,2)*Scaling size(BW,1)*Scaling]);
    h1 = imshow(imadjust(uint16(round(KymoResized))),'InitialMagnification','fit','Parent',ha); hold on;
    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
    h2 = imshow(green);
    set(h2, 'AlphaData', BW);
    title(num2str(choice));
    set(gca,'FontSize',18,'FontWeight','b');
    % Plot red vertical lines for frame start and ends
    xPlot = find(ismember(FrameAssignment,FramesMic)); xPlot = xPlot(2:end);
    for ii = 1:length(xPlot)
        plot([xPlot(ii)*ResizeFactor xPlot(ii)*ResizeFactor],[0 size(MicKymograph,1)*ResizeFactor],'r');
    end
    % Mark unconnected regions with blue lines and dots
    rp = regionprops(BW, 'PixelIdxList', 'PixelList');
    for ii = 1:length(rp)
        Pixels = rp(ii).PixelList;
        PixelIDs = rp(ii).PixelIdxList;
        PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
        Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
        rp(ii).PixelIdxList = PixelIDs;
        rp(ii).PixelList = Pixels;
    end
    BW2 = logical(zeros(size(BW)));
    BW2(vertcat(rp.PixelIdxList)) = true;
    MaxBW = max(BW2);
    FirstInd = find(MaxBW==1,1,'first');
    LastInd = find(MaxBW==1,1,'last');
    rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
    hPlots = zeros(length(rp2),4);
    for i = 1:length(rp2)
        hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
        hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
        hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
        hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
    end

    % Loop to delete regions
    % Get the rows and columns of all ones in BW
    [yPos,xPos] = ind2sub(size(BW),find(BW==1)); 
    % Create region to delete and loop to delete regions
    rp = regionprops(BW, 'Area');
    DeletedPoints = cell(length(rp),1);
    DeleteInd = 1;
    StopLoop = 0;
    while StopLoop == 0
        % Get polyline
        pan on;
        disp('Please identify regions to delete. Double click when done to close polygon.');
        h = warndlg(sprintf('(Delete) Zoom in/out or pan to desired region before clicking OK'), 'Delete Zoom/ Pan');
        set(h, 'Position', [50 250 280 60]);
%         set(h, 'Position', [200 400 280 60]);
        uiwait(h); 
        choice = questdlg('Continue deleting?','Continue?', ...
                'Yes','No','Redo','Yes');

        switch choice
            case 'Yes'
                % Get polyline
                disp('Please select area to delete edges. Double click when done to close polygon.');
%                     [xBorder,yBorder] = getline(gca);

                h = imfreehand;
                Borders = getPosition(h);
                delete(h)

                % Find points within border
                BoundPoints = inpolygon(xPos,yPos,Borders(:,1),Borders(:,2));

%                     % Draw polyline
%                     plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'r'); 

%                     % Find points within polyline
%                     BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                % Delete all BoundPoints
                DeletedPoints(DeleteInd) = {[yPos(BoundPoints) xPos(BoundPoints)]};
                BW(sub2ind(size(BW),yPos(BoundPoints),xPos(BoundPoints))) = false;
                delete(h2);
                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                h2 = imshow(green);
                set(h2, 'AlphaData', BW);
                DeleteInd = DeleteInd + 1;

                % Mark unconnected regions with blue lines and dots
                delete(hPlots);
                rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                for ii = 1:length(rp)
                    Pixels = rp(ii).PixelList;
                    PixelIDs = rp(ii).PixelIdxList;
                    PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    rp(ii).PixelIdxList = PixelIDs;
                    rp(ii).PixelList = Pixels;
                end
                BW2 = logical(zeros(size(BW)));
                BW2(vertcat(rp.PixelIdxList)) = true;
                MaxBW = max(BW2);
                FirstInd = find(MaxBW==1,1,'first');
                LastInd = find(MaxBW==1,1,'last');
                rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                hPlots = zeros(length(rp2),4);
                for i = 1:length(rp2)
                    hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                end
            case 'No'
                StopLoop = 1;
            case 'Redo'
                DeleteInd = DeleteInd - 1;
                BW(sub2ind(size(BW),DeletedPoints{DeleteInd}(:,1),DeletedPoints{DeleteInd}(:,2))) = true;
                delete(h2);
                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                h2 = imshow(green);
                set(h2, 'AlphaData', BW);

                % Mark unconnected regions with blue lines and dots
                delete(hPlots);
                rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                for ii = 1:length(rp)
                    Pixels = rp(ii).PixelList;
                    PixelIDs = rp(ii).PixelIdxList;
                    PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    rp(ii).PixelIdxList = PixelIDs;
                    rp(ii).PixelList = Pixels;
                end
                BW2 = logical(zeros(size(BW)));
                BW2(vertcat(rp.PixelIdxList)) = true;
                MaxBW = max(BW2);
                FirstInd = find(MaxBW==1,1,'first');
                LastInd = find(MaxBW==1,1,'last');
                rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                hPlots = zeros(length(rp2),4);
                for i = 1:length(rp2)
                    hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                end
        end
    end

    % Mark unconnected regions with blue lines and dots
    delete(hPlots);
    rp = regionprops(BW, 'PixelIdxList', 'PixelList');
    for ii = 1:length(rp)
        Pixels = rp(ii).PixelList;
        PixelIDs = rp(ii).PixelIdxList;
        PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
        Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
        rp(ii).PixelIdxList = PixelIDs;
        rp(ii).PixelList = Pixels;
    end
    BW2 = logical(zeros(size(BW)));
    BW2(vertcat(rp.PixelIdxList)) = true;
    MaxBW = max(BW2);
    FirstInd = find(MaxBW==1,1,'first');
    LastInd = find(MaxBW==1,1,'last');
    rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
    hPlots = zeros(length(rp2),4);
    for i = 1:length(rp2)
        hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
        hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
        hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
        hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
    end

    % Patch unconnected regions and allow deletion
    zoom out;
    StopLoop = 0;
    RedoMatrix = zeros(length(rp),1);   % 0 if 'Patch' is selection, 1 if 'Delete' is selected
    RedoInd = 1;
    PatchPoints = cell(length(rp),1);
    PatchInd = 1;
    while StopLoop == 0
        % Get polyline
        pan on;
        disp('Please draw regions to patch (especially areas marked red). Double click when done to close polygon.');
        h = warndlg(sprintf('(Patching) Zoom in/out or pan to desired region before clicking OK'), 'Patching Zoom/ Pan');
        set(h, 'Position', [50 250 290 60]);
    %         set(h, 'Position', [400 680 290 60]);
        uiwait(h); 
    %             choice = questdlg('Continue patching/ deleting?','Continue?', ...
    %                     'Patch','Delete','Redo','No','Patch');
        [choice,v] = listdlg('PromptString','Continue?',...
            'SelectionMode','single', 'ListSize', [100 60], ...
            'ListString',{'Patch','Delete','Redo','No'});
        [yPos,xPos] = ind2sub(size(BW),find(BW==1));    % Get the rows and columns of all ones in BW

        switch choice
            case 1
                % Get polyline
                disp('Please select area to delete edges. Double click when done to close polygon.');
                [xBorder,yBorder] = getline(gca);

    %                 % Draw polyline
    %                 plot(xBorder,yBorder,'r');

                % Get pixels to patch
                xBorder = round(xBorder);
                yBorder = round(yBorder);
                tBorder = 1:length(xBorder);
                tInterp = 1:0.001:length(xBorder);
                xInterp = round(interp1(tBorder,xBorder,tInterp,'linear'))';
                yInterp = round(interp1(tBorder,yBorder,tInterp,'linear'))';

                % Get unique values
                PatchPixels = unique([yInterp xInterp],'rows');

                % Draw patch
                BW(sub2ind(size(BW),PatchPixels(:,1),PatchPixels(:,2))) = true;
                delete(h2);
                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                h2 = imshow(green);
                set(h2, 'AlphaData', BW);

                % Remember Patch
                PatchPoints(PatchInd) = {PatchPixels};

                % Increase RedoInd and PatchInd. RedoMatrix is automatically zero
                RedoMatrix(RedoInd) = 0;
                RedoInd = RedoInd + 1;
                PatchInd = PatchInd + 1;
            case 2
                % Get polyline
%                     disp('Please select area to delete edges. Double click when done to close polygon.');
%                     [xBorder,yBorder] = getline(gca);

    %                 % Draw polyline
    %                 plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'r'); 

                % Start free hand selection
                h = imfreehand;
                Borders = getPosition(h);
                delete(h)

                % Find points within border
                BoundPoints = inpolygon(xPos,yPos,Borders(:,1),Borders(:,2));

%                     % Find points within polyline
%                     BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                % Delete all BoundPoints
                DeletedPoints(DeleteInd) = {[yPos(BoundPoints) xPos(BoundPoints)]};
                BW(sub2ind(size(BW),yPos(BoundPoints),xPos(BoundPoints))) = false;
                delete(h2);
                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                h2 = imshow(green);
                set(h2, 'AlphaData', BW);
                DeleteInd = DeleteInd + 1;
                RedoMatrix(RedoInd) = 1;
                RedoInd = RedoInd + 1;
            case 3
                RedoInd = RedoInd - 1;
                if RedoMatrix(RedoInd) == 0
                    % Start redoing 'Patch'
                    PatchInd = PatchInd - 1;
                    BW(sub2ind(size(BW),PatchPoints{PatchInd}(:,1),PatchPoints{PatchInd}(:,2))) = false;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);
                else
                    % Start redoing 'Delete'
                    DeleteInd = DeleteInd - 1;
                    BW(sub2ind(size(BW),DeletedPoints{DeleteInd}(:,1),DeletedPoints{DeleteInd}(:,2))) = true;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);
                end
            case 4
                StopLoop = 1;
        end

        % Mark unconnected regions with blue lines and dots
        delete(hPlots);
        rp = regionprops(BW, 'PixelIdxList', 'PixelList');
        for ii = 1:length(rp)
            Pixels = rp(ii).PixelList;
            PixelIDs = rp(ii).PixelIdxList;
            PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            rp(ii).PixelIdxList = PixelIDs;
            rp(ii).PixelList = Pixels;
        end
        BW2 = logical(zeros(size(BW)));
        BW2(vertcat(rp.PixelIdxList)) = true;
        MaxBW = max(BW2);
        FirstInd = find(MaxBW==1,1,'first');
        LastInd = find(MaxBW==1,1,'last');
        rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
        hPlots = zeros(length(rp2),4);
        for i = 1:length(rp2)
            hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
        end
    end
    % Zoom out
    zoom out;

    % Final patch and delete with base image off
    delete(h1); delete(hPlots);
    % Mark unconnected regions with blue lines and dots
    rp = regionprops(BW, 'PixelIdxList', 'PixelList');
    for ii = 1:length(rp)
        Pixels = rp(ii).PixelList;
        PixelIDs = rp(ii).PixelIdxList;
        PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
        Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
        rp(ii).PixelIdxList = PixelIDs;
        rp(ii).PixelList = Pixels;
    end
    BW2 = logical(zeros(size(BW)));
    BW2(vertcat(rp.PixelIdxList)) = true;
    MaxBW = max(BW2);
    FirstInd = find(MaxBW==1,1,'first');
    LastInd = find(MaxBW==1,1,'last');
    rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
    hPlots = zeros(length(rp2),4);
    for i = 1:length(rp2)
        hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
        hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
        hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
        hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
    end
    FinalPatchLoop = 0;
    while FinalPatchLoop == 0
        % Start patching unconnected area. Allow deletion too.
        StopLoop = 0;
        RedoMatrix = zeros(length(rp),1);   % 0 if 'Patch' is selection, 1 if 'Delete' is selected
        RedoInd = 1;
        PatchPoints = cell(length(rp),1);
        PatchInd = 1;
        while StopLoop == 0
            % Get polyline
            pan on;
            disp('Please draw regions to patch (especially areas marked red). Double click when done to close polygon.');
            h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK'), 'Zoom/ Pan');
            set(h, 'Position', [50 250 250 60]);
%             set(h, 'Position', [400 680 250 60]);
            uiwait(h); 
    %             choice = questdlg('Continue patching/ deleting?','Continue?', ...
    %                     'Patch','Delete','Redo','No','Patch');
            [choice,v] = listdlg('PromptString','Continue?',...
                'SelectionMode','single', 'ListSize', [100 60], ...
                'ListString',{'Patch','Delete','Redo','No'});
            [yPos,xPos] = ind2sub(size(BW),find(BW==1));    % Get the rows and columns of all ones in BW

            switch choice
                case 1
                    % Get polyline
                    disp('Please select area to delete edges. Double click when done to close polygon.');
                    [xBorder,yBorder] = getline(gca);

    %                 % Draw polyline
    %                 plot(xBorder,yBorder,'r');

                    % Get pixels to patch
                    xBorder = round(xBorder);
                    yBorder = round(yBorder);
                    tBorder = 1:length(xBorder);
                    tInterp = 1:0.001:length(xBorder);
                    xInterp = round(interp1(tBorder,xBorder,tInterp,'linear'))';
                    yInterp = round(interp1(tBorder,yBorder,tInterp,'linear'))';

                    % Get unique values
                    PatchPixels = unique([yInterp xInterp],'rows');

                    % Draw patch
                    BW(sub2ind(size(BW),PatchPixels(:,1),PatchPixels(:,2))) = true;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);

                    % Remember Patch
                    PatchPoints(PatchInd) = {PatchPixels};

                    % Increase RedoInd and PatchInd. RedoMatrix is automatically zero
                    RedoMatrix(RedoInd) = 0;
                    RedoInd = RedoInd + 1;
                    PatchInd = PatchInd + 1;
                case 2
%                         % Get polyline
%                         disp('Please select area to delete edges. Double click when done to close polygon.');
%                         [xBorder,yBorder] = getline(gca);
% 
%                         % Find points within polyline
%                         BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                    % Start free hand selection
                    h = imfreehand;
                    Borders = getPosition(h);
                    delete(h);

                    % Find points within border
                    BoundPoints = inpolygon(xPos,yPos,Borders(:,1),Borders(:,2));

                    % Delete all BoundPoints
                    DeletedPoints(DeleteInd) = {[yPos(BoundPoints) xPos(BoundPoints)]};
                    BW(sub2ind(size(BW),yPos(BoundPoints),xPos(BoundPoints))) = false;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);
                    DeleteInd = DeleteInd + 1;
                    RedoMatrix(RedoInd) = 1;
                    RedoInd = RedoInd + 1;
                case 3
                    RedoInd = RedoInd - 1;
                    if RedoMatrix(RedoInd) == 0
                        % Start redoing 'Patch'
                        PatchInd = PatchInd - 1;
                        BW(sub2ind(size(BW),PatchPoints{PatchInd}(:,1),PatchPoints{PatchInd}(:,2))) = false;
                        delete(h2);
                        green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                        h2 = imshow(green);
                        set(h2, 'AlphaData', BW);
                    else
                        % Start redoing 'Delete'
                        DeleteInd = DeleteInd - 1;
                        BW(sub2ind(size(BW),DeletedPoints{DeleteInd}(:,1),DeletedPoints{DeleteInd}(:,2))) = true;
                        delete(h2);
                        green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                        h2 = imshow(green);
                        set(h2, 'AlphaData', BW);
                    end
                case 4
                    StopLoop = 1;
            end

            % Mark unconnected regions with blue lines and dots
            delete(hPlots);
            rp = regionprops(BW, 'PixelIdxList', 'PixelList');
            for ii = 1:length(rp)
                Pixels = rp(ii).PixelList;
                PixelIDs = rp(ii).PixelIdxList;
                PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                rp(ii).PixelIdxList = PixelIDs;
                rp(ii).PixelList = Pixels;
            end
            BW2 = logical(zeros(size(BW)));
            BW2(vertcat(rp.PixelIdxList)) = true;
            MaxBW = max(BW2);
            FirstInd = find(MaxBW==1,1,'first');
            LastInd = find(MaxBW==1,1,'last');
            rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
            hPlots = zeros(length(rp2),4);
            for i = 1:length(rp2)
                hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            end
        end

        % Ask if we want to stop
        choice = questdlg('Continue patching/ deleting?','Continue?', ...
                'Yes','No','Yes');
        switch choice
            case 'Yes'
            case 'No'
                FinalPatchLoop = 1;
                h1 = imshow(imadjust(uint16(round(KymoResized))),'InitialMagnification','fit','Parent',ha); hold on;
                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                h2 = imshow(green);
                set(h2, 'AlphaData', BW);
                % Mark unconnected regions with blue lines and dots
                delete(hPlots);
                rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                for ii = 1:length(rp)
                    Pixels = rp(ii).PixelList;
                    PixelIDs = rp(ii).PixelIdxList;
                    PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    rp(ii).PixelIdxList = PixelIDs;
                    rp(ii).PixelList = Pixels;
                end
                BW2 = logical(zeros(size(BW)));
                BW2(vertcat(rp.PixelIdxList)) = true;
                MaxBW = max(BW2);
                FirstInd = find(MaxBW==1,1,'first');
                LastInd = find(MaxBW==1,1,'last');
                rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                hPlots = zeros(length(rp2),4);
                for i = 1:length(rp2)
                    hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                end
        end
    end

    % Prepare to save Processed Kymograph, then save
    zoom out;
    set(hFig,'units','pixels','outerposition',[0.05*screenWidth 0.01*screenHeight 0.95*screenWidth 0.95*screenHeight]);
    title(['Processed Kymograph (Fr. Overlap: ' num2str(FrameOverlap) '. Length Thres.: ' num2str(LengthThreshold) '. SubPixel Det.: ' num2str(UseSubPixelDetection) '. Resize Fact.: ' num2str(ResizeFactor) ')']);
    xlabel('Time (Frame)'); ylabel('D (pixel)');
    set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[],'yTick',[]);
    % Save Processed Kymograph and BW
    cd(DataPath);
    print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Processed Kymograph).png')]);
    savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Processed Kymograph).fig')]);
    save(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Binary Kymograph)')],'BW');

    % Compile all velocities
    rp = regionprops(BW, 'PixelList');
    Velocities = NaN(length(rp),size(BW,2)/ResizeFactor);
    for i = 1:length(rp)
        % Average all rows in the same column
        ColumnRow = sortrows(rp(i).PixelList,1);
        [Column,ia,ic] = unique(ColumnRow(:,1));
    %         h = histogram(ic,'BinMethod','integers');
    %         DuplicatedIndices = find(h.Values>1);
        h = histcounts(ic,'BinMethod','integers');
        DuplicatedIndices = find(h>1);
        Row = ColumnRow(ia,2);
        % Find mean for duplicated indices
        for ii = 1:length(DuplicatedIndices)
            Row(DuplicatedIndices(ii)) = mean(ColumnRow(ic==DuplicatedIndices(ii),2));
        end
        % Get smoothed data
        SmoothedRow = smooth(Row,100*ResizeFactor);
        Velocity = (SmoothedRow(2:end) - SmoothedRow(1:end-1))*PixelSize./((Column(2:end)-Column(1:end-1))*ExposureTime);   % Velocity in nm/s
        NewColumn = ((Column-1)/ResizeFactor)+1; NewColumn = NewColumn(2:end);
        VelIndex = find(rem(NewColumn,1)==0,1);
        VelLastIndex = find(rem(NewColumn(1:end-ResizeFactor+1),1)==0,1,'last');
        NewColumn = NewColumn(VelIndex:ResizeFactor:VelLastIndex);
        NewVelocity = mean(reshape(Velocity(VelIndex:VelLastIndex+ResizeFactor-1),[],ResizeFactor),2);
        if VelIndex > 1
            if isempty(VelLastIndex)
                NewVelocity = [mean(Velocity(1:VelIndex-1)); NewVelocity];
                NewColumn = ((Column-1)/ResizeFactor)+1; NewColumn = NewColumn(2:end);
                NewColumn = floor(NewColumn(1));
            else
                NewVelocity = [mean(Velocity(1:VelIndex-1)); NewVelocity];
                NewColumn = [NewColumn(1)-1; NewColumn];
            end
        end
        if VelLastIndex+ResizeFactor-1 < length(Velocity)
            NewVelocity = [NewVelocity; mean(Velocity(VelLastIndex+ResizeFactor:end))];
            NewColumn = [NewColumn; NewColumn(end)+1];
        end
        Velocities(i,FrameAssignment(NewColumn)) = NewVelocity';
    end

    % Average over column to get mean velocities at every time point
    MeanVel = nanmean(Velocities,1);

    % Get displacement at every time point
    Displacement = cumsum(MeanVel*ExposureTime,'omitnan');

    % Plot Mean Velocity
    figure;
    plot((1:size(BW,2)/ResizeFactor)*ExposureTime,MeanVel,'LineWidth',2);
    ylabel('Velocity (nm/s)');
    xlim([0 (size(BW,2)/ResizeFactor)*ExposureTime]);
    xlabel('Time (s)');
    title('Microtubule Velocity over Time');
    set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[]); 

    % Save velocity figure
    print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Velocity).png')]);
    savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Velocity).fig')]);

    % Draw Displacement diagram
    figure;
    plot((1:size(BW,2)/ResizeFactor)*ExposureTime,Displacement/1000,'LineWidth',2);
    ylabel('Disp. (um)');
    xlim([0 (size(BW,2)/ResizeFactor)*ExposureTime]);
    xlabel('Time (s)');
    title('Microtubule Disp. over Time');
    set(gca,'FontSize',TextSize,'FontWeight','b');

    % Save displacement figure
    print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Displacement).png')]);
    savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Displacement).fig')]);

    MicTimeDispVel = [(1:(size(BW,2)/ResizeFactor))'*ExposureTime Displacement' MeanVel'];
    MicTimeDispVel(isnan(MeanVel),:) = [];
    
    % Save kin-Qd and microtubule velocity data
    FinalResult = cell(dim(3)+1,RegionNo*3+3);
    FinalResult(1,1:3) = {'Microtubule Time (s)','Microtubule Displacement (nm)','Microtubule Velocity (nm/s)'};
    for i = 1:RegionNo
        FinalResult(1,3*i+1:3*i+3) = {['Point ' num2str(i) ' Time (s)'],['Point ' num2str(i) ' On-Axis Displacement (nm)'],['Point ' num2str(i) ' Off-Axis Displacement (nm)']};
    end
    
    FinalResult(2:size(MicTimeDispVel,1)+1,1:3)=num2cell(MicTimeDispVel);
    for i = 1:RegionNo
        FinalResult(2:length(RotatedTraceMatrix{i})+1,3*i+1:3*i+3) = num2cell([TraceMatrix{i}(:,4)+ExposureTime RotatedTraceMatrix{i}]);
    end
    
    % Write FinalResult to EXCEL spreadsheet
    xlswrite(strrep(InputName,TrackMateFileSuffix,'(Final Result).xlsx'),FinalResult);
    
    % Write PixelSize and ExposureTime to EXCEL spreadsheet
    xlswrite(strrep(InputName,TrackMateFileSuffix,'(Variables).xlsx'),[PixelSize ExposureTime]);

%     % Import the three Final Result files
%     % Final Result
%     [~,~,raw] = xlsread([FileName '(Final Result).xlsx']);
%     raw(2:end,1:3) = {NaN};
%     raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
%     xlswrite([FileName '(Final Result).xlsx'],raw);     % Write FinalResult to EXCEL spreadsheet
%     % Final Result2
%     [~,~,raw] = xlsread([FileName '(Final Result2).xlsx']);
%     raw(2:end,1:3) = {NaN};
%     raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
%     xlswrite([FileName '(Final Result2).xlsx'],raw);    % Write FinalResult to EXCEL spreadsheet
%     % Final Result3
%     [~,~,raw] = xlsread([FileName '(Final Result3).xlsx']);
%     raw(2:end,1:3) = {NaN};
%     raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
%     xlswrite([FileName '(Final Result3).xlsx'],raw);    % Write FinalResult to EXCEL spreadsheet

    % Clear StartEndMatrix for next processing
    clear StartEndMatrix;

    % Display progress
    disp('Done importing positions. Proceed to plotting microtubule velocity and kin-Qd graphs...');
        %% (Repeated) Ask for the number of stuck and dark spots
        TrackMateFileSuffix = '(KinTrackMateSpots).csv';
        InputName = dir(['*' TrackMateFileSuffix]);
        InputName = InputName.name;
        
        TitleDialog = {'No. of stuck spots (Appear to have motor, but no displacement):','No. of dark spots (Appear to have motor, but no fluorescence):'};
        StuckDark = inputdlg(TitleDialog,'Number of Stuck and Dark Spots',1,{'0','0'});

        % Save StuckDark in EXCEL
        xlswrite(strrep(InputName,TrackMateFileSuffix,'(Stuck-Dark).xlsx'),{'No. Stuck', 'No. Dark'; StuckDark{1},StuckDark{2}});
        %% (Repeated) Draw polyline for microtubule trace and rotate traces
        % Put warning dialog to ask user to be careful about the order
        h = warndlg(sprintf(' Please trace microtubule axis. Press <Enter> when done.\n Critical: Order is important! \n Trace Microtubule starting from tail to head.'), 'Trace microtubule');
        uiwait(h); 

        % Get polyline
        disp('Please trace microtubule axis. Press <Enter> when done.');
        [xAxis,yAxis] = getline(gca);

        % Interpolate lines
        Precision = 0.001;
        MaxDistance = max(max(xAxis)-min(xAxis),max(yAxis)-min(yAxis));
        NoOfPoints = MaxDistance/Precision;
        xyInit = 1:length(xAxis);
        xyFinal = 1:1/NoOfPoints:length(xAxis);

        xAxisFine = interp1(xyInit,xAxis,xyFinal);
        yAxisFine = interp1(xyInit,yAxis,xyFinal);
        plot(xAxisFine,yAxisFine,'black','LineWidth',1);

        % Calculate the distance of each point to the first point to get OnAxisDistance
        OnAxisDistance=sqrt((xAxisFine(2:end)-xAxisFine(1:end-1)).^2+(yAxisFine(2:end)-yAxisFine(1:end-1)).^2);
        OnAxisDistance=[0;OnAxisDistance']; OnAxisDistance=cumsum(OnAxisDistance);

        % Initialize RotatedTraceMatrix
        RotatedTraceMatrix = cell(RegionNo,1);

        % Loop traces and find closest distance to polyline for the OnAxis
        % distance, then find the OffAxis distance
        for i = 1:RegionNo
            [OnAxisIndex,OffAxisDistance] = knnsearch([xAxisFine' yAxisFine'],TraceMatrix{i}(:,2:3));
            RotatedTraceMatrix{i} = [OnAxisDistance(OnAxisIndex)*PixelSize OffAxisDistance*PixelSize];
        end

        % Save Point Locations figure
        if exist('Image Files','dir')~=7; mkdir('Image Files'); end
        print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Point Locations).png')]);

        % Get coordinate graph
        cc=hsv(RegionNo)';
        figure('units','normalized','outerposition',[0.05 0 0.95 0.6]);
    %     subplot(3,1,3);
        for i = 1:RegionNo
            plot(RotatedTraceMatrix{i}(:,1),RotatedTraceMatrix{i}(:,2),'Color',cc(:,i)); hold on;
            text(max(RotatedTraceMatrix{i}(:,1)),max(RotatedTraceMatrix{i}(:,2)),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b');
        end
        title('Rotated Trace vs time');
        xlabel('X (nm)'); ylabel('Y (nm)');
        set(gca,'FontSize',TextSize,'FontWeight','b');

        % Save coordinate graph
        if exist('Image Files','dir')~=7; mkdir('Image Files'); end
        print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Coordinate-Graph).png')]);
        
        
        
        
        % Start rotating image
        HalfWidth = round(dim(2)/2);
        MappingTFORM = fitgeotrans([[xAxis(1);xAxis(end)] [yAxis(1);yAxis(end)]], [[HalfWidth;HalfWidth] [0;OnAxisDistance(end)]],'nonreflectivesimilarity');
        TransformedImage = imwarp(MicInputMean,MappingTFORM);
        figure('units','normalized','outerposition',[0 0 1 1]);
        imshow(imadjust(TransformedImage));
        disp('Please draw rectangle along the rotated microtubule to make kymograph...');
        disp('Double click inside rectangle when done.');
        h = imrect;
        RectPosition = wait(h); RectPosition = round(RectPosition);
        delete(h);
        rectangle('Position',RectPosition,'EdgeColor','red');

        RotatedMic = uint16(zeros([RectPosition(4)+1 RectPosition(3)+1 dim(3)]));
        for i = 1:size(MicInput,3)
            TransformedImage = imwarp(MicInput(:,:,i),MappingTFORM);
            RotatedMic(:,:,i) = TransformedImage(RectPosition(2):RectPosition(2)+RectPosition(4),RectPosition(1):RectPosition(1)+RectPosition(3));
        end

        MicKymograph = mean(RotatedMic,2);
        MicKymograph = permute(MicKymograph,[1 3 2]);
        figure('units','normalized','outerposition',[0 0 1 1]);
        subplot(2,1,1);
        imshow(imadjust(uint16(round(MicKymograph)))); hold on;
        title('Kymograph of Microtubule Displacement');
        xlabel('Time (Frame)'); ylabel('D (pixel)');
        set(gca,'FontSize',TextSize,'FontWeight','b'); 

        % Display progress
        disp('Done drawing polyline and rotate trace. Proceed to plotting velocity lines on kymographs...');
        %% (Repeated) Plot velocity lines on kymograph
        MicKymograph = mean(RotatedMic,2);
        MicKymograph = permute(MicKymograph,[1 3 2]);
        figure('units','normalized','outerposition',[0 0 1 1]);
        subplot(2,1,1);
        imshow(imadjust(uint16(round(MicKymograph)))); hold on;
        title('Kymograph of Microtubule Displacement');
        xlabel('Time (Frame)'); ylabel('D (pixel)');
        set(gca,'FontSize',TextSize,'FontWeight','b'); 
        
        disp('Please trace the microtubule kymograph for displacement...');
        disp('Click on an earlier time point to indicate new line....');
        disp('Do not go outside the kymograph boundary, otherwise the y position will be recorded as zero....');
        disp('Press <Enter> when done.');
        [xKymo,yKymo] = getline(gca);
        NewLineIndicator = xKymo(2:end)-xKymo(1:end-1);
        KymoLineEnd = find(NewLineIndicator<1);
        KymoLineStart = KymoLineEnd+1;
        KymoLineStart = [1; KymoLineStart]; KymoLineEnd = [KymoLineEnd; length(xKymo)];  
        %AllLines = cell(length(KymoLineStart),1);
        CombinedLine = zeros(dim(3),1);

        for i = 1:length(KymoLineStart)
            xLineFine = ceil(xKymo(KymoLineStart(i))):floor(xKymo(KymoLineEnd(i)));
            yLineFine = interp1(xKymo(KymoLineStart(i):KymoLineEnd(i)),yKymo(KymoLineStart(i):KymoLineEnd(i)),xLineFine);
            %AllLines{i} = [xLineFine yLineFine];
            plot(xLineFine,yLineFine,'r'); hold on;

            if i==1 
                CombinedLine(xLineFine) = yLineFine;
            else
                MeanEnd = find(CombinedLine~=0,1,'last');
                yLineFine = yLineFine + (CombinedLine(xLineFine(1))-yLineFine(1));

                if xLineFine(end)<MeanEnd
                    CombinedEnd = CombinedLine(xLineFine(end));
                    CombinedLine(xLineFine(1):xLineFine(end)) = (CombinedLine(xLineFine(1):xLineFine(end))+yLineFine(:))/2;
                    CombinedLine(xLineFine(end)+1:MeanEnd) = CombinedLine(xLineFine(end)+1:MeanEnd)-(CombinedEnd-CombinedLine(xLineFine(end)));
                else
                    CombinedLine(xLineFine(1):MeanEnd) = (CombinedLine(xLineFine(1):MeanEnd)+(yLineFine(1:find(xLineFine==MeanEnd)))')/2;
                    CombinedLine(MeanEnd+1:xLineFine(end)) = (yLineFine(find(xLineFine==MeanEnd)+1:end))'-(yLineFine(find(xLineFine==MeanEnd))-CombinedLine(MeanEnd));
                end
            end
        end

        CombinedLine = [(1:dim(3))' CombinedLine];
        CombinedLine(CombinedLine(:,2)==0,:)=[];
        if CombinedLine(end,2)<CombinedLine(1,2); CombinedLine(:,2)=-CombinedLine(:,2); ChangeDirection=1; end
        CombinedLine(:,2) = CombinedLine(:,2)-CombinedLine(1,2);

        % Plot figure
        subplot(2,1,2); 
        yyaxis left;
        plot(CombinedLine(:,1)*ExposureTime,CombinedLine(:,2)*PixelSize,'r','LineWidth',2);
        yyaxis left; ylabel('Displacement (nm)');
        MicVelocity = (CombinedLine(2:end,2)-CombinedLine(1:end-1,2))*PixelSize./((CombinedLine(2:end,1)-CombinedLine(1:end-1,1))*ExposureTime); hold on;
        yyaxis right;
        plot(CombinedLine(2:end,1)*ExposureTime,MicVelocity,'b','LineWidth',2);
        yyaxis right; ylabel('Velocity (nm/s)');
        xlim([0 max(CombinedLine(:,1)*ExposureTime)]);
        title('Microtubule Displacement (Red) and Velocity (Blue) over Time');
        xlabel('Time (s)'); 
        set(gca,'FontSize',TextSize,'FontWeight','b'); 
        
        
        
        
        
        % Save kin-Qd and microtubule velocity data
        FinalResult = cell(dim(3)+1,RegionNo*3+3);
        FinalResult(1,1:3) = {'Microtubule Time (s)','Microtubule Displacement (nm)','Microtubule Velocity (nm/s)'};
        for i = 1:RegionNo
            FinalResult(1,3*i+1:3*i+3) = {['Point ' num2str(i) ' Time (s)'],['Point ' num2str(i) ' On-Axis Displacement (nm)'],['Point ' num2str(i) ' Off-Axis Displacement (nm)']};
        end

        FinalResult(2:length(CombinedLine),1:3)=num2cell([CombinedLine(2:end,1)*ExposureTime CombinedLine(2:end,2)*PixelSize MicVelocity]);
        for i = 1:RegionNo
            FinalResult(2:length(RotatedTraceMatrix{i})+1,3*i+1:3*i+3) = num2cell([TraceMatrix{i}(:,4)+ExposureTime RotatedTraceMatrix{i}]);
        end

        % Write FinalResult to EXCEL spreadsheet
        xlswrite(strrep(InputName,TrackMateFileSuffix,'(Final Result).xlsx'),FinalResult);

        % Write PixelSize and ExposureTime to EXCEL spreadsheet
        xlswrite(strrep(InputName,TrackMateFileSuffix,'(Variables).xlsx'),[PixelSize ExposureTime]);

        % Save Kymograph
        print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Kymograph).png')]);
        
        % Clear StartEndMatrix for next processing
        clear StartEndMatrix;
        
        % Display progress
        disp('Done plotting velocity lines on kymographs. Proceed to plotting microtubule velocity and kin-Qd graphs...');
        %% Get velocity using edge detection and save in excel files
        clear all; fclose all; close all;

        % Define all suffixes
        KinFileSuffix = '(Kin).tif';
        MicFileSuffix = '(Mic).tif'; 
        TrackMateFileSuffix = '(KinTrackMateSpots).csv';

        TextSize = 16;
        InputName = dir(['*' TrackMateFileSuffix]);
        InputName = InputName.name;
        FileName = strrep(InputName,TrackMateFileSuffix,'');
%         disp('Downloading csv file...');
%         Input = csvread(InputName,1,3);
%         disp('Csv file downloaded...');
%         Input = Input(:,1:6);
%         Input = sortrows(Input,6);

        % Ask for pixel size
        PixelTime = inputdlg({'Pixel size (nm):','Exposure Time (s):'},'Parameters Input',1,{'160','0.2'});
        PixelSize = str2num(PixelTime{1}); ExposureTime = str2num(PixelTime{2});
        %PixelSize = 106.7; ExposureTime = 0.1;
        
        % Get microtubule data
        MicFile = strrep(InputName,TrackMateFileSuffix,MicFileSuffix);
        MicInputinfo = imfinfo(MicFile);
        dim = [MicInputinfo(1).Height MicInputinfo(1).Width length(MicInputinfo)];
        MicInput = uint16(zeros(dim));
        % Get arrays
        for i = 1:dim(3)
            MicInput(:,:,i)=imread(MicFile, i, 'Info', MicInputinfo);
        end
        MicInputMean = imadjust(uint16(round(mean(MicInput,3))));
        
        % Issue dialog box to ask for frames to rotate microtubules
        if exist('FramesMic','var')
            FramesMic = inputdlg('Choose start frames when microtubule changes angle (comma separated). Include Frame 1.','Select Frames',1,{strjoin(arrayfun(@(x) num2str(x),FramesMic,'UniformOutput',false),',')});
        else
            FramesMic = inputdlg('Choose start frames when microtubule changes angle (comma separated). Include Frame 1.','Select Frames',1,{'1'});
        end
        FramesMic = strsplit(FramesMic{1},',');   % Separate the commas to generate cells of strings
        FramesMic = cellfun(@str2num,FramesMic);  % Change cells of strings to matrix

        % Get MicInputMeans
        MicInputMeans = uint16(zeros([dim(1) dim(2) length(FramesMic)]));
        for i = 1:length(FramesMic)
            if i == length(FramesMic)
                MicInputMeans(:,:,i) = imadjust(uint16(round(mean(MicInput(:,:,FramesMic(i):dim(3)),3))));
            else
                MicInputMeans(:,:,i) = imadjust(uint16(round(mean(MicInput(:,:,FramesMic(i):FramesMic(i+1)-1),3))));
            end
        end

        % Find points to rotate microtubules
        screenSize = get(0,'screensize');
        screenWidth = screenSize(3);
        screenHeight = screenSize(4);
        hFig = figure('units','pixels','outerposition',[0.2*screenWidth 0*screenHeight 0.8*screenWidth 1*screenHeight]);
        fpos = get(hFig,'Position');
        Scaling = min([fpos(3)/size(MicInputMean,2) fpos(4)/size(MicInputMean,1)])*0.90;
        axOffset = (fpos(3:4)-[size(MicInputMean,2)*Scaling size(MicInputMean,1)*Scaling])/2;
        ha = axes('Parent',hFig,'Units','pixels',...
                'Position',[axOffset size(MicInputMean,2)*Scaling size(MicInputMean,1)*Scaling]);
        % Store rotation points in Axes
        Axes = zeros(4,length(FramesMic));
        for i = 1:length(FramesMic)
            % Draw microtubule
            imshow(MicInputMeans(:,:,i),'Parent',ha);

            % Draw polyline for microtubule trace and rotate traces
            % Put warning dialog to ask user to be careful about the order
            h = warndlg(sprintf('Click two points at start & end of microtubule.\nPress <Enter> when done.\n\nCritical: Order is important! \nTrace Microtubule starting from tail to head.\nMake sure line is parallel to most of microtubule.'), 'Trace microtubule');
            uiwait(h); 

            % Get polyline
            disp('Click two points at start & end of microtubule. Press <Enter> when done.');
            [xAxis,yAxis] = getline(gca);

            Axes(:,i) = [xAxis(1); xAxis(end); yAxis(1); yAxis(end)];
        end

        % Rotate microtubules and draw rectangle to make kymograph
        ResizeFactor = 1;
        FrameOverlap = 15;       % Define how many frames will be overlapping
        LengthThreshold = 10*ResizeFactor;   % Determine lengths of kymographs below which lines are deleted
        UseSubPixelDetection = 0;
        StopLoop = 0;
        while StopLoop == 0
            % Change ResizeFactor to 1 if no sub-pixel detection is opted
            if UseSubPixelDetection == 0
                ResizeFactor = 1;
            end
            MicKymographs = cell(length(FramesMic),1);    % Initialize MicKymographs to store kymographs
            KymoDim = [0 0];

            for i = 1:length(FramesMic)
                % Calculate angle
                Angle = atan2d(Axes(4,i)-Axes(3,i),Axes(2,i)-Axes(1,i))-90;
                TransformedImage = imrotate(MicInputMeans(:,:,i),Angle);
        %         HalfWidth = round(dim(2)/2);
        %         MappingTFORM = fitgeotrans([[Axes(1,i);Axes(2,i)] [Axes(3,i);Axes(4,i)]], [[HalfWidth;HalfWidth] [0;OnAxisDistance(end)]],'nonreflectivesimilarity');
        %         TransformedImage = imwarp(MicInputMeans(:,:,i),MappingTFORM);
                h=figure;
                imshow(imadjust(TransformedImage));
                set(h,'units','normalized','outerposition',[0 0 1 1]);
                disp('Please draw rectangle along the rotated microtubule to make kymograph...');
                disp('Double click inside rectangle when done.');
                h = imrect;
                RectPosition = wait(h); RectPosition = round(RectPosition);
                delete(h);
                rectangle('Position',RectPosition,'EdgeColor','red');

                if i == length(FramesMic)
                    StartFrame = max(FramesMic(i)-FrameOverlap,1);
                    EndFrame = dim(3);
                else
                    StartFrame = max(FramesMic(i)-FrameOverlap,1);
                    EndFrame = min(FramesMic(i+1)+FrameOverlap,dim(3));
                end

                RotatedMic = uint16(zeros([RectPosition(4)+1 RectPosition(3)+1 EndFrame-StartFrame+1]));
                RotInd = 1;
                for j = StartFrame:EndFrame
                    TransformedImage = imrotate(MicInput(:,:,j),Angle);
                    RotatedMic(:,:,RotInd) = TransformedImage(RectPosition(2):RectPosition(2)+RectPosition(4),RectPosition(1):RectPosition(1)+RectPosition(3));
                    RotInd = RotInd + 1;
                end

                MicKymograph = mean(RotatedMic,2);
                MicKymograph = permute(MicKymograph,[1 3 2]);
                MicKymographs(i) = {MicKymograph};

                % Get Kymograph Dimension
                KymoDim(1) = max(KymoDim(1),size(MicKymograph,1));
                KymoDim(2) = KymoDim(2) + size(MicKymograph,2) + 1;
            end

            % Combine MicKymographs to one Kymograph
            MicKymograph = zeros(KymoDim);
            FrameAssignment = zeros(KymoDim(2),1);
            ColumnStart = 1;
            for i = 1:length(FramesMic)
                MicKymograph(1:size(MicKymographs{i},1),ColumnStart:ColumnStart+size(MicKymographs{i},2)-1) = MicKymographs{i};

                if i == length(FramesMic)
                    StartFrame = max(FramesMic(i)-FrameOverlap,1);
                    EndFrame = dim(3);
                else
                    StartFrame = max(FramesMic(i)-FrameOverlap,1);
                    EndFrame = min(FramesMic(i+1)+FrameOverlap,dim(3));
                end

                FrameAssignment(ColumnStart:ColumnStart+size(MicKymographs{i},2)-1) = StartFrame:EndFrame;
                ColumnStart = ColumnStart + size(MicKymographs{i},2) + 1;
            end

            figure('units','normalized','outerposition',[0 0 1 1]);
            imshow(imadjust(uint16(round(MicKymograph)))); hold on;
            title('Kymograph of Microtubule Displacement');
            xlabel('Time (Frame)'); ylabel('D (pixel)');
            set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[],'yTick',[]); 

            % Plot red vertical lines for frame start and ends
            xPlot = find(ismember(FrameAssignment,FramesMic)); xPlot = xPlot(2:end);
            for ii = 1:length(xPlot)
                plot([xPlot(ii) xPlot(ii)],[0 size(MicKymograph,1)],'r');
            end

            % Save Kymograph
            print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Kymograph).png')]);
            savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Kymograph).fig')]);

            % Generate warning and ask for code path if we cannot find code in the active
            % folder and a few folder above
            DataPath = pwd;
            CodeToSearch = 'subpixelEdges.m';
            StopSearch = 0;
            SearchIndex = 0;
            while StopSearch == 0
                if isempty(dir(CodeToSearch))
                    cd ..       % Go to one folder up and look for code there
                    SearchIndex = SearchIndex + 1;
                    if SearchIndex == 6
                        % Generate warning and ask for the code path
                        h=warndlg(['Cannot find ' CodeToSearch ' in active folder. Locate folder with ' CodeToSearch ' in the next window']);
                        uiwait(h);
                        CodePath = uigetdir(DataPath,['Select folder containing ' CodeToSearch]);
                        cd(CodePath);
                        StopSearch = 1;
                    end
                else; CodePath = pwd; StopSearch = 1;
                end;
            end

            % Resize Kymograph
            KymoResized = imresize(MicKymograph,ResizeFactor);
            threshold = [5 10 15 20];

            % Loop to draw four edge detections for user to choose
            figure('units','normalized','outerposition',[0.05 0 0.95 0.95]);
            ha = tight_subplot(2,2,[.05 .01],[.1 .1],[.03 .03]);

            % Draw base image from KymoResized
            for j = 1:4
                axes(ha(j));
                imshow(imadjust(uint16(round(KymoResized))),'InitialMagnification','fit'); hold on;
            end

            % Do edge detection and plot
            for j = 1:4
                if UseSubPixelDetection == 1
                    edges = subpixelEdges(MicKymograph, threshold(j), 'SmoothingIter', 1);
                    BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                else
                    % Get velocity using edge detection
                    if j == 1
                        [BW,cannyThreshold] = edge(uint16(round(MicKymograph)),'canny');
                    elseif j == 2
                        BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.025 cannyThreshold(1)+0.05]);
                    elseif j == 3
                        BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.05 cannyThreshold(1)+0.1]);
                    elseif j == 4
                        BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.075 cannyThreshold(1)+0.15]);   
                    end
                end

                % Make skeleton out of image and remove branchpoints
                BW = bwmorph(BW,'skel');
                BW2 = bwmorph(BW,'branchpoints');
                BW(BW2)=false;

                % Remove vertical lines more than 2 pixels
                for i = 1:size(BW,2)
                    rp = regionprops(BW(:,i), 'PixelIdxList', 'Area');
                    rp = rp(abs([rp.Area])>2);
                    BW(vertcat(rp.PixelIdxList),i) = false;
                end

                % Remove branchpoints
                BW2 = bwmorph(BW,'branchpoints');
                BW(BW2)=false;
                
                % Remove regions with areas less than LengthThreshold
                rp = regionprops(BW, 'PixelIdxList', 'Area', 'Orientation');
                rp = rp(abs([rp.Area])<LengthThreshold);
                BW(vertcat(rp.PixelIdxList)) = false;

                % Remove regions which are roughly vertical
                rp = regionprops(BW, 'PixelIdxList', 'Orientation');
                rp = rp([rp.Orientation]>40 | [rp.Orientation]<-70);
                BW(vertcat(rp.PixelIdxList)) = false;
                
                % Remove regions that form loop
                rp = regionprops(BW, 'PixelList','PixelIdxList');
                for i = 1:length(rp)
                    Pixels = rp(i).PixelList;
                    PixelIDs = rp(i).PixelIdxList;
                    [N,edges,bin]=histcounts(Pixels(:,1),'BinMethod','integers');
                    Centers = (edges(2:end)+edges(1:end-1))/2;
                    DoubledIndex = find(N>1);
                    DiffMatrix = zeros(length(DoubledIndex),1);
                    DiffCount = 1;
                    for ii = DoubledIndex
                        yPix = Pixels(Pixels(:,1)==Centers(ii),2);
                        yDiff = max(yPix) - min(yPix);
                        DiffMatrix(DiffCount) = yDiff;
                        DiffCount = DiffCount + 1;
                    end

                    if ~isempty(DiffMatrix) && max(DiffMatrix)>1
                        DoubledIndex = DoubledIndex(DiffMatrix>1);
                        DiffMatrix = DiffMatrix(DiffMatrix>1);
                        
                        if DoubledIndex(1) == 1
                            StartIndex = DoubledIndex(1);
                        else
                            StartIndex = DoubledIndex(1) - 1;
                        end
                        yPix1Real = Pixels(Pixels(:,1)==Centers(StartIndex),2);
                        yPix1 = Pixels(Pixels(:,1)==Centers(StartIndex+1),2);
                        yPix1Range = yPix1(1):yPix1(end);
                        yPix1 = yPix1Range(~ismember(yPix1Range,yPix1));
%                         yPix1 = yPix1(1)+1:yPix1(2)-1;
                        
                        if length(Centers) == DoubledIndex(end)
                            EndIndex = DoubledIndex(end);
                        else
                            EndIndex = DoubledIndex(end) + 1;
                        end
                        yPix2Real = Pixels(Pixels(:,1)==Centers(EndIndex),2);
                        yPix2 = Pixels(Pixels(:,1)==Centers(EndIndex-1),2);
                        yPix2Range = yPix2(1):yPix2(end);
                        yPix2 = yPix2Range(~ismember(yPix2Range,yPix2));

                        if ismember(yPix1,yPix1Real)        % Delete DoubledIndex(1) side if yPix1 is a member of yPix1Real, which means it is the side forming loop
                            BW(PixelIDs(Pixels(:,1)==Centers(StartIndex) | Pixels(:,1)==Centers(StartIndex+1))) = false;
                        elseif ismember(yPix2,yPix2Real)
                            BW(PixelIDs(Pixels(:,1)==Centers(EndIndex) | Pixels(:,1)==Centers(EndIndex-1))) = false;
                        end
                    end
                end
                
                % Remove regions with areas less than LengthThreshold
                rp = regionprops(BW, 'PixelIdxList', 'Area', 'Orientation');
                rp = rp(abs([rp.Area])<LengthThreshold);
                BW(vertcat(rp.PixelIdxList)) = false;

                % Plot green image then add alphadata to make only those with BW green
                axes(ha(j));
                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                h2 = imshow(green);
                set(h2, 'AlphaData', BW);
                title(num2str(j));
                set(gca,'FontSize',18,'FontWeight','b');

                % Plot red vertical lines for frame start and ends
                xPlot = find(ismember(FrameAssignment,FramesMic)); xPlot = xPlot(2:end);
                for ii = 1:length(xPlot)
                    plot([xPlot(ii)*ResizeFactor xPlot(ii)*ResizeFactor],[0 size(MicKymograph,1)*ResizeFactor],'r');
                end

                % Mark unconnected regions with blue lines and dots
                rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                for ii = 1:length(rp)
                    Pixels = rp(ii).PixelList;
                    PixelIDs = rp(ii).PixelIdxList;
                    PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    rp(ii).PixelIdxList = PixelIDs;
                    rp(ii).PixelList = Pixels;
                end
                BW2 = logical(zeros(size(BW)));
                BW2(vertcat(rp.PixelIdxList)) = true;
                MaxBW = max(BW2);
                FirstInd = find(MaxBW==1,1,'first');
                LastInd = find(MaxBW==1,1,'last');
                rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                hPlots = zeros(length(rp2),4);
                for i = 1:length(rp2)
                    hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                end
            end

            % Allow zooming before choosing 
            disp('Please choose which edge detection to use...');
            h = warndlg(sprintf('Please choose which edge detection to use...\nZoom in/out or pan to desired region before clicking OK.'), 'Zoom/ Pan');
            set(h, 'Position', [50 250 250 60]);
%             set(h, 'Position', [400 680 250 60]);
            uiwait(h); 

            % Get a dialog box to ask which edge detection to choose
            [choice,v] = listdlg('PromptString','Which edge detection parameter to use?',...
                    'SelectionMode','single', 'ListSize', [200 75], ...
                    'ListString',{'1','2','3','4','Change Variables'});
            if UseSubPixelDetection == 1
                switch choice
                    case 1
                        edges = subpixelEdges(MicKymograph, threshold(1), 'SmoothingIter', 4);
                        BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                        StopLoop = 1;
                    case 2
                        edges = subpixelEdges(MicKymograph, threshold(2), 'SmoothingIter', 4);
                        BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                        StopLoop = 1;
                    case 3
                        edges = subpixelEdges(MicKymograph, threshold(3), 'SmoothingIter', 4);
                        BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                        StopLoop = 1;
                    case 4
                        edges = subpixelEdges(MicKymograph, threshold(4), 'SmoothingIter', 4);
                        BW = subpixelImage(edges, size(MicKymograph), ResizeFactor);
                        StopLoop = 1;
                    case 5
                        % Get user input for FrameOverlap, LengthThreshold and UseSubPixelDetection
                        NewVariables = inputdlg({['Frame Overlap (Last value = ' num2str(FrameOverlap) '):'], ['Length Threshold (Last value = ' num2str(LengthThreshold) '):'], ['Use SubPixel Detection (0 or 1)? (Last value = ' num2str(UseSubPixelDetection) '):'], ['Resize Factor (1 for No SubPixel Det.) (Last value = ' num2str(ResizeFactor) '):']},...
                                     'Enter new variable values', [1 65],{num2str(FrameOverlap),num2str(LengthThreshold),num2str(UseSubPixelDetection),num2str(ResizeFactor)});
                        NewVariables = cellfun(@str2num,NewVariables);
                        FrameOverlap = NewVariables(1);
                        LengthThreshold = NewVariables(2);
                        UseSubPixelDetection = NewVariables(3);
                        ResizeFactor = NewVariables(4);
                end
            else
                switch choice
                    case 1
                        BW = edge(uint16(round(MicKymograph)),'canny');
                        StopLoop = 1;
                    case 2
                        BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.025 cannyThreshold(1)+0.05]);
                        StopLoop = 1;
                    case 3
                        BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.05 cannyThreshold(1)+0.1]);
                        StopLoop = 1;
                    case 4
                        BW = edge(uint16(round(MicKymograph)),'canny',[cannyThreshold(1)+0.075 cannyThreshold(1)+0.15]);
                        StopLoop = 1;
                    case 5
                        % Get user input for FrameOverlap, LengthThreshold and UseSubPixelDetection
                        NewVariables = inputdlg({['Frame Overlap (Last value = ' num2str(FrameOverlap) '):'], ['Length Threshold (Last value = ' num2str(LengthThreshold) '):'], ['Use SubPixel Detection (0 or 1)? (Last value = ' num2str(UseSubPixelDetection) '):'], ['Resize Factor (1 for No SubPixel Det.) (Last value = ' num2str(ResizeFactor) '):']},...
                                     'Enter new variable values', [1 65],{num2str(FrameOverlap),num2str(LengthThreshold),num2str(UseSubPixelDetection),num2str(ResizeFactor)});
                        NewVariables = cellfun(@str2num,NewVariables);
                        FrameOverlap = NewVariables(1);
                        LengthThreshold = NewVariables(2);
                        UseSubPixelDetection = NewVariables(3);
                        ResizeFactor = NewVariables(4);
                end
            end
        end

        % Draw out chosen edge detection
        % Make skeleton out of image and remove branchpoints
        BW = bwmorph(BW,'skel');
        BW2 = bwmorph(BW,'branchpoints');
        BW(BW2)=false;
        % Remove vertical lines more than 2 pixels
        for i = 1:size(BW,2)
            rp = regionprops(BW(:,i), 'PixelIdxList', 'Area');
            rp = rp(abs([rp.Area])>2);
            BW(vertcat(rp.PixelIdxList),i) = false;
        end
        % Remove branchpoints
        BW2 = bwmorph(BW,'branchpoints');
        BW(BW2)=false;
        % Remove regions with areas less than LengthThreshold
        rp = regionprops(BW, 'PixelIdxList', 'Area');
        rp = rp(abs([rp.Area])<LengthThreshold);
        BW(vertcat(rp.PixelIdxList)) = false;
        % Remove regions which are roughly vertical
        rp = regionprops(BW, 'PixelIdxList', 'Orientation');
        rp = rp([rp.Orientation]>50 | [rp.Orientation]<-80);
        BW(vertcat(rp.PixelIdxList)) = false;
        % Remove regions that form loop
        rp = regionprops(BW, 'PixelList','PixelIdxList');
        for i = 1:length(rp)
            Pixels = rp(i).PixelList;
            PixelIDs = rp(i).PixelIdxList;
            [N,edges,bin]=histcounts(Pixels(:,1),'BinMethod','integers');
            Centers = (edges(2:end)+edges(1:end-1))/2;
            DoubledIndex = find(N>1);
            DiffMatrix = zeros(length(DoubledIndex),1);
            DiffCount = 1;
            for ii = DoubledIndex
                yPix = Pixels(Pixels(:,1)==Centers(ii),2);
                yDiff = max(yPix) - min(yPix);
                DiffMatrix(DiffCount) = yDiff;
                DiffCount = DiffCount + 1;
            end

            if ~isempty(DiffMatrix) && max(DiffMatrix)>1
                DoubledIndex = DoubledIndex(DiffMatrix>1);
                DiffMatrix = DiffMatrix(DiffMatrix>1);

                if DoubledIndex(1) == 1
                    StartIndex = DoubledIndex(1);
                else
                    StartIndex = DoubledIndex(1) - 1;
                end
                yPix1Real = Pixels(Pixels(:,1)==Centers(StartIndex),2);
                yPix1 = Pixels(Pixels(:,1)==Centers(StartIndex+1),2);
                yPix1Range = yPix1(1):yPix1(end);
                yPix1 = yPix1Range(~ismember(yPix1Range,yPix1));
%                         yPix1 = yPix1(1)+1:yPix1(2)-1;

                if length(Centers) == DoubledIndex(end)
                    EndIndex = DoubledIndex(end);
                else
                    EndIndex = DoubledIndex(end) + 1;
                end
                yPix2Real = Pixels(Pixels(:,1)==Centers(EndIndex),2);
                yPix2 = Pixels(Pixels(:,1)==Centers(EndIndex-1),2);
                yPix2Range = yPix2(1):yPix2(end);
                yPix2 = yPix2Range(~ismember(yPix2Range,yPix2));

                if ismember(yPix1,yPix1Real)        % Delete DoubledIndex(1) side if yPix1 is a member of yPix1Real, which means it is the side forming loop
                    BW(PixelIDs(Pixels(:,1)==Centers(StartIndex) | Pixels(:,1)==Centers(StartIndex+1))) = false;
                elseif ismember(yPix2,yPix2Real)
                    BW(PixelIDs(Pixels(:,1)==Centers(EndIndex) | Pixels(:,1)==Centers(EndIndex-1))) = false;
                end
            end
        end
        % Remove regions with areas less than LengthThreshold
        rp = regionprops(BW, 'PixelIdxList', 'Area', 'Orientation');
        rp = rp(abs([rp.Area])<LengthThreshold);
        BW(vertcat(rp.PixelIdxList)) = false;
        % Plot processed BW image
        screenSize = get(0,'screensize');
        screenWidth = screenSize(3);
        screenHeight = screenSize(4);
        hFig = figure('units','pixels','outerposition',[0.05*screenWidth 0*screenHeight 0.95*screenWidth 0.9*screenHeight]);
        fpos = get(hFig,'Position');
        Scaling = min([fpos(3)/size(BW,2) fpos(4)/size(BW,1)])*0.90;
        axOffset = (fpos(3:4)-[size(BW,2)*Scaling size(BW,1)*Scaling])/2;
        ha = axes('Parent',hFig,'Units','pixels',...
                'Position',[axOffset size(BW,2)*Scaling size(BW,1)*Scaling]);
        h1 = imshow(imadjust(uint16(round(KymoResized))),'InitialMagnification','fit','Parent',ha); hold on;
        green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
        h2 = imshow(green);
        set(h2, 'AlphaData', BW);
        title(num2str(choice));
        set(gca,'FontSize',18,'FontWeight','b');
        % Plot red vertical lines for frame start and ends
        xPlot = find(ismember(FrameAssignment,FramesMic)); xPlot = xPlot(2:end);
        for ii = 1:length(xPlot)
            plot([xPlot(ii)*ResizeFactor xPlot(ii)*ResizeFactor],[0 size(MicKymograph,1)*ResizeFactor],'r');
        end
        % Mark unconnected regions with blue lines and dots
        rp = regionprops(BW, 'PixelIdxList', 'PixelList');
        for ii = 1:length(rp)
            Pixels = rp(ii).PixelList;
            PixelIDs = rp(ii).PixelIdxList;
            PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            rp(ii).PixelIdxList = PixelIDs;
            rp(ii).PixelList = Pixels;
        end
        BW2 = logical(zeros(size(BW)));
        BW2(vertcat(rp.PixelIdxList)) = true;
        MaxBW = max(BW2);
        FirstInd = find(MaxBW==1,1,'first');
        LastInd = find(MaxBW==1,1,'last');
        rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
        hPlots = zeros(length(rp2),4);
        for i = 1:length(rp2)
            hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
        end

        % Loop to delete regions
        % Get the rows and columns of all ones in BW
        [yPos,xPos] = ind2sub(size(BW),find(BW==1)); 
        % Create region to delete and loop to delete regions
        rp = regionprops(BW, 'Area');
        DeletedPoints = cell(length(rp),1);
        DeleteInd = 1;
        StopLoop = 0;
        while StopLoop == 0
            % Get polyline
            pan on;
            disp('Please identify regions to delete. Double click when done to close polygon.');
            h = warndlg(sprintf('(Delete) Zoom in/out or pan to desired region before clicking OK'), 'Delete Zoom/ Pan');
            set(h, 'Position', [50 250 280 60]);
%             set(h, 'Position', [400 680 280 60]);
            uiwait(h); 
            choice = questdlg('Continue deleting?','Continue?', ...
                    'Yes','No','Redo','Yes');

            switch choice
                case 'Yes'
                    % Get polyline
                    disp('Please select area to delete edges. Double click when done to close polygon.');
%                     [xBorder,yBorder] = getline(gca);
                    
                    h = imfreehand;
                    Borders = getPosition(h);
                    delete(h)
                    
                    % Find points within border
                    BoundPoints = inpolygon(xPos,yPos,Borders(:,1),Borders(:,2));

%                     % Draw polyline
%                     plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'r'); 

%                     % Find points within polyline
%                     BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                    % Delete all BoundPoints
                    DeletedPoints(DeleteInd) = {[yPos(BoundPoints) xPos(BoundPoints)]};
                    BW(sub2ind(size(BW),yPos(BoundPoints),xPos(BoundPoints))) = false;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);
                    DeleteInd = DeleteInd + 1;

                    % Mark unconnected regions with blue lines and dots
                    delete(hPlots);
                    rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                    for ii = 1:length(rp)
                        Pixels = rp(ii).PixelList;
                        PixelIDs = rp(ii).PixelIdxList;
                        PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                        Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                        rp(ii).PixelIdxList = PixelIDs;
                        rp(ii).PixelList = Pixels;
                    end
                    BW2 = logical(zeros(size(BW)));
                    BW2(vertcat(rp.PixelIdxList)) = true;
                    MaxBW = max(BW2);
                    FirstInd = find(MaxBW==1,1,'first');
                    LastInd = find(MaxBW==1,1,'last');
                    rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                    hPlots = zeros(length(rp2),4);
                    for i = 1:length(rp2)
                        hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                        hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                        hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                        hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    end
                case 'No'
                    StopLoop = 1;
                case 'Redo'
                    DeleteInd = DeleteInd - 1;
                    BW(sub2ind(size(BW),DeletedPoints{DeleteInd}(:,1),DeletedPoints{DeleteInd}(:,2))) = true;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);

                    % Mark unconnected regions with blue lines and dots
                    delete(hPlots);
                    rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                    for ii = 1:length(rp)
                        Pixels = rp(ii).PixelList;
                        PixelIDs = rp(ii).PixelIdxList;
                        PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                        Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                        rp(ii).PixelIdxList = PixelIDs;
                        rp(ii).PixelList = Pixels;
                    end
                    BW2 = logical(zeros(size(BW)));
                    BW2(vertcat(rp.PixelIdxList)) = true;
                    MaxBW = max(BW2);
                    FirstInd = find(MaxBW==1,1,'first');
                    LastInd = find(MaxBW==1,1,'last');
                    rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                    hPlots = zeros(length(rp2),4);
                    for i = 1:length(rp2)
                        hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                        hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                        hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                        hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    end
            end
        end

        % Mark unconnected regions with blue lines and dots
        delete(hPlots);
        rp = regionprops(BW, 'PixelIdxList', 'PixelList');
        for ii = 1:length(rp)
            Pixels = rp(ii).PixelList;
            PixelIDs = rp(ii).PixelIdxList;
            PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            rp(ii).PixelIdxList = PixelIDs;
            rp(ii).PixelList = Pixels;
        end
        BW2 = logical(zeros(size(BW)));
        BW2(vertcat(rp.PixelIdxList)) = true;
        MaxBW = max(BW2);
        FirstInd = find(MaxBW==1,1,'first');
        LastInd = find(MaxBW==1,1,'last');
        rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
        hPlots = zeros(length(rp2),4);
        for i = 1:length(rp2)
            hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
        end

        % Patch unconnected regions and allow deletion
        zoom out;
        StopLoop = 0;
        RedoMatrix = zeros(length(rp),1);   % 0 if 'Patch' is selection, 1 if 'Delete' is selected
        RedoInd = 1;
        PatchPoints = cell(length(rp),1);
        PatchInd = 1;
        while StopLoop == 0
            % Get polyline
            pan on;
            disp('Please draw regions to patch (especially areas marked red). Double click when done to close polygon.');
            h = warndlg(sprintf('(Patching) Zoom in/out or pan to desired region before clicking OK'), 'Patching Zoom/ Pan');
            set(h, 'Position', [50 250 290 60]);
%             set(h, 'Position', [400 680 290 60]);
            uiwait(h); 
        %             choice = questdlg('Continue patching/ deleting?','Continue?', ...
        %                     'Patch','Delete','Redo','No','Patch');
            [choice,v] = listdlg('PromptString','Continue?',...
                'SelectionMode','single', 'ListSize', [100 60], ...
                'ListString',{'Patch','Delete','Redo','No'});
            [yPos,xPos] = ind2sub(size(BW),find(BW==1));    % Get the rows and columns of all ones in BW

            switch choice
                case 1
                    % Get polyline
                    disp('Please select area to delete edges. Double click when done to close polygon.');
                    [xBorder,yBorder] = getline(gca);

        %                 % Draw polyline
        %                 plot(xBorder,yBorder,'r');

                    % Get pixels to patch
                    xBorder = round(xBorder);
                    yBorder = round(yBorder);
                    tBorder = 1:length(xBorder);
                    tInterp = 1:0.001:length(xBorder);
                    xInterp = round(interp1(tBorder,xBorder,tInterp,'linear'))';
                    yInterp = round(interp1(tBorder,yBorder,tInterp,'linear'))';

                    % Get unique values
                    PatchPixels = unique([yInterp xInterp],'rows');

                    % Draw patch
                    BW(sub2ind(size(BW),PatchPixels(:,1),PatchPixels(:,2))) = true;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);

                    % Remember Patch
                    PatchPoints(PatchInd) = {PatchPixels};

                    % Increase RedoInd and PatchInd. RedoMatrix is automatically zero
                    RedoMatrix(RedoInd) = 0;
                    RedoInd = RedoInd + 1;
                    PatchInd = PatchInd + 1;
                case 2
                    % Get polyline
%                     disp('Please select area to delete edges. Double click when done to close polygon.');
%                     [xBorder,yBorder] = getline(gca);

        %                 % Draw polyline
        %                 plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'r'); 
                    
                    % Start free hand selection
                    h = imfreehand;
                    Borders = getPosition(h);
                    delete(h)
                    
                    % Find points within border
                    BoundPoints = inpolygon(xPos,yPos,Borders(:,1),Borders(:,2));

%                     % Find points within polyline
%                     BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                    % Delete all BoundPoints
                    DeletedPoints(DeleteInd) = {[yPos(BoundPoints) xPos(BoundPoints)]};
                    BW(sub2ind(size(BW),yPos(BoundPoints),xPos(BoundPoints))) = false;
                    delete(h2);
                    green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                    h2 = imshow(green);
                    set(h2, 'AlphaData', BW);
                    DeleteInd = DeleteInd + 1;
                    RedoMatrix(RedoInd) = 1;
                    RedoInd = RedoInd + 1;
                case 3
                    RedoInd = RedoInd - 1;
                    if RedoMatrix(RedoInd) == 0
                        % Start redoing 'Patch'
                        PatchInd = PatchInd - 1;
                        BW(sub2ind(size(BW),PatchPoints{PatchInd}(:,1),PatchPoints{PatchInd}(:,2))) = false;
                        delete(h2);
                        green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                        h2 = imshow(green);
                        set(h2, 'AlphaData', BW);
                    else
                        % Start redoing 'Delete'
                        DeleteInd = DeleteInd - 1;
                        BW(sub2ind(size(BW),DeletedPoints{DeleteInd}(:,1),DeletedPoints{DeleteInd}(:,2))) = true;
                        delete(h2);
                        green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                        h2 = imshow(green);
                        set(h2, 'AlphaData', BW);
                    end
                case 4
                    StopLoop = 1;
            end

            % Mark unconnected regions with blue lines and dots
            delete(hPlots);
            rp = regionprops(BW, 'PixelIdxList', 'PixelList');
            for ii = 1:length(rp)
                Pixels = rp(ii).PixelList;
                PixelIDs = rp(ii).PixelIdxList;
                PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                rp(ii).PixelIdxList = PixelIDs;
                rp(ii).PixelList = Pixels;
            end
            BW2 = logical(zeros(size(BW)));
            BW2(vertcat(rp.PixelIdxList)) = true;
            MaxBW = max(BW2);
            FirstInd = find(MaxBW==1,1,'first');
            LastInd = find(MaxBW==1,1,'last');
            rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
            hPlots = zeros(length(rp2),4);
            for i = 1:length(rp2)
                hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            end
        end
        % Zoom out
        zoom out;

        % Final patch and delete with base image off
        delete(h1); delete(hPlots);
        % Mark unconnected regions with blue lines and dots
        rp = regionprops(BW, 'PixelIdxList', 'PixelList');
        for ii = 1:length(rp)
            Pixels = rp(ii).PixelList;
            PixelIDs = rp(ii).PixelIdxList;
            PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
            rp(ii).PixelIdxList = PixelIDs;
            rp(ii).PixelList = Pixels;
        end
        BW2 = logical(zeros(size(BW)));
        BW2(vertcat(rp.PixelIdxList)) = true;
        MaxBW = max(BW2);
        FirstInd = find(MaxBW==1,1,'first');
        LastInd = find(MaxBW==1,1,'last');
        rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
        hPlots = zeros(length(rp2),4);
        for i = 1:length(rp2)
            hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
            hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
        end
        FinalPatchLoop = 0;
        while FinalPatchLoop == 0
            % Start patching unconnected area. Allow deletion too.
            StopLoop = 0;
            RedoMatrix = zeros(length(rp),1);   % 0 if 'Patch' is selection, 1 if 'Delete' is selected
            RedoInd = 1;
            PatchPoints = cell(length(rp),1);
            PatchInd = 1;
            while StopLoop == 0
                % Get polyline
                pan on;
                disp('Please draw regions to patch (especially areas marked red). Double click when done to close polygon.');
                h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK'), 'Zoom/ Pan');
                set(h, 'Position', [50 250 250 60]);
%                 set(h, 'Position', [400 680 250 60]);
                uiwait(h); 
        %             choice = questdlg('Continue patching/ deleting?','Continue?', ...
        %                     'Patch','Delete','Redo','No','Patch');
                [choice,v] = listdlg('PromptString','Continue?',...
                    'SelectionMode','single', 'ListSize', [100 60], ...
                    'ListString',{'Patch','Delete','Redo','No'});
                [yPos,xPos] = ind2sub(size(BW),find(BW==1));    % Get the rows and columns of all ones in BW

                switch choice
                    case 1
                        % Get polyline
                        disp('Please select area to delete edges. Double click when done to close polygon.');
                        [xBorder,yBorder] = getline(gca);

        %                 % Draw polyline
        %                 plot(xBorder,yBorder,'r');

                        % Get pixels to patch
                        xBorder = round(xBorder);
                        yBorder = round(yBorder);
                        tBorder = 1:length(xBorder);
                        tInterp = 1:0.001:length(xBorder);
                        xInterp = round(interp1(tBorder,xBorder,tInterp,'linear'))';
                        yInterp = round(interp1(tBorder,yBorder,tInterp,'linear'))';

                        % Get unique values
                        PatchPixels = unique([yInterp xInterp],'rows');

                        % Draw patch
                        BW(sub2ind(size(BW),PatchPixels(:,1),PatchPixels(:,2))) = true;
                        delete(h2);
                        green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                        h2 = imshow(green);
                        set(h2, 'AlphaData', BW);

                        % Remember Patch
                        PatchPoints(PatchInd) = {PatchPixels};

                        % Increase RedoInd and PatchInd. RedoMatrix is automatically zero
                        RedoMatrix(RedoInd) = 0;
                        RedoInd = RedoInd + 1;
                        PatchInd = PatchInd + 1;
                    case 2
%                         % Get polyline
%                         disp('Please select area to delete edges. Double click when done to close polygon.');
%                         [xBorder,yBorder] = getline(gca);
% 
%                         % Find points within polyline
%                         BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                        % Start free hand selection
                        h = imfreehand;
                        Borders = getPosition(h);
                        delete(h)

                        % Find points within border
                        BoundPoints = inpolygon(xPos,yPos,Borders(:,1),Borders(:,2));

                        % Delete all BoundPoints
                        DeletedPoints(DeleteInd) = {[yPos(BoundPoints) xPos(BoundPoints)]};
                        BW(sub2ind(size(BW),yPos(BoundPoints),xPos(BoundPoints))) = false;
                        delete(h2);
                        green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                        h2 = imshow(green);
                        set(h2, 'AlphaData', BW);
                        DeleteInd = DeleteInd + 1;
                        RedoMatrix(RedoInd) = 1;
                        RedoInd = RedoInd + 1;
                    case 3
                        RedoInd = RedoInd - 1;
                        if RedoMatrix(RedoInd) == 0
                            % Start redoing 'Patch'
                            PatchInd = PatchInd - 1;
                            BW(sub2ind(size(BW),PatchPoints{PatchInd}(:,1),PatchPoints{PatchInd}(:,2))) = false;
                            delete(h2);
                            green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                            h2 = imshow(green);
                            set(h2, 'AlphaData', BW);
                        else
                            % Start redoing 'Delete'
                            DeleteInd = DeleteInd - 1;
                            BW(sub2ind(size(BW),DeletedPoints{DeleteInd}(:,1),DeletedPoints{DeleteInd}(:,2))) = true;
                            delete(h2);
                            green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                            h2 = imshow(green);
                            set(h2, 'AlphaData', BW);
                        end
                    case 4
                        StopLoop = 1;
                end

                % Mark unconnected regions with blue lines and dots
                delete(hPlots);
                rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                for ii = 1:length(rp)
                    Pixels = rp(ii).PixelList;
                    PixelIDs = rp(ii).PixelIdxList;
                    PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                    rp(ii).PixelIdxList = PixelIDs;
                    rp(ii).PixelList = Pixels;
                end
                BW2 = logical(zeros(size(BW)));
                BW2(vertcat(rp.PixelIdxList)) = true;
                MaxBW = max(BW2);
                FirstInd = find(MaxBW==1,1,'first');
                LastInd = find(MaxBW==1,1,'last');
                rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                hPlots = zeros(length(rp2),4);
                for i = 1:length(rp2)
                    hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                    hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                end
            end

            % Ask if we want to stop
            choice = questdlg('Continue patching/ deleting?','Continue?', ...
                    'Yes','No','Yes');
            switch choice
                case 'Yes'
                case 'No'
                    FinalPatchLoop = 1;
            end
        end

        % Prepare to save Processed Kymograph, then save
        zoom out;
        set(hFig,'units','pixels','outerposition',[0.05*screenWidth 0.01*screenHeight 0.95*screenWidth 0.95*screenHeight]);
        title(['Processed Kymograph (Fr. Overlap: ' num2str(FrameOverlap) '. Length Thres.: ' num2str(LengthThreshold) '. SubPixel Det.: ' num2str(UseSubPixelDetection) '. Resize Fact.: ' num2str(ResizeFactor) ')']);
        xlabel('Time (Frame)'); ylabel('D (pixel)');
        set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[],'yTick',[]);
        % Save Processed Kymograph and BW
        cd(DataPath);
        print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Processed Kymograph).png')]);
        savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Processed Kymograph).fig')]);
        save(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Binary Kymograph)')],'BW');

        % Compile all velocities
        rp = regionprops(BW, 'PixelList');
        Velocities = NaN(length(rp),size(BW,2)/ResizeFactor);
        for i = 1:length(rp)
            % Average all rows in the same column
            ColumnRow = sortrows(rp(i).PixelList,1);
            [Column,ia,ic] = unique(ColumnRow(:,1));
        %         h = histogram(ic,'BinMethod','integers');
        %         DuplicatedIndices = find(h.Values>1);
            h = histcounts(ic,'BinMethod','integers');
            DuplicatedIndices = find(h>1);
            Row = ColumnRow(ia,2);
            % Find mean for duplicated indices
            for ii = 1:length(DuplicatedIndices)
                Row(DuplicatedIndices(ii)) = mean(ColumnRow(ic==DuplicatedIndices(ii),2));
            end
            % Get smoothed data
            SmoothedRow = smooth(Row,5*ResizeFactor);
            Velocity = (SmoothedRow(2:end) - SmoothedRow(1:end-1))*PixelSize./((Column(2:end)-Column(1:end-1))*ExposureTime);   % Velocity in nm/s
            NewColumn = ((Column-1)/ResizeFactor)+1; NewColumn = NewColumn(2:end);
            VelIndex = find(rem(NewColumn,1)==0,1);
            VelLastIndex = find(rem(NewColumn(1:end-ResizeFactor+1),1)==0,1,'last');
            NewColumn = NewColumn(VelIndex:ResizeFactor:VelLastIndex);
            NewVelocity = mean(reshape(Velocity(VelIndex:VelLastIndex+ResizeFactor-1),[],ResizeFactor),2);
            if VelIndex > 1
                if isempty(VelLastIndex)
                    NewVelocity = [mean(Velocity(1:VelIndex-1)); NewVelocity];
                    NewColumn = ((Column-1)/ResizeFactor)+1; NewColumn = NewColumn(2:end);
                    NewColumn = floor(NewColumn(1));
                else
                    NewVelocity = [mean(Velocity(1:VelIndex-1)); NewVelocity];
                    NewColumn = [NewColumn(1)-1; NewColumn];
                end
            end
            if VelLastIndex+ResizeFactor-1 < length(Velocity)
                NewVelocity = [NewVelocity; mean(Velocity(VelLastIndex+ResizeFactor:end))];
                NewColumn = [NewColumn; NewColumn(end)+1];
            end
            Velocities(i,FrameAssignment(NewColumn)) = NewVelocity';
        end

        % Average over column to get mean velocities at every time point
        MeanVel = nanmean(Velocities,1);

        % Get displacement at every time point
        Displacement = cumsum(MeanVel*ExposureTime,'omitnan');

        % Plot Mean Velocity
        figure;
        plot((1:size(BW,2)/ResizeFactor)*ExposureTime,MeanVel,'LineWidth',2);
        ylabel('Velocity (nm/s)');
        xlim([0 (size(BW,2)/ResizeFactor)*ExposureTime]);
        xlabel('Time (s)');
        title('Microtubule Velocity over Time');
        set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[]); 

        % Save velocity figure
        print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Velocity).png')]);
        savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Velocity).fig')]);

        % Draw Displacement diagram
        figure;
        plot((1:size(BW,2)/ResizeFactor)*ExposureTime,Displacement/1000,'LineWidth',2);
        ylabel('Disp. (um)');
        xlim([0 (size(BW,2)/ResizeFactor)*ExposureTime]);
        xlabel('Time (s)');
        title('Microtubule Disp. over Time');
        set(gca,'FontSize',TextSize,'FontWeight','b');

        % Save displacement figure
        print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Displacement).png')]);
        savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Displacement).fig')]);

        MicTimeDispVel = [(1:(size(BW,2)/ResizeFactor))'*ExposureTime Displacement' MeanVel'];
        MicTimeDispVel(isnan(MeanVel),:) = [];
        
        % Import the three Final Result files
        % Final Result
        [~,~,raw] = xlsread([FileName '(Final Result).xlsx']);
        raw(2:end,1:3) = {NaN};
        raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
        xlswrite([FileName '(Final Result).xlsx'],raw);     % Write FinalResult to EXCEL spreadsheet
        % Final Result2
        [~,~,raw] = xlsread([FileName '(Final Result2).xlsx']);
        raw(2:end,1:3) = {NaN};
        raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
        xlswrite([FileName '(Final Result2).xlsx'],raw);    % Write FinalResult to EXCEL spreadsheet
        % Final Result3
        [~,~,raw] = xlsread([FileName '(Final Result3).xlsx']);
        raw(2:end,1:3) = {NaN};
        raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
        xlswrite([FileName '(Final Result3).xlsx'],raw);    % Write FinalResult to EXCEL spreadsheet

        % Clear StartEndMatrix for next processing
        clear StartEndMatrix;

        % Display progress
        disp('Done importing positions. Proceed to plotting microtubule velocity and kin-Qd graphs...');
            %% (Repeat) Final patch and delete with base image off
            delete(h1); delete(hPlots);
            % Mark unconnected regions with blue lines and dots
            rp = regionprops(BW, 'PixelIdxList', 'PixelList');
            for ii = 1:length(rp)
                Pixels = rp(ii).PixelList;
                PixelIDs = rp(ii).PixelIdxList;
                PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                rp(ii).PixelIdxList = PixelIDs;
                rp(ii).PixelList = Pixels;
            end
            BW2 = logical(zeros(size(BW)));
            BW2(vertcat(rp.PixelIdxList)) = true;
            MaxBW = max(BW2);
            FirstInd = find(MaxBW==1,1,'first');
            LastInd = find(MaxBW==1,1,'last');
            rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
            hPlots = zeros(length(rp2),4);
            for i = 1:length(rp2)
                hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
            end
            FinalPatchLoop = 0;
            while FinalPatchLoop == 0
                % Start patching unconnected area. Allow deletion too.
                StopLoop = 0;
                RedoMatrix = zeros(length(rp),1);   % 0 if 'Patch' is selection, 1 if 'Delete' is selected
                RedoInd = 1;
                PatchPoints = cell(length(rp),1);
                PatchInd = 1;
                while StopLoop == 0
                    % Get polyline
                    pan on;
                    disp('Please draw regions to patch (especially areas marked red). Double click when done to close polygon.');
                    h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK'), 'Zoom/ Pan');
                    set(h, 'Position', [50 250 250 60]);
%                     set(h, 'Position', [400 680 250 60]);
                    uiwait(h); 
            %             choice = questdlg('Continue patching/ deleting?','Continue?', ...
            %                     'Patch','Delete','Redo','No','Patch');
                    [choice,v] = listdlg('PromptString','Continue?',...
                        'SelectionMode','single', 'ListSize', [100 60], ...
                        'ListString',{'Patch','Delete','Redo','No'});
                    [yPos,xPos] = ind2sub(size(BW),find(BW==1));    % Get the rows and columns of all ones in BW

                    switch choice
                        case 1
                            % Get polyline
                            disp('Please select area to delete edges. Double click when done to close polygon.');
                            [xBorder,yBorder] = getline(gca);

            %                 % Draw polyline
            %                 plot(xBorder,yBorder,'r');

                            % Get pixels to patch
                            xBorder = round(xBorder);
                            yBorder = round(yBorder);
                            tBorder = 1:length(xBorder);
                            tInterp = 1:0.001:length(xBorder);
                            xInterp = round(interp1(tBorder,xBorder,tInterp,'linear'))';
                            yInterp = round(interp1(tBorder,yBorder,tInterp,'linear'))';

                            % Get unique values
                            PatchPixels = unique([yInterp xInterp],'rows');

                            % Draw patch
                            BW(sub2ind(size(BW),PatchPixels(:,1),PatchPixels(:,2))) = true;
                            delete(h2);
                            green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                            h2 = imshow(green);
                            set(h2, 'AlphaData', BW);

                            % Remember Patch
                            PatchPoints(PatchInd) = {PatchPixels};

                            % Increase RedoInd and PatchInd. RedoMatrix is automatically zero
                            RedoMatrix(RedoInd) = 0;
                            RedoInd = RedoInd + 1;
                            PatchInd = PatchInd + 1;
                        case 2
    %                         % Get polyline
    %                         disp('Please select area to delete edges. Double click when done to close polygon.');
    %                         [xBorder,yBorder] = getline(gca);
    % 
    %                         % Find points within polyline
    %                         BoundPoints = inpolygon(xPos,yPos,xBorder,yBorder);

                            % Start free hand selection
                            h = imfreehand;
                            Borders = getPosition(h);
                            delete(h)

                            % Find points within border
                            BoundPoints = inpolygon(xPos,yPos,Borders(:,1),Borders(:,2));

                            % Delete all BoundPoints
                            DeletedPoints(DeleteInd) = {[yPos(BoundPoints) xPos(BoundPoints)]};
                            BW(sub2ind(size(BW),yPos(BoundPoints),xPos(BoundPoints))) = false;
                            delete(h2);
                            green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                            h2 = imshow(green);
                            set(h2, 'AlphaData', BW);
                            DeleteInd = DeleteInd + 1;
                            RedoMatrix(RedoInd) = 1;
                            RedoInd = RedoInd + 1;
                        case 3
                            RedoInd = RedoInd - 1;
                            if RedoMatrix(RedoInd) == 0
                                % Start redoing 'Patch'
                                PatchInd = PatchInd - 1;
                                BW(sub2ind(size(BW),PatchPoints{PatchInd}(:,1),PatchPoints{PatchInd}(:,2))) = false;
                                delete(h2);
                                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                                h2 = imshow(green);
                                set(h2, 'AlphaData', BW);
                            else
                                % Start redoing 'Delete'
                                DeleteInd = DeleteInd - 1;
                                BW(sub2ind(size(BW),DeletedPoints{DeleteInd}(:,1),DeletedPoints{DeleteInd}(:,2))) = true;
                                delete(h2);
                                green = cat(3, zeros(size(BW)), ones(size(BW))*0.8, zeros(size(BW)));
                                h2 = imshow(green);
                                set(h2, 'AlphaData', BW);
                            end
                        case 4
                            StopLoop = 1;
                    end

                    % Mark unconnected regions with blue lines and dots
                    delete(hPlots);
                    rp = regionprops(BW, 'PixelIdxList', 'PixelList');
                    for ii = 1:length(rp)
                        Pixels = rp(ii).PixelList;
                        PixelIDs = rp(ii).PixelIdxList;
                        PixelIDs(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                        Pixels(Pixels(:,1) == min(Pixels(:,1)),:) = [];
                        rp(ii).PixelIdxList = PixelIDs;
                        rp(ii).PixelList = Pixels;
                    end
                    BW2 = logical(zeros(size(BW)));
                    BW2(vertcat(rp.PixelIdxList)) = true;
                    MaxBW = max(BW2);
                    FirstInd = find(MaxBW==1,1,'first');
                    LastInd = find(MaxBW==1,1,'last');
                    rp2 = regionprops(imcomplement(MaxBW(FirstInd:LastInd)), 'PixelIdxList', 'Area');
                    hPlots = zeros(length(rp2),4);
                    for i = 1:length(rp2)
                        hPlots(i,1) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 1],'*-b','LineWidth',3,'MarkerSize',3);
                        hPlots(i,2) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[size(BW,1) size(BW,1)],'*-b','LineWidth',3,'MarkerSize',3);
                        hPlots(i,3) = plot([FirstInd+rp2(i).PixelIdxList(1)-1 FirstInd+rp2(i).PixelIdxList(1)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                        hPlots(i,4) = plot([FirstInd+rp2(i).PixelIdxList(end)-1 FirstInd+rp2(i).PixelIdxList(end)-1],[1 size(BW,1)],'*-b','LineWidth',1,'MarkerSize',1);
                    end
                end

                % Ask if we want to stop
                choice = questdlg('Continue patching/ deleting?','Continue?', ...
                        'Yes','No','Yes');
                switch choice
                    case 'Yes'
                    case 'No'
                        FinalPatchLoop = 1;
                end
            end

            % Prepare to save Processed Kymograph, then save
            zoom out;
            set(hFig,'units','pixels','outerposition',[0.05*screenWidth 0.01*screenHeight 0.95*screenWidth 0.95*screenHeight]);
            title(['Processed Kymograph (Fr. Overlap: ' num2str(FrameOverlap) '. Length Thres.: ' num2str(LengthThreshold) '. SubPixel Det.: ' num2str(UseSubPixelDetection) '. Resize Fact.: ' num2str(ResizeFactor) ')']);
            xlabel('Time (Frame)'); ylabel('D (pixel)');
            set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[],'yTick',[]);
            % Save Processed Kymograph and BW
            cd(DataPath);
            print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Processed Kymograph).png')]);
            savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Processed Kymograph).fig')]);
            save(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Binary Kymograph)')],'BW');

            % Compile all velocities
            rp = regionprops(BW, 'PixelList');
            Velocities = NaN(length(rp),size(BW,2)/ResizeFactor);
            for i = 1:length(rp)
                % Average all rows in the same column
                ColumnRow = sortrows(rp(i).PixelList,1);
                [Column,ia,ic] = unique(ColumnRow(:,1));
            %         h = histogram(ic,'BinMethod','integers');
            %         DuplicatedIndices = find(h.Values>1);
                h = histcounts(ic,'BinMethod','integers');
                DuplicatedIndices = find(h>1);
                Row = ColumnRow(ia,2);
                % Find mean for duplicated indices
                for ii = 1:length(DuplicatedIndices)
                    Row(DuplicatedIndices(ii)) = mean(ColumnRow(ic==DuplicatedIndices(ii),2));
                end
                % Get smoothed data
                SmoothedRow = smooth(Row,5*ResizeFactor);
                Velocity = (SmoothedRow(2:end) - SmoothedRow(1:end-1))*PixelSize./((Column(2:end)-Column(1:end-1))*ExposureTime);   % Velocity in nm/s
                NewColumn = ((Column-1)/ResizeFactor)+1; NewColumn = NewColumn(2:end);
                VelIndex = find(rem(NewColumn,1)==0,1);
                VelLastIndex = find(rem(NewColumn(1:end-ResizeFactor+1),1)==0,1,'last');
                NewColumn = NewColumn(VelIndex:ResizeFactor:VelLastIndex);
                NewVelocity = mean(reshape(Velocity(VelIndex:VelLastIndex+ResizeFactor-1),[],ResizeFactor),2);
                if VelIndex > 1
                    if isempty(VelLastIndex)
                        NewVelocity = [mean(Velocity(1:VelIndex-1)); NewVelocity];
                        NewColumn = ((Column-1)/ResizeFactor)+1; NewColumn = NewColumn(2:end);
                        NewColumn = floor(NewColumn(1));
                    else
                        NewVelocity = [mean(Velocity(1:VelIndex-1)); NewVelocity];
                        NewColumn = [NewColumn(1)-1; NewColumn];
                    end
                end
                if VelLastIndex+ResizeFactor-1 < length(Velocity)
                    NewVelocity = [NewVelocity; mean(Velocity(VelLastIndex+ResizeFactor:end))];
                    NewColumn = [NewColumn; NewColumn(end)+1];
                end
                Velocities(i,FrameAssignment(NewColumn)) = NewVelocity';
            end

            % Average over column to get mean velocities at every time point
            MeanVel = nanmean(Velocities,1);

            % Get displacement at every time point
            Displacement = cumsum(MeanVel*ExposureTime,'omitnan');

            % Plot Mean Velocity
            figure;
            plot((1:size(BW,2)/ResizeFactor)*ExposureTime,MeanVel,'LineWidth',2);
            ylabel('Velocity (nm/s)');
            xlim([0 (size(BW,2)/ResizeFactor)*ExposureTime]);
            xlabel('Time (s)');
            title('Microtubule Velocity over Time');
            set(gca,'FontSize',TextSize,'FontWeight','b','xTick',[]); 

            % Save velocity figure
            print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Velocity).png')]);
            savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Velocity).fig')]);

            % Draw Displacement diagram
            figure;
            plot((1:size(BW,2)/ResizeFactor)*ExposureTime,Displacement/1000,'LineWidth',2);
            ylabel('Disp. (um)');
            xlim([0 (size(BW,2)/ResizeFactor)*ExposureTime]);
            xlabel('Time (s)');
            title('Microtubule Disp. over Time');
            set(gca,'FontSize',TextSize,'FontWeight','b');

            % Save displacement figure
            print('-djpeg','-r300',['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Displacement).png')]);
            savefig(['Image Files\' strrep(InputName,TrackMateFileSuffix,'(Displacement).fig')]);

            MicTimeDispVel = [(1:(size(BW,2)/ResizeFactor))'*ExposureTime Displacement' MeanVel'];
            MicTimeDispVel(isnan(MeanVel),:) = [];

            % Import the three Final Result files
            % Final Result
            [~,~,raw] = xlsread([FileName '(Final Result).xlsx']);
            raw(2:end,1:3) = {NaN};
            raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
            xlswrite([FileName '(Final Result).xlsx'],raw);     % Write FinalResult to EXCEL spreadsheet
            % Final Result2
            [~,~,raw] = xlsread([FileName '(Final Result2).xlsx']);
            raw(2:end,1:3) = {NaN};
            raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
            xlswrite([FileName '(Final Result2).xlsx'],raw);    % Write FinalResult to EXCEL spreadsheet
            % Final Result3
            [~,~,raw] = xlsread([FileName '(Final Result3).xlsx']);
            raw(2:end,1:3) = {NaN};
            raw(2:size(MicTimeDispVel,1)+1,1:3) = num2cell(MicTimeDispVel);
            xlswrite([FileName '(Final Result3).xlsx'],raw);    % Write FinalResult to EXCEL spreadsheet

            % Clear StartEndMatrix for next processing
            clear StartEndMatrix;

            % Display progress
            disp('Done importing positions. Proceed to plotting microtubule velocity and kin-Qd graphs...');
    %% 2) Plot microtubule velocity and kin-Qd graphs
    % Define parameters
    UseSIC = 1;     % 1 to use SIC to fit traces. 0 otherwise.
    
    % Read EXCEL file
    TextSize = 12;
    
    ResultName = dir('*(Final Result).xlsx');
    ResultName = ResultName.name;
    FinalResult = xlsread(ResultName);
    
    RegionNo = floor((size(FinalResult,2)-3)/3);
    ExposureTime = FinalResult(2,1)-FinalResult(1,1);
    
    % Apply SIC-stepfinder
    DataPath = pwd;
    % Generate warning and ask for code path if we cannot find code in the active
    % folder and a few folder above
    if isempty(dir('SICstepFinder.m'))
        cd ..;      % Go to one folder up and look for SICstepFinder.m there   
        if isempty(dir('SICstepFinder.m'))
            cd ..;      % Go to one folder up and look for SICstepFinder.m there
            if isempty(dir('SICstepFinder.m'))
                cd ..       % Go to one folder up and look for SICstepFinder.m there
                if isempty(dir('SICstepFinder.m'))
                    % Generate warning and ask for the code path
                    h=warndlg('Cannot find SICstepFinder.m in active folder. Locate folder with SICstepFinder.m in the next window');
                    uiwait(h);
                    CodePath = uigetdir(DataPath,'Select folder containing SICstepFinder');
                    cd(CodePath);
                else; CodePath = pwd; 
                end;
            else; CodePath = pwd; 
            end;
        else; CodePath = pwd; 
        end;
    else; CodePath = pwd; 
    end;
    
    % Initialize ResidualStd (On-axis for first few rows, then off-axis)
    ResidualStd = zeros(RegionNo*2,1);
    
    % Add more zeros to allocate memory to FinalResult
    FinalResultLength = size(FinalResult,1);
    FinalResult = [FinalResult zeros(size(FinalResult,1),RegionNo*2)];
    % Loop to apply SIC-stepfinders to all points
    for i = 1:RegionNo
        Step1=FinalResult(:,5+5*(i-1));
        
        if UseSIC == 1
            NaNVector = Step1(isnan(Step1));
            Step = Step1(~isnan(Step1));
            StepFit = SICstepFinder(Step');
            StepFit = StepFit.StepFit';
            Step1 = [StepFit; NaNVector];
            
            % Find out the residual standard deviation for on-axis
            ResidualStd(i) = std(Step-StepFit);
        end

        Step2=FinalResult(:,6+5*(i-1));
        
        if UseSIC == 1
            NaNVector = Step2(isnan(Step2));
            Step = Step2(~isnan(Step2));
            StepFit = SICstepFinder(Step');
            StepFit = StepFit.StepFit';
            Step2 = [StepFit; NaNVector];

            % Find out the residual standard deviation for off-axis
            ResidualStd(RegionNo+i) = std(Step-StepFit);
        end

        FinalResult = [FinalResult(:,1:5+5*(i-1)) Step1 FinalResult(:,6+5*(i-1)) Step2 FinalResult(:,7+5*(i-1):end-2)];
    end

    % Change directory to DataPath
    cd(DataPath);
    
    % Check to see if '(Final Result2).xlsx' exists
    FinalResult2 = dir('*(Final Result2).xlsx');
    if ~isempty(FinalResult2)
        FinalResult2 = FinalResult2.name;
        FinalResult2 = xlsread(FinalResult2);
        FileRegionNo = round((size(FinalResult2,2)-4)/7);
        if FileRegionNo == RegionNo
            FinalResult2 = FinalResult2(2*RegionNo+3:4*RegionNo+2,end);
        else
            FinalResult2 = [];
        end
    end

    % Ask for Start and End frame
    TitleDialog = cell(1,RegionNo);
    StartingValues = cell(1,RegionNo);
    for i = 1:RegionNo
        TitleDialog(i) = {['Point ' num2str(i) ' Start and End Frame While Microtuble is Hovering Above:']};
        if ~isempty(FinalResult2)
            StartingValues(i) = {num2str(FinalResult(FinalResult2(2*i-1:2*i),4+5*(i-1))/ExposureTime)};
        else
            if exist('StartEndMatrix','var') > 0
                StartingValues(i) = StartEndFrames(i);
            else
                StartingValues(i) = {num2str([1;FinalResultLength])};
            end
        end
    end
    
    if RegionNo > 10
        % Determine how many times we need to loop
        StartEndFrames = cell(1,RegionNo);
        for i = 1:ceil(length(StartingValues)/10)
            if i == ceil(length(StartingValues)/10)
                StartEndFrames((i-1)*10+1:RegionNo) = inputdlg(TitleDialog((i-1)*10+1:RegionNo),'Parameters Input',2,StartingValues((i-1)*10+1:RegionNo));
            else
                StartEndFrames((i-1)*10+1:i*10) = inputdlg(TitleDialog((i-1)*10+1:i*10),'Parameters Input',2,StartingValues((i-1)*10+1:i*10));
            end
        end
    else
        StartEndFrames = inputdlg(TitleDialog,'Parameters Input',2,StartingValues);
    end

    StartEndMatrix = zeros(RegionNo*2,1);

    for i=1:RegionNo
        StartEndMatrix(2*i-1:2*i) = str2num(StartEndFrames{i});
        
        StartInd = find(round(FinalResult(:,4+5*(i-1))/ExposureTime)==StartEndMatrix(2*i-1));
        EndInd = find(round(FinalResult(:,4+5*(i-1))/ExposureTime)==StartEndMatrix(2*i));
        if isempty(StartInd)
            StartInd = abs(round(FinalResult(:,4+5*(i-1))/ExposureTime)-StartEndMatrix(2*i-1));
            [~,StartInd] = min(StartInd);
        end
        
        if isempty(EndInd)
            EndInd = abs(round(FinalResult(:,4+5*(i-1))/ExposureTime)-StartEndMatrix(2*i));
            [~,EndInd] = min(EndInd);
        end
        
        StartEndMatrix(2*i-1:2*i) = [StartInd;EndInd];
    end
    
    % Get xMax, maximum value in x
    xMax = max(FinalResult(:,1));
    for i = 1:RegionNo
        xMax = max(xMax,max(FinalResult(:,4+5*(i-1))));
    end
    xMax = xMax/ExposureTime;

    figure('units','normalized','outerposition',[0 0 1 1]);
    subplot(9,1,1);
    plot([0 xMax*ExposureTime],[0 0],'red','LineWidth',2); hold on;
    plot(FinalResult(:,1),FinalResult(:,3),'black','LineWidth',2);
    xlim([0 xMax*ExposureTime]);
    if min(FinalResult(:,3))<0
        ylim([min(FinalResult(:,3)) max(FinalResult(:,3))]);
    else
        ylim([0 max(FinalResult(:,3))]);
    end
    yLbl = ylabel('V (nm/s)');
    set(yLbl, 'Units', 'Normalized', 'Position', [-0.06, 0.5, 0]);
    title('Microtubule Velocity');
    set(gca,'FontSize',TextSize,'XTick',[],'FontWeight','b');
    
    % Add certain magnitude to On-Axis and Off-Axis to bring them closer together
    subplot(9,1,3:6);
    for i = 1:RegionNo
        if i == 1
            MaxOnAxisDiff = max(FinalResult(:,5+5*(i-1)))-min(FinalResult(:,5+5*(i-1)));
            MaxOffAxisDiff = max(FinalResult(:,7+5*(i-1)))-min(FinalResult(:,7+5*(i-1)));
        else
            MaxOnAxisDiff = max(MaxOnAxisDiff,max(FinalResult(:,5+5*(i-1)))-min(FinalResult(:,5+5*(i-1))));
            MaxOffAxisDiff = max(MaxOffAxisDiff,max(FinalResult(:,7+5*(i-1)))-min(FinalResult(:,7+5*(i-1))));
        end
    end
    
%     for i = 1:RegionNo
%         FinalResult(:,6+5*(i-1)) = FinalResult(:,6+5*(i-1))-min(FinalResult(:,5+5*(i-1)))+((i-1)*MaxOnAxisDiff)+MaxOnAxisDiff*0.05;
%         FinalResult(:,5+5*(i-1)) = FinalResult(:,5+5*(i-1))-min(FinalResult(:,5+5*(i-1)))+((i-1)*MaxOnAxisDiff)+MaxOnAxisDiff*0.05;
%         FinalResult(:,8+5*(i-1)) = FinalResult(:,8+5*(i-1))-min(FinalResult(:,7+5*(i-1)))+((i-1)*MaxOffAxisDiff)+MaxOffAxisDiff*0.05;
%         FinalResult(:,7+5*(i-1)) = FinalResult(:,7+5*(i-1))-min(FinalResult(:,7+5*(i-1)))+((i-1)*MaxOffAxisDiff)+MaxOffAxisDiff*0.05;
%     end
    
    for i = 1:RegionNo
        FinalResult(:,6+5*(i-1)) = FinalResult(:,6+5*(i-1)) - (max(FinalResult(:,5+5*(i-1)))+min(FinalResult(:,5+5*(i-1))))/2 + ((i-1)*MaxOnAxisDiff) + MaxOnAxisDiff*0.55;
        FinalResult(:,5+5*(i-1)) = FinalResult(:,5+5*(i-1)) - (max(FinalResult(:,5+5*(i-1)))+min(FinalResult(:,5+5*(i-1))))/2 + ((i-1)*MaxOnAxisDiff) + MaxOnAxisDiff*0.55;
        FinalResult(:,8+5*(i-1)) = FinalResult(:,8+5*(i-1)) - (max(FinalResult(:,7+5*(i-1)))+min(FinalResult(:,7+5*(i-1))))/2 + ((i-1)*MaxOffAxisDiff) + MaxOffAxisDiff*0.55;
        FinalResult(:,7+5*(i-1)) = FinalResult(:,7+5*(i-1)) - (max(FinalResult(:,7+5*(i-1)))+min(FinalResult(:,7+5*(i-1))))/2 + ((i-1)*MaxOffAxisDiff) + MaxOffAxisDiff*0.55;
    end
    
    % plot On-Axis Displacement
    cc=hsv(RegionNo)';
    subplot(9,1,3:6);
    %LegendName = cell(RegionNo,1);
    for i = 1:RegionNo
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,5+5*(i-1)),'Color',cc(:,i),'LineWidth',2,'LineStyle','-'); hold on;
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,6+5*(i-1)),'black','LineWidth',1,'LineStyle','-.'); hold on;
        %LegendName(i) = {['Point ' num2str(i)]};
        
        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),FinalResult(StartEndMatrix(2*i-1),5+5*(i-1)),'pk','LineWidth',2,'MarkerFaceColor','black'); hold on;
        plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),FinalResult(StartEndMatrix(2*i),5+5*(i-1)),'ok','LineWidth',2,'MarkerFaceColor','black'); hold on;
    end
    
    % Add label for points
    yHeight = ylim; yHeight = yHeight(2)-yHeight(1);
    for i = 1:RegionNo
        LastIndex = find(~isnan(FinalResult(:,4+5*(i-1))),1,'last');
        text(FinalResult(LastIndex,4+5*(i-1))*1.02,FinalResult(LastIndex,5+5*(i-1))+(yHeight/25),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b'); hold on;
%         if i == 1
%             MaxOnAxisDiff = max(FinalResult(:,3*i+1))-min(FinalResult(:,3*i+1));
%             MaxOffAxisDiff = max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3));
%         else
%             MaxOnAxisDiff = max(MaxOnAxisDiff,max(FinalResult(:,3*i+1))-min(FinalResult(:,3*i+1)));
%             MaxOffAxisDiff = max(MaxOffAxisDiff,max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3)));
%         end
    end
    
    xlim([0 xMax*ExposureTime]);
    ylim([0 MaxOnAxisDiff*RegionNo*1.05]);
    yLbl = ylabel('On-Axis Displacement (nm)');
    set(yLbl, 'Units', 'Normalized', 'Position', [-0.06, 0.5, 0]);
    title('Kinesin On-Axis Displacement');
    set(gca,'XTick',[],'FontSize',TextSize,'FontWeight','b');
    
%     % Add certain magnitude to Off-Axis so that the order on plot is the same as On-Axis 
%     if FinalResult(1,3*RegionNo+2)<FinalResult(1,5)
%         for i = 1:RegionNo
%             FinalResult(:,3*i+3) = FinalResult(:,3*i+3)-min(FinalResult(:,3*i+3))+((3-i)*MaxOffAxisDiff)+20;
%         end
%     else
%         for i = 1:RegionNo
%             FinalResult(:,3*i+3) = FinalResult(:,3*i+3)-min(FinalResult(:,3*i+3))+((i-1)*MaxOffAxisDiff)+20;
%         end
%     end
    
%     % Adjust magnitude of the Off-Axis so that it is well-spread apart
%     subplot(9,1,8:9);
%     for i = 1:RegionNo
%         FinalResult(:,3*i+3) = FinalResult(:,3*i+3)-min(FinalResult(:,3*i+3))+((i-1)*MaxOffAxisDiff)+20;
%     end
    
    subplot(9,1,8:9);
    % Plot Off-Axis Displacement
    for i = 1:RegionNo
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,7+5*(i-1)),'Color',cc(:,i),'LineWidth',2,'LineStyle','-'); hold on;
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,8+5*(i-1)),'black','LineWidth',1,'LineStyle','-.'); hold on;
        %LegendName(i) = {['Point ' num2str(i)]};
        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),FinalResult(StartEndMatrix(2*i-1),7+5*(i-1)),'pk','LineWidth',2,'MarkerFaceColor','black'); hold on;
        plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),FinalResult(StartEndMatrix(2*i),7+5*(i-1)),'ok','LineWidth',2,'MarkerFaceColor','black'); hold on;
    end
    
    % Add label for points
    yHeight = MaxOffAxisDiff*3;
    for i = 1:RegionNo
        LastIndex = find(~isnan(FinalResult(:,4+5*(i-1))),1,'last');
        text(FinalResult(LastIndex,4+5*(i-1))*1.02,FinalResult(LastIndex,7+5*(i-1))+(yHeight/25),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b'); hold on;
    end
    
    xlim([0 xMax*ExposureTime]);
    ylim([0 MaxOffAxisDiff*RegionNo*1.05]);
    yLbl = ylabel('Off-Axis Displacement (nm)');
    set(yLbl, 'Units', 'Normalized', 'Position', [-0.06, 0.5, 0]);
    xLbl = xlabel('Time (s)');
    set(xLbl, 'Units', 'Normalized', 'Position', [0.5, -0.1, 0]);
    title('Kinesin Off-Axis Displacement');
    set(gca,'FontSize',TextSize,'FontWeight','b');
    
    % Initialize DistanceForces matrix to record driving and hindering distance forces
    DistanceForces = zeros(FinalResultLength,RegionNo*2);
    
    % Add equilibrium lines and save figure and equilibrium lines
    EquilibriumPos = zeros(RegionNo*2,1);
    subplot(9,1,3:6);
    yHeight = ylim; yHeight = yHeight(2)-yHeight(1);
    
    % Loop for on-axis equilibrium lines
    for i = 1:RegionNo
        RectanglePos = [0 min(FinalResult(:,5+5*(i-1)))-yHeight/25 xMax*ExposureTime max(FinalResult(:,5+5*(i-1)))-min(FinalResult(:,5+5*(i-1)))+yHeight/12.5];
        r = rectangle('Position',RectanglePos,'EdgeColor','magenta','LineWidth',2);
        ylim([min(FinalResult(:,5+5*(i-1)))-yHeight/25 max(FinalResult(:,5+5*(i-1)))+yHeight/25]);
        disp(['Please click two points to trace the equilibrium position for Point ' num2str(i) '...']);
        disp('Press <Enter> when done.');
        [xEqui,yEqui] = getline(gca);
        delete(r);
        EquilibriumPos(i) = mean(yEqui);
        plot([0 xMax],[EquilibriumPos(i) EquilibriumPos(i)],':black','LineWidth',2);
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,5+5*(i-1)),'Color',cc(:,i),'LineWidth',2);
        
        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),FinalResult(StartEndMatrix(2*i-1),5+5*(i-1)),'pk','LineWidth',2,'MarkerFaceColor','black'); hold on;
        plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),FinalResult(StartEndMatrix(2*i),5+5*(i-1)),'ok','LineWidth',2,'MarkerFaceColor','black'); hold on;
        
        % Find out on-axis DistanceForces
        % Get DistanceForce1 and DistanceForce2 from real trace and step-fit
        DistanceForce1 = FinalResult(:,5+5*(i-1)) - EquilibriumPos(i);
        DistanceForce2 = FinalResult(:,6+5*(i-1)) - EquilibriumPos(i);
        % Make those near equilibrium lines to be zero
%         DistanceForce1(DistanceForce1<1.5*ResidualStd(i) & DistanceForce1>-1.5*ResidualStd(i)) = 0;
%         DistanceForce1(DistanceForce2<1.5*ResidualStd(i) & DistanceForce2>-1.5*ResidualStd(i)) = 0;
        % Make those outside the StartEndFrames to be zero as well
%         DistanceForce1(1:StartEndMatrix(2*i-1)) = 0;
%         DistanceForce1(StartEndMatrix(2*i):find(isnan(DistanceForce1),1)-1) = 0;
        % Store in DistanceForces
        DistanceForces(:,i) = DistanceForce1;
        
        % Plot the points in equilibrium as black lines
        plot(FinalResult(DistanceForce1==0,4+5*(i-1)),FinalResult(DistanceForce1==0,5+5*(i-1)),'ok','LineWidth',2,'MarkerSize',2);
    end
    ylim([0 MaxOnAxisDiff*RegionNo*1.05]);
    
    % Loop for off-axis equilibrium lines
    subplot(9,1,8:9);
    for i = 1:RegionNo
        RectanglePos = [0 min(FinalResult(:,7+5*(i-1)))-yHeight/25 xMax*ExposureTime max(FinalResult(:,7+5*(i-1)))-min(FinalResult(:,7+5*(i-1)))+yHeight/12.5];
        r = rectangle('Position',RectanglePos,'EdgeColor','magenta','LineWidth',2);
        ylim([min(FinalResult(:,7+5*(i-1)))-yHeight/25 max(FinalResult(:,7+5*(i-1)))+yHeight/25]);
        disp(['Please click two points to trace the equilibrium position for Point ' num2str(i) '...']);
        disp('Press <Enter> when done.');
        [xEqui,yEqui] = getline(gca);
        delete(r);
        EquilibriumPos(RegionNo+i) = mean(yEqui);
        plot([0 xMax],[EquilibriumPos(RegionNo+i) EquilibriumPos(RegionNo+i)],':black','LineWidth',2);
        %plot(FinalResult(:,3*i+1),FinalResult(:,3*i+2),'Color',cc(:,i),'LineWidth',2);
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,7+5*(i-1)),'Color',cc(:,i),'LineWidth',2,'LineStyle','-');
    
        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),FinalResult(StartEndMatrix(2*i-1),7+5*(i-1)),'pk','LineWidth',2,'MarkerFaceColor','black'); hold on;
        plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),FinalResult(StartEndMatrix(2*i),7+5*(i-1)),'ok','LineWidth',2,'MarkerFaceColor','black'); hold on;
    
        % Find out off-axis DistanceForces
        % Get DistanceForce1 and DistanceForce2 from real trace and step-fit
        DistanceForce1 = FinalResult(:,7+5*(i-1)) - EquilibriumPos(RegionNo+i);
        
        % For off-axis distance, just record all distances so we can
        % calculate full distance later. If we want to account for off-axis
        % equilibrium distances also, the commented codes below will help
%         DistanceForce2 = FinalResult(:,8+5*(i-1)) - EquilibriumPos(RegionNo+i);
%         % Make those near equilibrium lines to be zero
%         DistanceForce1(DistanceForce1<1.5*ResidualStd(RegionNo+i) & DistanceForce1>-1.5*ResidualStd(RegionNo+i)) = 0;
%         DistanceForce1(DistanceForce2<1.5*ResidualStd(RegionNo+i) & DistanceForce2>-1.5*ResidualStd(RegionNo+i)) = 0;
        
        % Store in DistanceForces
        DistanceForces(:,RegionNo+i) = DistanceForce1;
        
%         % Plot the points in equilibrium as black lines
%         plot(FinalResult(DistanceForce1==0,4+5*(i-1)),FinalResult(DistanceForce1==0,7+5*(i-1)),'ok','LineWidth',2,'MarkerSize',2);
    end
    ylim([0 MaxOffAxisDiff*RegionNo*1.05]);
    
    % Save figure
    if exist('Image Files','dir')~=7; mkdir('Image Files'); end
    print('-djpeg','-r300',['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs).png')]);
    
    
    
    
    % Plot just microtubule velocity and off-axis with shading
    figure('units','normalized','outerposition',[0 0 1 1]);
    subplot(6,1,1:2); hold on;
    
    xTrace = FinalResult(:,1);
    xTrace = xTrace(~isnan(xTrace));
    xFill = min(xTrace):ExposureTime:max(xTrace);
    Trace = FinalResult(:,3);
    Trace = Trace(~isnan(Trace));
    yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
    MicVel = yTrace;
    EquiLine = zeros(1,length(yTrace));
    xFill = [xFill, fliplr(xFill)];
    yTrace = [EquiLine, fliplr(yTrace)];
    fill(xFill,yTrace,[EquiLine, fliplr(MicVel)]);
    
    plot([0 xMax*ExposureTime],[0 0],':black','LineWidth',2);
    plot(FinalResult(:,1),FinalResult(:,3),'black','LineWidth',4);
    xlim([0 xMax*ExposureTime]);
    if min(FinalResult(:,3))<0
        ylim([min(FinalResult(:,3)) max(FinalResult(:,3))]);
    else
        ylim([-max(FinalResult(:,3))*0.2 max(FinalResult(:,3))*1.2]);
    end
    yLbl = ylabel('Mic. Vel. (nm/s)');
    set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
    set(gca,'FontSize',TextSize+8,'XTick',[],'FontWeight','b','LineWidth',3);
    text(0.13,1.15,'Microtubule Velocity and Kinesin Off-Axis Displacement','Units','normalized','FontSize',TextSize+8,'FontWeight','bold');

    colorbar;
    caxis([0 800]);
    
    % Loop for off-axis equilibrium lines
    subplot(6,1,3:6); hold on;
    for i = 1:RegionNo
        % Draw driving and hindering areas
        xTrace = FinalResult(:,4+5*(i-1));
        xTrace = xTrace(~isnan(xTrace));
        xFill = min(xTrace):ExposureTime:max(xTrace);
        Trace = FinalResult(:,7+5*(i-1));
        Trace = Trace(~isnan(Trace));
        yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
        DistForces = yTrace - EquilibriumPos(RegionNo+i);
        yFill = yTrace;
        EquiLine = ones(1,length(yFill))*EquilibriumPos(RegionNo+i);
        ZeroLine = zeros(1,length(yFill));
        xFill = [xFill, fliplr(xFill)];
        yFill = [EquiLine, fliplr(yFill)];
        fill(xFill,yFill,[ZeroLine, fliplr(DistForces)]);
        
        plot([0 xMax],[EquilibriumPos(RegionNo+i) EquilibriumPos(RegionNo+i)],':black','LineWidth',2);
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,7+5*(i-1)),'k','LineWidth',4,'LineStyle','-');
    
        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),FinalResult(StartEndMatrix(2*i-1),7+5*(i-1)),'pb','LineWidth',10,'MarkerFaceColor','blue','MarkerSize',10); hold on;
        plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),FinalResult(StartEndMatrix(2*i),7+5*(i-1)),'ob','LineWidth',10,'MarkerFaceColor','blue','MarkerSize',10); hold on;
    end

    xlim([0 xMax*ExposureTime]);
    ylim([0 MaxOffAxisDiff*RegionNo*1.05]);
    yLbl = ylabel('Kinesin Off-Axis Displacement (nm)');
    set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
    xLbl = xlabel('Time (s)');
    set(xLbl, 'Units', 'Normalized', 'Position', [0.5, -0.1, 0]);
    set(gca,'FontSize',TextSize+8,'FontWeight','b','LineWidth',3);
    
    colorbar;
    caxis([-500 500]);
    colormap(hot);
    
    % Add label for points
    yHeight = MaxOffAxisDiff*3;
    for i = 1:RegionNo
        Trace = FinalResult(:,7+5*(i-1));
        Trace = Trace(~isnan(Trace));
        DistForces = Trace - EquilibriumPos(RegionNo+i);
        LabelSign = sign(max(DistForces)-abs(min(DistForces)));
        EndIndex = str2num(StartEndFrames{i});
        text(EndIndex(2)*ExposureTime,EquilibriumPos(RegionNo+i)+LabelSign*(yHeight/12),num2str(i),'Color','black','FontSize',18,'fontweight','b'); hold on;
    end

    % Save figure for Off-axis shading
    if exist('Image Files','dir')~=7; mkdir('Image Files'); end
    print('-djpeg','-r300',['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-Off-Axis).png')]);
    savefig(['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-Off-Axis).fig')]);
    
    
    
    
    % Plot just microtubule velocity and on-axis with shading
    hOnAxis = figure('units','normalized','outerposition',[0 0 1 1]);
    subplot(6,1,1:2); hold on;
    
    xTrace = FinalResult(:,1);
    xTrace = xTrace(~isnan(xTrace));
    xFill = min(xTrace):ExposureTime:max(xTrace);
    Trace = FinalResult(:,3);
    Trace = Trace(~isnan(Trace));
    yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
    MicVel = yTrace;
    EquiLine = zeros(1,length(yTrace));
    xFill = [xFill, fliplr(xFill)];
    yTrace = [EquiLine, fliplr(yTrace)];
    fill(xFill,yTrace,[EquiLine, fliplr(MicVel)]);
    
    plot([0 xMax*ExposureTime],[0 0],':black','LineWidth',2);
    plot(FinalResult(:,1),FinalResult(:,3),'black','LineWidth',4);
    xlim([0 xMax*ExposureTime]);
    if min(FinalResult(:,3))<0
        ylim([min(FinalResult(:,3)) max(FinalResult(:,3))]);
    else
        ylim([-max(FinalResult(:,3))*0.2 max(FinalResult(:,3))*1.2]);
    end
    yLbl = ylabel('Mic. Vel. (nm/s)');
    set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
    set(gca,'FontSize',TextSize+8,'XTick',[],'FontWeight','b','LineWidth',3);
    text(0.13,1.15,'Microtubule Velocity and Kinesin On-Axis Displacement','Units','normalized','FontSize',TextSize+8,'FontWeight','bold');

    colorbar;
    caxis([0 800]);
    
    % Loop for on-axis equilibrium lines
    subplot(6,1,3:6); hold on;
    for i = 1:RegionNo
        % Draw hindering areas
        xTrace = FinalResult(:,4+5*(i-1));
        xTrace = xTrace(~isnan(xTrace));
        xFill = min(xTrace):ExposureTime:max(xTrace);
        Trace = FinalResult(:,5+5*(i-1));
        Trace = Trace(~isnan(Trace));
        yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
        DistForces = yTrace - EquilibriumPos(i);
        yFill = yTrace;
        EquiLine = ones(1,length(yFill))*EquilibriumPos(i);
        ZeroLine = zeros(1,length(yFill));
        xFill = [xFill, fliplr(xFill)];
        yFill = [EquiLine, fliplr(yFill)];
        %fill(xFill,yFill,[EquiLine, fliplr(DistForces)]);
        fill(xFill,yFill,[ZeroLine, fliplr(DistForces)]);
        %fill(xFill,yFill,[EquiLine, EquiLine]);
        
        plot([0 xMax],[EquilibriumPos(i) EquilibriumPos(i)],':black','LineWidth',2);
        plot(FinalResult(:,4+5*(i-1)),FinalResult(:,5+5*(i-1)),'k','LineWidth',4);

        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),FinalResult(StartEndMatrix(2*i-1),5+5*(i-1)),'pb','LineWidth',10,'MarkerFaceColor','blue','MarkerSize',10); hold on;
        plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),FinalResult(StartEndMatrix(2*i),5+5*(i-1)),'ob','LineWidth',10,'MarkerFaceColor','blue','MarkerSize',10); hold on;
    end
    
    xlim([0 xMax*ExposureTime]);
    ylim([0 MaxOnAxisDiff*RegionNo*1.05]);
    yLbl = ylabel('Kinesin On-Axis Displacement (nm)');
    set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
    xLbl = xlabel('Time (s)');
    set(xLbl, 'Units', 'Normalized', 'Position', [0.5, -0.1, 0]);
    set(gca,'FontSize',TextSize+8,'FontWeight','b','LineWidth',3);
    
    c = colorbar;
    caxis([-500 500]);
    colormap(hot);
    %c.Location = 'westoutside';
    
    % Add label for points
    yHeight = ylim; yHeight = yHeight(2)-yHeight(1);
    for i = 1:RegionNo
%         LastIndex = find(~isnan(FinalResult(:,4+5*(i-1))),1,'last');
%         text(FinalResult(LastIndex,4+5*(i-1))*1.02,FinalResult(LastIndex,7+5*(i-1))+(yHeight/25),num2str(i),'Color','black','FontSize',18,'fontweight','b'); hold on;
        Trace = FinalResult(:,5+5*(i-1));
        Trace = Trace(~isnan(Trace));
        DistForces = Trace - EquilibriumPos(i);
        LabelSign = sign(max(DistForces)-abs(min(DistForces)));
        EndIndex = str2num(StartEndFrames{i});
%         text(EndIndex(2)*ExposureTime,EquilibriumPos(i)+LabelSign*(yHeight/12),num2str(i),'Color','black','FontSize',18,'fontweight','b'); hold on;
        text(xMax*ExposureTime*0.981,EquilibriumPos(i),num2str(i),'Color','black','FontSize',18,'fontweight','b','BackgroundColor','w'); hold on;
    end
    
    % Save figure for On-axis shading
    if exist('Image Files','dir')~=7; mkdir('Image Files'); end
    print('-djpeg','-r300',['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-On-Axis).png')]);
    savefig(['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-On-Axis).fig')]);
    
    
    
    
    % Allocate more memories to FinalResult to include DistanceForces
    FinalResult2 = [FinalResult zeros(FinalResultLength,RegionNo*2)];
    
    % Write to EXCEL spreadsheet
    FirstLine = cell(1,RegionNo*7+4);
    FirstLine(1,1:3) = {'Microtubule Time (s)','Microtubule Displacement (nm)','Microtubule Velocity (nm/s)'};
    for i = 1:RegionNo
        % Define titles of each column
        FirstLine(1,4+7*(i-1):10+7*(i-1)) = {['Point ' num2str(i) ' Time (s)'],['Point ' num2str(i) ' On-Axis Displacement (nm)'],['Point ' num2str(i) ' On-Axis Step-Fit (nm)'],['Point ' num2str(i) ' On-Axis DistanceForce (nm)'],['Point ' num2str(i) ' Off-Axis Displacement (nm)'],['Point ' num2str(i) ' Off-Axis Step-Fit (nm)'],['Point ' num2str(i) ' Off-Axis DistanceForce (nm)']};
        
        % Insert DistanceForces into FinalResult2
        FinalResult2 = [FinalResult2(:,1:6+7*(i-1)) DistanceForces(:,i) FinalResult2(:,7+7*(i-1):8+7*(i-1)) DistanceForces(:,RegionNo+i) FinalResult2(:,9+7*(i-1):end-2)];
    end
    FirstLine(1,end) = {'On-Axis Equilibrium Point'};
    
    FinalResult2 = num2cell(FinalResult2);
    FinalResult2{1,RegionNo*7+4}=[];
    FinalResult2 = [FirstLine; FinalResult2];
    FinalResult2(2:RegionNo+1,end) = num2cell(EquilibriumPos(1:RegionNo));
    FinalResult2(RegionNo+2,end) = {'Off-Axis Equilibrium Point'};
    FinalResult2(RegionNo+3:2*RegionNo+2,end) = num2cell(EquilibriumPos(RegionNo+1:RegionNo*2));
    FinalResult2(2*RegionNo+3,end) = {'Start and End Frames'};
    FinalResult2(2*RegionNo+4:4*RegionNo+3,end) = num2cell(StartEndMatrix);
    
    % Write FinalResult2 to EXCEL file
    xlswrite(strrep(ResultName,'(Final Result).xlsx','(Final Result2).xlsx'),FinalResult2);
    
    
    
    
    
    
    % Import FinalResult2
    FileName = dir('*(Final Result2).xlsx');
    FileName = FileName.name;
    FinalResult2 = xlsread(FileName);
    FileInitial = strrep(FileName,' (Final Result2).xlsx','');
    
    % Get DistanceForces
    DistanceForces = zeros(size(FinalResult2,1),RegionNo*2);
    for i = 1:RegionNo
        DistanceForces(:,2*i-1:2*i) = [FinalResult2(:,7+7*(i-1)) FinalResult2(:,10+7*(i-1))];
    end

    % Find out total number of points to initialize DurDistForces
    TotalPoints = 0;
    for i = 1:RegionNo
        % Draw hindering areas
        Time = FinalResult(:,4+5*(i-1));
        Time = Time(~isnan(Time));
        TotalPoints = TotalPoints + length(Time);
    end

    % Initialize DurDistForces matrix
    DurDistForces = zeros(TotalPoints,6);
    DMMIndex = 1;
    hText = zeros(TotalPoints,6);
    hPlot = zeros(TotalPoints,6);
    TraceLengths = zeros(TotalPoints,1);

    set(hOnAxis,'units','normalized','outerposition',[0.15 0 0.85 1]);

    % Start loop to choose driving/ hindering regions
    StopLoop = 0;
    LineIndex = 1;
    while StopLoop == 0
        % Get polyline
        pan on;
        disp('Please draw a line on an equilibrium line for the duration of driving/ hindering. Double click on second point.');
        h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK'), 'Zoom/ Pan');
        set(h, 'Position', [10 250 250 60]);
        uiwait(h); 
        choice = questdlg('Continue choosing regions?.','Continue?', ...
                'Yes','No','Redo','Yes');

        switch choice
            case 'Yes'
                % Get polyline
                disp(['Please draw line ' num2str(LineIndex) '. Double click when done to close polygon.']);
                [xBorder,yBorder] = getline(gca);

                % Write LineIndex
                hText(LineIndex) = text(mean(xBorder),mean(yBorder),num2str(LineIndex),'Color','y','FontSize',18,'fontweight','b','HorizontalAlignment','center','BackgroundColor','k');

                % Initialize ClosestDistance
                ClosestDistance = zeros(RegionNo,1);

                % Find closest points
                for i = 1:RegionNo
                    % Draw hindering areas
                    Time = FinalResult(:,4+5*(i-1));
                    Time = Time(~isnan(Time));
                    Trace = FinalResult(:,5+5*(i-1));
                    Trace = Trace(~isnan(Trace));
                    [~,D] = knnsearch(Trace,yBorder);
                    ClosestDistance(i,:) = mean(D);
                end

                % Find closest distance
                [~,RegionIndex] = min(ClosestDistance);

                % Draw lines between points
                Time = FinalResult(:,4+5*(RegionIndex-1));
                Time = Time(~isnan(Time));
                Trace = FinalResult(:,5+5*(RegionIndex-1));
                Trace = Trace(~isnan(Trace));
                CloseIndex = knnsearch(Time,xBorder);
                hPlot(LineIndex) = plot(Time(CloseIndex(1):CloseIndex(2)),Trace(CloseIndex(1):CloseIndex(2)),'y','LineWidth',1);
                TraceLengths(LineIndex) = length(Trace(CloseIndex(1):CloseIndex(2)));
                TraceLength = TraceLengths(LineIndex);

                DurDistForces(DMMIndex:DMMIndex+TraceLength-1,1:5) = [ones(TraceLength,1)*RegionIndex ones(TraceLength,1)*LineIndex Time(CloseIndex(1):CloseIndex(2)) DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex-1:2*RegionIndex)];
                DurDistForces(DMMIndex:DMMIndex+TraceLength-1,6) = sign(DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex-1)).*sqrt(DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex-1).^2 + DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex).^2);
                DMMIndex = DMMIndex + TraceLength;

                LineIndex = LineIndex + 1;
            case 'No'
                StopLoop = 1;
            case 'Redo'
                % Undo record on DMMIndex, DurDistForces and LineIndex
                LineIndex = LineIndex - 1;
                TraceLength = TraceLengths(LineIndex);
                DMMIndex = DMMIndex - TraceLength;
                DurDistForces(DMMIndex:DMMIndex+TraceLength-1,:) = 0;
                
                % Delete previous text and trace plot
                delete(hText(LineIndex));
                delete(hPlot(LineIndex));
        end
    end
    set(hOnAxis,'units','normalized','outerposition',[0 0 1 1]);

    % Zoom out, save image with spot regions, then close
    zoom out;
    
    % Delete zeros in DurDistForces
    DurDistForces(DurDistForces(:,1)==0,:) = [];

    % Get unique values in DurDistForces
    DurDistForcesUnique = unique(DurDistForces(:,[1 3:6]),'rows');

    % Set DistanceForces without hinder/drive to be zero
    for i = 1:RegionNo
        % Find times for the specific region
        DurTime = DurDistForcesUnique(DurDistForcesUnique(:,1)==i,2);

        % Find non NaN values in time for specific region
        RegionTime = FinalResult2(~isnan(FinalResult2(:,4+7*(i-1))),4+7*(i-1));

        % Find indices that has no hinder/ drive
        NoHinderDriveIndex = ~ismember(RegionTime,DurTime);

        % Set on and off-axis DistanceForces without Drive/Hinder to be 0
        FinalResult2(NoHinderDriveIndex,7+7*(i-1)) = 0;
        FinalResult2(NoHinderDriveIndex,10+7*(i-1)) = 0;
    end

    % Write to EXCEL spreadsheet
    FirstLine = cell(1,RegionNo*7+4);
    FirstLine(1,1:3) = {'Microtubule Time (s)','Microtubule Displacement (nm)','Microtubule Velocity (nm/s)'};
    for i = 1:RegionNo
        % Define titles of each column
        FirstLine(1,4+7*(i-1):10+7*(i-1)) = {['Point ' num2str(i) ' Time (s)'],['Point ' num2str(i) ' On-Axis Displacement (nm)'],['Point ' num2str(i) ' On-Axis Step-Fit (nm)'],['Point ' num2str(i) ' On-Axis DistanceForce (nm)'],['Point ' num2str(i) ' Off-Axis Displacement (nm)'],['Point ' num2str(i) ' Off-Axis Step-Fit (nm)'],['Point ' num2str(i) ' Off-Axis DistanceForce (nm)']};
    end
    FirstLine(1,end) = {'On-Axis Equilibrium Point'};

    FinalResult2 = num2cell(FinalResult2);
    FinalResult2 = [FirstLine; FinalResult2];
    FinalResult2(RegionNo+2,end) = {'Off-Axis Equilibrium Point'};
    FinalResult2(2*RegionNo+3,end) = {'Start and End Frames'};

    % Write FinalResult2 to EXCEL file
    delete([FileInitial ' (Final Result3).xlsx']);
    xlswrite([FileInitial ' (Final Result3).xlsx'],FinalResult2);

    % Write DurDistForces to EXCEL file
    delete([FileInitial ' (Dur-DistForces).xlsx']);
    xlswrite([FileInitial ' (Dur-DistForces).xlsx'],DurDistForces);

    % Save figure for Dur-DistForces
    if exist('Image Files','dir')~=7; mkdir('Image Files'); end
    print('-djpeg','-r300',['Image Files\' strrep(FileName,'(Final Result2).xlsx','(Dur-DistForces).png')]);
    savefig(['Image Files\' strrep(FileName,'(Final Result2).xlsx','(Dur-DistForces).fig')]);
    
    % Display progress
    disp('Done plotting microtubule velocity and kin-Qd graphs. Proceed to get microtubule lengths...');
        %% Manually choose drives and hinders from on-axis figure and save as FinalResult3
        % Import FinalResult to get MaxOnAxisDiff
        FileName = dir('*(Final Result).xlsx');
        FileName = FileName.name;
        FinalResult = xlsread(FileName);
        RegionNo = round((size(FinalResult,2)-3)/3);

        % Get MaxOnAxisDiff
        for i = 1:RegionNo
            if i == 1
                MaxOnAxisDiff = max(FinalResult(:,5+3*(i-1)))-min(FinalResult(:,5+3*(i-1)));
                MaxOffAxisDiff = max(FinalResult(:,6+3*(i-1)))-min(FinalResult(:,6+3*(i-1)));
            else
                MaxOnAxisDiff = max(MaxOnAxisDiff,max(FinalResult(:,5+3*(i-1)))-min(FinalResult(:,5+3*(i-1))));
                MaxOffAxisDiff = max(MaxOffAxisDiff,max(FinalResult(:,6+3*(i-1)))-min(FinalResult(:,6+3*(i-1))));
            end
        end

        % Import FinalResult2 and assign to FinalResult
        FileName = dir('*(Final Result2).xlsx');
        FileName = FileName.name;
        FinalResult = xlsread(FileName);
        RegionNo = round((size(FinalResult,2)-4)/7);
        TextSize = 12;

        % Get PixelSize and ExposureTime
        FileInitial = strrep(FileName,' (Final Result2).xlsx','');
        ExposureTime = xlsread([FileInitial ' (Variables).xlsx']);
        ExposureTime = ExposureTime(2);    % Exposure time in sec
        %PixelSize = PixelSize(1);       % PixelSize in nm

        % Get EquilibriumPos and StartEndMatrix
        EquilibriumPos = FinalResult(1:RegionNo,end);
        StartEndMatrix = FinalResult(2*RegionNo+3:4*RegionNo+2,end);

        % Get DistanceForces
        DistanceForces = zeros(size(FinalResult,1),RegionNo*2);
        for i = 1:RegionNo
            DistanceForces(:,2*i-1:2*i) = [FinalResult(:,7+7*(i-1)) FinalResult(:,10+7*(i-1))];
        end

        % Delete DistanceForces column
        for i = RegionNo:-1:1
            % Delete DistanceForces from FinalResult2
            FinalResult(:,10+7*(i-1)) = [];
            FinalResult(:,7+7*(i-1)) = [];
        end

        % Get xMax, maximum value in x
        xMax = max(FinalResult(:,1));
        for i = 1:RegionNo
            xMax = max(xMax,max(FinalResult(:,4+5*(i-1))));
        end
        xMax = xMax/ExposureTime;

        % Plot just microtubule velocity and on-axis with shading
        hOnAxis = figure('units','normalized','outerposition',[0 0 1 1]);
        subplot(6,1,1:2); hold on;

        xTrace = FinalResult(:,1);
        xTrace = xTrace(~isnan(xTrace));
        xFill = min(xTrace):ExposureTime:max(xTrace);
        Trace = FinalResult(:,3);
        Trace = Trace(~isnan(Trace));
        yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
        MicVel = yTrace;
        EquiLine = zeros(1,length(yTrace));
        xFill = [xFill, fliplr(xFill)];
        yTrace = [EquiLine, fliplr(yTrace)];
        fill(xFill,yTrace,[EquiLine, fliplr(MicVel)]);

        plot([0 xMax*ExposureTime],[0 0],':black','LineWidth',2);
        plot(FinalResult(:,1),FinalResult(:,3),'black','LineWidth',4);
        xlim([0 xMax*ExposureTime]);
        if min(FinalResult(:,3))<0
            ylim([min(FinalResult(:,3)) max(FinalResult(:,3))]);
        else
            ylim([-max(FinalResult(:,3))*0.2 max(FinalResult(:,3))*1.2]);
        end
        yLbl = ylabel('Mic. Vel. (nm/s)');
        set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
        set(gca,'FontSize',TextSize+8,'XTick',[],'FontWeight','b','LineWidth',3);
        text(0.13,1.15,'Microtubule Velocity and Kinesin On-Axis Displacement','Units','normalized','FontSize',TextSize+8,'FontWeight','bold');

        colorbar;
        caxis([0 800]);

        % Loop for on-axis equilibrium lines
        subplot(6,1,3:6); hold on;
        for i = 1:RegionNo
            % Draw hindering areas
            xTrace = FinalResult(:,4+5*(i-1));
            xTrace = xTrace(~isnan(xTrace));
            xFill = min(xTrace):ExposureTime:max(xTrace);
            Trace = FinalResult(:,5+5*(i-1));
            Trace = Trace(~isnan(Trace));
            yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
            DistForces = yTrace - EquilibriumPos(i);
            yFill = yTrace;
            EquiLine = ones(1,length(yFill))*EquilibriumPos(i);
            ZeroLine = zeros(1,length(yFill));
            xFill = [xFill, fliplr(xFill)];
            yFill = [EquiLine, fliplr(yFill)];
            %fill(xFill,yFill,[EquiLine, fliplr(DistForces)]);
            fill(xFill,yFill,[ZeroLine, fliplr(DistForces)]);
            %fill(xFill,yFill,[EquiLine, EquiLine]);

            plot([0 xMax],[EquilibriumPos(i) EquilibriumPos(i)],':black','LineWidth',2);
            plot(FinalResult(:,4+5*(i-1)),FinalResult(:,5+5*(i-1)),'k','LineWidth',4);

            % Plot the start and end points
            plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),FinalResult(StartEndMatrix(2*i-1),5+5*(i-1)),'pb','LineWidth',2,'MarkerFaceColor','blue','MarkerSize',10); hold on;
            plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),FinalResult(StartEndMatrix(2*i),5+5*(i-1)),'ob','LineWidth',2,'MarkerFaceColor','blue','MarkerSize',10); hold on;
        end

        xlim([0 xMax*ExposureTime]);
        ylim([0 MaxOnAxisDiff*RegionNo*1.08]);
        yLbl = ylabel('Kinesin On-Axis Displacement (nm)');
        set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
        xLbl = xlabel('Time (s)');
        set(xLbl, 'Units', 'Normalized', 'Position', [0.5, -0.1, 0]);
        set(gca,'FontSize',TextSize+8,'FontWeight','b','LineWidth',3);

        colorbar;
        caxis([-500 500]);
        colormap(hot);
        
        % Add label for points
        yHeight = ylim; yHeight = yHeight(2)-yHeight(1);
        for i = 1:RegionNo
    %         LastIndex = find(~isnan(FinalResult(:,4+5*(i-1))),1,'last');
    %         text(FinalResult(LastIndex,4+5*(i-1))*1.02,FinalResult(LastIndex,7+5*(i-1))+(yHeight/25),num2str(i),'Color','black','FontSize',18,'fontweight','b'); hold on;
            Trace = FinalResult(:,5+5*(i-1));
            Trace = Trace(~isnan(Trace));
            DistForces = Trace - EquilibriumPos(i);
            LabelSign = sign(max(DistForces)-abs(min(DistForces)));
%             EndIndex = str2num(StartEndFrames{i});
    %         text(EndIndex(2)*ExposureTime,EquilibriumPos(i)+LabelSign*(yHeight/12),num2str(i),'Color','black','FontSize',18,'fontweight','b'); hold on;
%             text(xMax*ExposureTime*1.01,EquilibriumPos(i),num2str(i),'Color','black','FontSize',18,'fontweight','b'); hold on;
            text(xMax*ExposureTime*0.981,EquilibriumPos(i),num2str(i),'Color','black','FontSize',18,'fontweight','b','BackgroundColor','w'); hold on;
        end

        % Find out total number of points to initialize DurDistForces
        TotalPoints = 0;
        for i = 1:RegionNo
            % Draw hindering areas
            Time = FinalResult(:,4+5*(i-1));
            Time = Time(~isnan(Time));
            TotalPoints = TotalPoints + length(Time);
        end

        % Initialize DurDistForces matrix
        DurDistForces = zeros(TotalPoints,6);
        DMMIndex = 1;
        hText = zeros(TotalPoints,6);
        hPlot = zeros(TotalPoints,6);
        TraceLengths = zeros(TotalPoints,1);

        set(hOnAxis,'units','normalized','outerposition',[0.15 0 0.85 1]);

        % Start loop to choose driving/ hindering regions
        StopLoop = 0;
        LineIndex = 1;
        while StopLoop == 0
            % Get polyline
            pan on;
            disp('Please draw a line on an equilibrium line for the duration of driving/ hindering. Double click on second point.');
            h = warndlg(sprintf('Zoom in/out or pan to desired region before clicking OK'), 'Zoom/ Pan');
            set(h, 'Position', [10 250 250 60]);
            uiwait(h); 
            choice = questdlg('Continue choosing regions?.','Continue?', ...
                    'Yes','No','Redo','Yes');

            switch choice
                case 'Yes'
                    % Get polyline
                    disp(['Please draw line ' num2str(LineIndex) '. Double click when done to close polygon.']);
                    [xBorder,yBorder] = getline(gca);

                    % Draw polyline
    %                 plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'.-m','LineWidth',5); 
    %                 plot([xBorder; xBorder(1)],[yBorder; yBorder(1)],'g','LineWidth',2); 

                    % Write LineIndex
                    hText(LineIndex) = text(mean(xBorder),mean(yBorder),num2str(LineIndex),'Color','y','FontSize',18,'fontweight','b','HorizontalAlignment','center','BackgroundColor','k');

                    % Initialize ClosestDistance
                    ClosestDistance = zeros(RegionNo,1);

                    % Find closest points
                    for i = 1:RegionNo
                        % Draw hindering areas
                        Time = FinalResult(:,4+5*(i-1));
                        Time = Time(~isnan(Time));
                        Trace = FinalResult(:,5+5*(i-1));
                        Trace = Trace(~isnan(Trace));
                        [~,D] = knnsearch(Trace,yBorder);
                        ClosestDistance(i,:) = mean(D);
                    end

                    % Find closest distance
                    [~,RegionIndex] = min(ClosestDistance);

                    % Draw lines between points
                    Time = FinalResult(:,4+5*(RegionIndex-1));
                    Time = Time(~isnan(Time));
                    Trace = FinalResult(:,5+5*(RegionIndex-1));
                    Trace = Trace(~isnan(Trace));
                    CloseIndex = knnsearch(Time,xBorder);
                    hPlot(LineIndex) = plot(Time(CloseIndex(1):CloseIndex(2)),Trace(CloseIndex(1):CloseIndex(2)),'y','LineWidth',1);
                    TraceLengths(LineIndex) = length(Trace(CloseIndex(1):CloseIndex(2)));
                    TraceLength = TraceLengths(LineIndex);

                    DurDistForces(DMMIndex:DMMIndex+TraceLength-1,1:5) = [ones(TraceLength,1)*RegionIndex ones(TraceLength,1)*LineIndex Time(CloseIndex(1):CloseIndex(2)) DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex-1:2*RegionIndex)];
                    DurDistForces(DMMIndex:DMMIndex+TraceLength-1,6) = sign(DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex-1)).*sqrt(DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex-1).^2 + DistanceForces(CloseIndex(1):CloseIndex(2),2*RegionIndex).^2);
                    DMMIndex = DMMIndex + TraceLength;

                    LineIndex = LineIndex + 1;
                case 'No'
                    StopLoop = 1;
                case 'Redo'
                    % Undo record on DMMIndex, DurDistForces and LineIndex
                    LineIndex = LineIndex - 1;
                    TraceLength = TraceLengths(LineIndex);
                    DMMIndex = DMMIndex - TraceLength;
                    DurDistForces(DMMIndex:DMMIndex+TraceLength-1,:) = 0;

                    % Delete previous text and trace plot
                    delete(hText(LineIndex));
                    delete(hPlot(LineIndex));
            end
        end
        set(hOnAxis,'units','normalized','outerposition',[0 0 1 1]);
        
        % Zoom out, save image with spot regions, then close
        zoom out;

        % Delete zeros in DurDistForces
        DurDistForces(DurDistForces(:,1)==0,:) = [];

        % Import FinalResult2 to assign zeros to DistanceForces without hinder/ drive
        FinalResult2 = xlsread(FileName);

        % Get unique values in DurDistForces
        DurDistForcesUnique = unique(DurDistForces(:,[1 3:6]),'rows');

        % Set DistanceForces without hinder/drive to be zero
        for i = 1:RegionNo
            % Find times for the specific region
            DurTime = DurDistForcesUnique(DurDistForcesUnique(:,1)==i,2);

            % Find non NaN values in time for specific region
            RegionTime = FinalResult2(~isnan(FinalResult2(:,4+7*(i-1))),4+7*(i-1));

            % Find indices that has no hinder/ drive
            NoHinderDriveIndex = ~ismember(RegionTime,DurTime);

            % Set on and off-axis DistanceForces without Drive/Hinder to be 0
            FinalResult2(NoHinderDriveIndex,7+7*(i-1)) = 0;
            FinalResult2(NoHinderDriveIndex,10+7*(i-1)) = 0;
        end

        % Save figure for Dur-DistForces
        if exist('Image Files','dir')~=7; mkdir('Image Files'); end
        print('-djpeg','-r300',['Image Files\' strrep(FileName,'(Final Result2).xlsx','(Dur-DistForces).png')]);
        savefig(['Image Files\' strrep(FileName,'(Final Result2).xlsx','(Dur-DistForces).fig')]);

        % Write to EXCEL spreadsheet
        FirstLine = cell(1,RegionNo*7+4);
        FirstLine(1,1:3) = {'Microtubule Time (s)','Microtubule Displacement (nm)','Microtubule Velocity (nm/s)'};
        for i = 1:RegionNo
            % Define titles of each column
            FirstLine(1,4+7*(i-1):10+7*(i-1)) = {['Point ' num2str(i) ' Time (s)'],['Point ' num2str(i) ' On-Axis Displacement (nm)'],['Point ' num2str(i) ' On-Axis Step-Fit (nm)'],['Point ' num2str(i) ' On-Axis DistanceForce (nm)'],['Point ' num2str(i) ' Off-Axis Displacement (nm)'],['Point ' num2str(i) ' Off-Axis Step-Fit (nm)'],['Point ' num2str(i) ' Off-Axis DistanceForce (nm)']};
        end
        FirstLine(1,end) = {'On-Axis Equilibrium Point'};

        FinalResult2 = num2cell(FinalResult2);
        FinalResult2 = [FirstLine; FinalResult2];
        FinalResult2(RegionNo+2,end) = {'Off-Axis Equilibrium Point'};
        FinalResult2(2*RegionNo+3,end) = {'Start and End Frames'};

        % Write FinalResult2 to EXCEL file
        delete([FileInitial ' (Final Result3).xlsx']);
        xlswrite([FileInitial ' (Final Result3).xlsx'],FinalResult2);

        % Write DurDistForces to EXCEL file
        delete([FileInitial ' (Dur-DistForces).xlsx']);
        xlswrite([FileInitial ' (Dur-DistForces).xlsx'],DurDistForces);
        %% Resave figure for On-axis shading
        if exist('Image Files','dir')~=7; mkdir('Image Files'); end
        print('-djpeg','-r300',['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-On-Axis).png')]);
        savefig(['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-On-Axis).fig')]);
        %% Resave figure for Off-axis? shading
        if exist('Image Files','dir')~=7; mkdir('Image Files'); end
        print('-djpeg','-r300',['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-Off-Axis).png')]);
        savefig(['Image Files\' strrep(ResultName,'(Final Result).xlsx','(Graphs-Off-Axis).fig')]);
        %% Plot Mic. Velocity and Total DistanceForce
        % Define xLimit
        %xLimit = [15 40];           % Comment out if not needed
        
        % Import FinalResult to get MaxOnAxisDiff
        FileName = dir('*(Final Result).xlsx');
        FileName = FileName.name;
        FinalResult = xlsread(FileName);
        RegionNo = round((size(FinalResult,2)-3)/3);

        % Get MaxOnAxisDiff and MaxOffAxisDiff
        for i = 1:RegionNo
            if i == 1
                MaxOnAxisDiff = max(FinalResult(:,5+3*(i-1)))-min(FinalResult(:,5+3*(i-1)));
                MaxOffAxisDiff = max(FinalResult(:,6+3*(i-1)))-min(FinalResult(:,6+3*(i-1)));
            else
                MaxOnAxisDiff = max(MaxOnAxisDiff,max(FinalResult(:,5+3*(i-1)))-min(FinalResult(:,5+3*(i-1))));
                MaxOffAxisDiff = max(MaxOffAxisDiff,max(FinalResult(:,6+3*(i-1)))-min(FinalResult(:,6+3*(i-1))));
            end
        end

        % Import FinalResult2 and assign to FinalResult
        FileName = dir('*(Final Result2).xlsx');
        FileName = FileName.name;
        FinalResult = xlsread(FileName);
        RegionNo = round((size(FinalResult,2)-4)/7);
        TextSize = 12;

        % Get PixelSize and ExposureTime
        FileInitial = strrep(FileName,' (Final Result2).xlsx','');
        ExposureTime = xlsread([FileInitial ' (Variables).xlsx']);
        ExposureTime = ExposureTime(2);    % Exposure time in sec
        %PixelSize = PixelSize(1);       % PixelSize in nm

        % Get EquilibriumPos and StartEndMatrix
%         EquilibriumPos = FinalResult(1:RegionNo,end);
        StartEndMatrix = FinalResult(2*RegionNo+3:4*RegionNo+2,end);

        % Get DistanceForces and calculate TotalDistanceForces
        DistanceForces = zeros(size(FinalResult,1),RegionNo*2);
        TotalDistanceForces = zeros(size(FinalResult,1),RegionNo);
        for i = 1:RegionNo
            DistanceForce = [FinalResult(:,7+7*(i-1)) FinalResult(:,10+7*(i-1))];
            DistanceForces(:,2*i-1:2*i) = DistanceForce;
            TotalDistanceForces(:,i) = sqrt(DistanceForce(:,1).^2 + DistanceForce(:,2).^2).*sign(DistanceForce(:,1));
        end
        
        % Get TotalDistDiff
        TotalDistDiff = zeros(RegionNo,1);
        for i = 1:RegionNo
            TotalDistDiff(i) = max(TotalDistanceForces(:,i))-min(TotalDistanceForces(:,i));
        end
        
        % Get CumulativeDistDiff
        CumulativeDistDiff = [0; cumsum(TotalDistDiff + mean(TotalDistDiff)*0.1)];
        
        % Get Offsetted TotalDistanceForces and EquilibriumPos
        OffsetDistForces = zeros(size(FinalResult,1),RegionNo);
        EquilibriumPos = zeros(RegionNo,1);
        for i = 1:RegionNo
            OffsetDistForces(:,i) = TotalDistanceForces(:,i) + CumulativeDistDiff(i) - min(TotalDistanceForces(:,1))*1.1;
            EquilibriumPos(i) = CumulativeDistDiff(i) - min(TotalDistanceForces(:,1))*1.1;
        end
        
%         for i = 1:RegionNo
%             FinalResult(:,6+5*(i-1)) = FinalResult(:,6+5*(i-1)) - (max(FinalResult(:,5+5*(i-1)))+min(FinalResult(:,5+5*(i-1))))/2 + ((i-1)*MaxOnAxisDiff) + MaxOnAxisDiff*0.55;
%             FinalResult(:,5+5*(i-1)) = FinalResult(:,5+5*(i-1)) - (max(FinalResult(:,5+5*(i-1)))+min(FinalResult(:,5+5*(i-1))))/2 + ((i-1)*MaxOnAxisDiff) + MaxOnAxisDiff*0.55;
%             FinalResult(:,8+5*(i-1)) = FinalResult(:,8+5*(i-1)) - (max(FinalResult(:,7+5*(i-1)))+min(FinalResult(:,7+5*(i-1))))/2 + ((i-1)*MaxOffAxisDiff) + MaxOffAxisDiff*0.55;
%             FinalResult(:,7+5*(i-1)) = FinalResult(:,7+5*(i-1)) - (max(FinalResult(:,7+5*(i-1)))+min(FinalResult(:,7+5*(i-1))))/2 + ((i-1)*MaxOffAxisDiff) + MaxOffAxisDiff*0.55;
%         end

        % Delete DistanceForces column
        for i = RegionNo:-1:1
            % Delete DistanceForces from FinalResult2
            FinalResult(:,10+7*(i-1)) = [];
            FinalResult(:,7+7*(i-1)) = [];
        end

        % Get xMax, maximum value in x
        xMax = max(FinalResult(:,1));
        for i = 1:RegionNo
            xMax = max(xMax,max(FinalResult(:,4+5*(i-1))));
        end
        xMax = xMax/ExposureTime;

        % Plot just microtubule velocity and on-axis with shading
        hOnAxis = figure('units','normalized','outerposition',[0 0 1 1]);
        MicAxes = subplot(6,1,1:2); hold on;

        xTrace = FinalResult(:,1);
        xTrace = xTrace(~isnan(xTrace));
        xFill = min(xTrace):ExposureTime:max(xTrace);
        Trace = FinalResult(:,3);
        Trace = Trace(~isnan(Trace));
        yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
        MicVel = yTrace;
        EquiLine = zeros(1,length(yTrace));
        xFill = [xFill, fliplr(xFill)];
        yTrace = [EquiLine, fliplr(yTrace)];
        fill(xFill,yTrace,[EquiLine, fliplr(MicVel)]);

        plot([0 xMax*ExposureTime],[0 0],':black','LineWidth',2);
        plot(FinalResult(:,1),FinalResult(:,3),'black','LineWidth',4);
        if exist('xLimit','var')
            xlim(xLimit);
        else
            xlim([0 xMax*ExposureTime]);
        end
        if min(FinalResult(:,3))<0
            ylim([min(FinalResult(:,3)) max(FinalResult(:,3))]);
        else
            ylim([-max(FinalResult(:,3))*0.2 max(FinalResult(:,3))*1.2]);
        end
        yLbl = ylabel('Mic. Vel. (nm/s)');
        set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
        set(gca,'FontSize',TextSize+8,'XTick',[],'FontWeight','b','LineWidth',3);
        text(0.13,1.15,'Microtubule Velocity and Kinesin On-Axis Displacement','Units','normalized','FontSize',TextSize+8,'FontWeight','bold');

        colorbar;
        caxis([0 800]);
        
        % Loop to draw TotalDistanceForces
        DistanceAxes = subplot(6,1,3:6); hold on;
        for i = 1:RegionNo
            % Draw hindering areas
            xTrace = FinalResult(:,4+5*(i-1));
            xTrace = xTrace(~isnan(xTrace));
            xFill = min(xTrace):ExposureTime:max(xTrace);
            Trace = OffsetDistForces(:,i);
            Trace = Trace(~isnan(Trace));
            yTrace = interp1(xTrace,Trace,xFill,'linear','extrap');
            DistForces = yTrace - EquilibriumPos(i);
            yFill = yTrace;
            EquiLine = ones(1,length(yFill))*EquilibriumPos(i);
            ZeroLine = zeros(1,length(yFill));
            xFill = [xFill, fliplr(xFill)];
            yFill = [EquiLine, fliplr(yFill)];
            %fill(xFill,yFill,[EquiLine, fliplr(DistForces)]);
            fill(xFill,yFill,[ZeroLine, fliplr(DistForces)]);
            %fill(xFill,yFill,[EquiLine, EquiLine]);

            plot([0 xMax],[EquilibriumPos(i) EquilibriumPos(i)],':black','LineWidth',2);
            plot(FinalResult(:,4+5*(i-1)),OffsetDistForces(:,i),'k','LineWidth',4);

            % Plot the start and end points
            plot(FinalResult(StartEndMatrix(2*i-1),4+5*(i-1)),OffsetDistForces(StartEndMatrix(2*i-1),i),'pb','LineWidth',10,'MarkerFaceColor','blue','MarkerSize',10); hold on;
            plot(FinalResult(StartEndMatrix(2*i),4+5*(i-1)),OffsetDistForces(StartEndMatrix(2*i),i),'ob','LineWidth',10,'MarkerFaceColor','blue','MarkerSize',10); hold on;
        end

        % Add label for points
        for i = 1:RegionNo
            Trace = OffsetDistForces(:,i);
            Trace = Trace(~isnan(Trace));
            DistForces = Trace - EquilibriumPos(i);
            LabelSign = sign(max(DistForces)-abs(min(DistForces)));
            EndIndex = str2num(StartEndFrames{i});
            text(xMax*ExposureTime*0.981,EquilibriumPos(i),num2str(i),'Color','black','FontSize',18,'fontweight','b','BackgroundColor','w'); hold on;
        end
        
        if exist('xLimit','var')
            xlim(xLimit);
        else
            xlim([0 xMax*ExposureTime]);
        end
        ylim([0 CumulativeDistDiff(RegionNo)*1.15 - min(TotalDistanceForces(:,1))*1.1]);
        yLbl = ylabel('Kinesin On-Axis Displacement (nm)');
        set(yLbl, 'Units', 'Normalized', 'Position', [-0.1, 0.5, 0]);
        xLbl = xlabel('Time (s)');
        set(xLbl, 'Units', 'Normalized', 'Position', [0.5, -0.1, 0]);
        set(gca,'FontSize',TextSize+8,'FontWeight','b','LineWidth',3);
        
        colorbar;
        caxis([-500 500]);
        colormap(hot);

        % Get StartEndFrames
        StartEndFrames = cell(1,RegionNo);
        StartEndIndex = FinalResult(2*RegionNo+3:4*RegionNo+2,end);
        for i = 1:RegionNo
            StartEndFrames(i) = {num2str(FinalResult(StartEndIndex(2*i-1:2*i),4+5*(i-1))/ExposureTime)};
        end

        % Draw lines along along y axis of cursor
        dcm_obj=datacursormode;
        h = msgbox('Choose one point to draw vertical line','Choose Cursors');
        set(h, 'Position', [80 250 152 60]);
        uiwait(h);
        Cursor=getCursorInfo(dcm_obj);
        dcm_obj.removeAllDataCursors();
        plot([Cursor.Position(1) Cursor.Position(1)],ylim,'r');
        
        axes(MicAxes); hold on;
        plot([Cursor.Position(1) Cursor.Position(1)],ylim,'r');
        
        % Save figure
        if exist('Image Files','dir')~=7; mkdir('Image Files'); end
        print('-djpeg','-r300',['Image Files\' strrep(FileName,'(Final Result2).xlsx','(ZoomedIn).png')]);
        savefig(['Image Files\' strrep(FileName,'(Final Result2).xlsx','(ZoomedIn).fig')]);
        %% Plot individual TotalDistanceForces traces
        L0 = 530;   % Contour length in nm
        xLabelLocation = 31.8;      % x Location to label for each trace
        
        for i=1:RegionNo
            xTrace = FinalResult(:,4+5*(i-1));
            xTrace = xTrace(~isnan(xTrace));
            Trace = TotalDistanceForces(:,i);
            Trace = Trace(~isnan(Trace));

            % Plot traces and save
            figure;
            plot(xTrace,Trace,'b','LineWidth',3); hold on;
            if exist('xLimit','var')
                xlim(xLimit);
            else
                xlim([0 xMax*ExposureTime]);
            end
            yLbl = ylabel('Distance Forces (nm)');
            title(['Trace (' num2str(i) '): Distance Forces over Time']);
            set(yLbl, 'Units', 'Normalized', 'Position', [-0.11, 0.5, 0]);
            xLbl = xlabel('Time (s)');
            set(xLbl, 'Units', 'Normalized', 'Position', [0.5, -0.07, 0]);
            set(gca,'FontSize',TextSize+1,'FontWeight','b','LineWidth',3);
            
            % Draw contour length line
            plot(xlim,[L0 L0],'r');
            plot(xlim,[-L0 -L0],'r');
            
            % Label DistanceForces at a certain point
            yDistForce = Trace(xTrace==xLabelLocation);
            plot(xLabelLocation,yDistForce,'*r');
            text(xLabelLocation*1.05,yDistForce,num2str(round(yDistForce)),'Color','black','FontSize',18,'fontweight','b'); hold on;
        
            % Save figure
            if exist('Image Files','dir')~=7; mkdir('Image Files'); end
            print('-djpeg','-r300',['Image Files\' strrep(FileName,'(Final Result2).xlsx',['(DistForce-' num2str(i) ').png'])]);
            savefig(['Image Files\' strrep(FileName,'(Final Result2).xlsx',['(DistForce-' num2str(i) ').fig'])]);
        end        
        %% Convert TotalDistanceForces to Force
        Force = NaN(size(TotalDistanceForces));
        DistanceOffset = 50;        % DistanceOffset in nm, representing the additional length of DNA
        % 10 nm SA-QD/2, 10 nm Anti-Dig, 5 nm SA, 5 nm PEG
        CorrectedDistance = abs(abs(TotalDistanceForces) - DistanceOffset);
        L0 = 530;   % Contour length in nm
        kT = 4.114; % Boltzmann constant time temperature in pN nm
        P = 45;     % Persistence length in nm
%         Offset = 2; % Graph offset in pN
        % Convert distance to force
        figure; hold on;
        for i=1:RegionNo
            xTrace = FinalResult(:,4+5*(i-1));
            xTrace = xTrace(~isnan(xTrace));
            Distance = CorrectedDistance(:,i);
            Distance = Distance(~isnan(Distance));
            ForceSign = sign(TotalDistanceForces(:,i));
            ForceSign = ForceSign(~isnan(ForceSign));
            
            % Make Distance above 0.98*L0 to be 0.98*L0
            SaturateIndex = find(Distance>L0*0.98);
            Distance(SaturateIndex)=L0*0.98;

            Force(1:length(Distance),i)=(1/4*(1-(Distance/L0)).^(-2)-1/4+(Distance/L0)-0.8*(Distance/L0).^2.15)*kT/P.*ForceSign;
            plot(xTrace,Force(1:length(Distance),i)); hold on;
        end
    %% 3) Get microtubule lengths
    % Parameters for fitting microtubule
    SmoothFraction = 0.5;       % Fraction of pixels to be smoothed
    LengthFactor = 0.4;         % How many times to lengthen a line to find connection
    AngleThreshold = 10;        % Define angle below which lines are considered the same
    LengthThreshold = 20;       % Define lengths to find the angle
    CoverageThreshold = 0.40;   % Coverage fraction needed to count SearchedLine as part of manually fitted spline
    ImageHistoryThreshold = 3;  % How many images before current images to look through for reference in deciding current image
    
    FileInputName = dir('*(Mic).tif');
    FileInputName = FileInputName.name;
    [pathstr,FileName,ext] = fileparts(FileInputName);
    FileInitial = strrep(FileInputName,' (Mic).tif','');
    info = imfinfo(FileInputName);
    %data=imread(FileInputName, 1, 'Info', info);
    num_images = numel(info);       
    dim = [info(1).Height info(1).Width num_images];
    dataBW = uint8(zeros(dim));
    FinalMicCoordinates = cell(num_images,1);
    MicLengths = zeros(num_images,1);
    
    % Get PixelSize and ExposureTime
    PixelSize = xlsread([FileInitial ' (Variables).xlsx']);
    ExposureTime = PixelSize(2);    % Exposure time in sec
    PixelSize = PixelSize(1);       % PixelSize in nm
   
    % Issue dialog box to ask for frames to manually fit microtubules
    if exist('Frames','var')
        Frames = inputdlg('Choose frames to manually fit microtubules (comma separated). Include Frame 1.','Select Frames',1,{strjoin(arrayfun(@(x) num2str(x),Frames,'UniformOutput',false),',')});
    else
        Frames = inputdlg('Choose frames to manually fit microtubules (comma separated). Include Frame 1.','Select Frames',1,{'1'});
    end
    Frames = strsplit(Frames{1},',');   % Separate the commas to generate cells of strings
    Frames = cellfun(@str2num,Frames);  % Change cells of strings to matrix
    
    % Issue dialog box to ask for frames to manually fit microtubules
    if exist('AllMicPresent','var')
        AllMicPresent = inputdlg({'Start (0 when no frames have entire microtubule present):','End (0 when no frames have entire microtubule present):'},'Entire microtubule present?',[1 80; 1 80],{num2str(AllMicPresent(1)),num2str(AllMicPresent(2))});
    else
        AllMicPresent = inputdlg({'Start (0 when no frames have entire microtubule present):','End (0 when no frames have entire microtubule present):'},'Entire microtubule present?',[1 80; 1 80],{'1',num2str(num_images)});
    end
    
    AllMicPresent = cellfun(@str2num,AllMicPresent);    % Change cells of strings to matrix
    AllMicThere = zeros(num_images,1);
    if sum(AllMicPresent) > 0
        AllMicThere(AllMicPresent(1):AllMicPresent(2)) = 1;
    end
    
    MicCoordinates = cell(length(Frames),1);
    MicCoordinatesLengths = zeros(length(Frames),1);
    close all;
    figure;
    for i = 1:length(Frames)
        data = imread(FileInputName,Frames(i),'Info',info);
        h=imshow(imadjust(data));
        set(gcf, 'Units', 'Normalized', 'OuterPosition', [0 0 1 1]);
        
        [xMic,yMic] = getline(gca);
        MicCoordinates{i} = [xMic yMic];
        if size(MicCoordinates{i},1) < 2
            MicCoordinates{i} = [MicCoordinates{i};MicCoordinates{i}+1];
        end
    end
    close all;
    
    % Initialize zero matrix to quickly initialize BinaryMatrix later
    ZeroMatrix = false(dim(1),dim(2));

    % Initialize ManualCount for manually fitted spline
    ManualCount = 1;
    tic;
    for i=1:num_images
        data = imread(FileInputName, i, 'Info', info);
        data = round(data*(255/double(max(max(data)))));

        BW = imbinarize(data,'adaptive','Sensitivity',0.4);
        BW_out = bwpropfilt(BW, 'Area', [50,100000]);
        BWMajority = bwmorph(BW_out,'majority');
        BWSpur = bwmorph(BWMajority,'spur');
        BWMajority = bwmorph(BWSpur,'majority');
        BW2 = bwmorph(BWMajority,'thin',Inf);
        BWBranch = bwmorph(BW2,'branchpoints');
        BWBranch = bwmorph(BWBranch,'thicken',1);
        BWBranch = 1 - BWBranch;
        BWSingled = BW2.*BWBranch;
        BWSingled = bwmorph(BWSingled,'clean');

        BWLabeled = uint16(bwlabel(BWSingled));     % Label each region with 1, 2, 3, etc
        stats = regionprops(BWLabeled,BWLabeled,'MaxIntensity','Area', 'PixelList');     % Find out areas for each region
        
        % Apply manual fitted spline if there is any. Otherwise look at
        % previous images for reference
        if i == Frames(ManualCount)
            % Interpolate MicCoordinates with 50x more points
            x = 1:size(MicCoordinates{ManualCount},1);
            xq = 1:0.02:size(MicCoordinates{ManualCount},1);
            if size(MicCoordinates{ManualCount},1) >=4
                InterpPoints = [interp1(x,MicCoordinates{ManualCount}(:,2),xq,'spline')' interp1(x,MicCoordinates{ManualCount}(:,1),xq,'spline')'];
            else
                InterpPoints = [interp1(x,MicCoordinates{ManualCount}(:,2),xq,'linear')' interp1(x,MicCoordinates{ManualCount}(:,1),xq,'linear')'];
            end
            
            % Find microtubule length and assign to MicCoordinatesLengths
            MicCoordinatesLengths(ManualCount) = sum(sqrt((InterpPoints(2:end,1)-InterpPoints(1:end-1,1)).^2 + (InterpPoints(2:end,2)-InterpPoints(1:end-1,2)).^2))*PixelSize;
   
            % Change interpolated lines to binary
            InterpPoints = round(InterpPoints);
            InterpPoints(find(InterpPoints(:,1)>=dim(1) | InterpPoints(:,1)<=1),:)=[];
            InterpPoints(find(InterpPoints(:,2)>=dim(2) | InterpPoints(:,2)<=1),:)=[];
            BinaryMatrix = ZeroMatrix;
            linearInd = sub2ind(size(BinaryMatrix),InterpPoints(:,1),InterpPoints(:,2));
            BinaryMatrix(linearInd) = 1;
            % figure; imshow(BinaryMatrix);

            % Thicken image
            se = strel('sphere',1);
            BinaryMatrix = imdilate(BinaryMatrix,se);

            % Find Overlapping lines by looking at intersection of extended
            % line (BinaryMatrix) with labeled non-extended lines (BWLabeled)
            OverlapMatrix = uint16(BinaryMatrix).*BWLabeled;
            OverlapID = unique(OverlapMatrix(OverlapMatrix>0));

            % Determine if Overlapped lines are part of the manually fitted spline
            % by looking at coverage fraction
            FinalPointsCount = 1;
            j = 1;
            while j < length(OverlapID)
                SearchedLine = stats(OverlapID(j)).PixelList;
                % Find how many pixels are overlapping
                SearchedMatrix = ZeroMatrix;
                linearInd = sub2ind(size(SearchedMatrix),SearchedLine(:,2),SearchedLine(:,1));
                SearchedMatrix(linearInd) = 1;
                SearchedMatrix = SearchedMatrix.*BinaryMatrix;  % Ones for overlapped pixels between SearchedMatrix and BinaryMatrix
                CoverageFraction = sum(sum(SearchedMatrix))/size(SearchedLine,1);
                if CoverageFraction < CoverageThreshold
                    OverlapID(j) = [];
                else
                    FinalPointsCount = FinalPointsCount + size(SearchedLine,1);
                    j = j + 1;
                end
            end
            
            if ManualCount < size(MicCoordinates,1)
                ManualCount = ManualCount + 1;
            end
        else
            % Look for past few frames for reference
            if i > ImageHistoryThreshold
                ImageLoop = ImageHistoryThreshold;
            else
                ImageLoop = i-1;
            end
            
            OverlapIDs = cell(ImageLoop,1);
            OverlapCounts = 0;
            
            for ii = 1:ImageLoop
                % Get BinaryMatrix from past images
                BinaryMatrix = logical(dataBW(:,:,i-ii));

                % Thicken image
                se = strel('sphere',1);
                BinaryMatrix = imdilate(BinaryMatrix,se);

                % Find Overlapping lines by looking at intersection of extended
                % line (BinaryMatrix) with labeled non-extended lines (BWLabeled)
                OverlapMatrix = uint16(BinaryMatrix).*BWLabeled;
                OverlapID = unique(OverlapMatrix(OverlapMatrix>0));

                % Determine if Overlapped lines are part of the manually fitted spline
                % by looking at coverage fraction
                FinalPointsCount = 1;
                while j < length(OverlapID)
                    SearchedLine = stats(OverlapID(j)).PixelList;
                    % Find how many pixels are overlapping
                    SearchedMatrix = ZeroMatrix;
                    linearInd = sub2ind(size(SearchedMatrix),SearchedLine(:,2),SearchedLine(:,1));
                    SearchedMatrix(linearInd) = 1;
                    SearchedMatrix = SearchedMatrix.*BinaryMatrix;  % Ones for overlapped pixels between SearchedMatrix and BinaryMatrix
                    CoverageFraction = sum(sum(SearchedMatrix))/size(SearchedLine,1);
                    if CoverageFraction < CoverageThreshold
                        OverlapID(j) = [];
                    else
                        FinalPointsCount = FinalPointsCount + size(SearchedLine,1);
                        j = j + 1;
                    end
                end

                OverlapCounts = OverlapCounts + length(OverlapID);
                OverlapIDs{ii} = OverlapID;
            end

            % Identify unique OverlapIDs
            OverlapID = zeros(OverlapCounts,1);
            OverlapCounts = 1;
            for ii = 1:length(OverlapIDs)
                if ~isempty(OverlapIDs{ii})
                    OverlapID(OverlapCounts:OverlapCounts+length(OverlapIDs{ii})-1) = OverlapIDs{ii};
                    OverlapCounts = OverlapCounts + length(OverlapIDs{ii});
                end
            end
            
            OverlapID = nonzeros(OverlapID);
            OverlapID = unique(OverlapID);
        end
        
        if ~isempty(OverlapID)
            % Combine all points that are overlapping
            FinalPoints = zeros(FinalPointsCount,2);
            FinalPointsCount = 1;
            for j = 1:length(OverlapID)
                SearchedLine = stats(OverlapID(j)).PixelList;
                FinalPoints(FinalPointsCount:FinalPointsCount+size(SearchedLine,1)-1,:) = SearchedLine;
            end

            % Get smoothed lines for overlapping points
            Sort1 = FinalPoints;
            Sort2 = sortrows(Sort1,2);
            if abs(Sort1(end,1)-Sort1(1,1)) < abs(Sort2(end,2)-Sort2(1,2))
                rowSmooth = smooth(Sort2(:,1),SmoothFraction,'loess');
                colSmooth = smooth(Sort2(:,2),SmoothFraction,'loess');
            else
                rowSmooth = smooth(Sort1(:,1),SmoothFraction,'loess');
                colSmooth = smooth(Sort1(:,2),SmoothFraction,'loess');
            end
            SmoothedFinalPoints = [rowSmooth colSmooth];

            % Interpolate SmoothedFinalPoints with 50x more points
            x = 1:size(SmoothedFinalPoints,1);
            xq = 1:0.02:size(SmoothedFinalPoints,1);
            if size(SmoothedFinalPoints,1) >=4
                InterpPoints = [interp1(x,SmoothedFinalPoints(:,2),xq,'spline')' interp1(x,SmoothedFinalPoints(:,1),xq,'spline')'];
            else
                InterpPoints = [interp1(x,SmoothedFinalPoints(:,2),xq,'linear')' interp1(x,SmoothedFinalPoints(:,1),xq,'linear')'];
            end

            % Find microtubule length and assign to MicLengths
            MicLengths(i) = sum(sqrt((InterpPoints(2:end,1)-InterpPoints(1:end-1,1)).^2 + (InterpPoints(2:end,2)-InterpPoints(1:end-1,2)).^2))*PixelSize;

            % Change smoothed lines to binary
            InterpPoints = round(InterpPoints);
            InterpPoints(find(InterpPoints(:,1)>=dim(1) | InterpPoints(:,1)<=1),:)=[];
            InterpPoints(find(InterpPoints(:,2)>=dim(2) | InterpPoints(:,2)<=1),:)=[];
            BinaryMatrix = ZeroMatrix;
            linearInd = sub2ind(size(BinaryMatrix),InterpPoints(:,1),InterpPoints(:,2));
            BinaryMatrix(linearInd) = 1;

            % Assign results into matrices
            dataBW(:,:,i) = BinaryMatrix;
            FinalMicCoordinates{i} = SmoothedFinalPoints;
        else
            MicLengths(i) = 0;
        end
        
if 0
        % Loop a number of times to connect lines
        for loop = 1:3
            % Get smoothed lines
%             if loop == 3
%                 figure; imshow(BWSingled); hold on;
%             end
            for ii = 1:size(stats,1)
                Sort1 = stats(ii).PixelList;
                Sort2 = sortrows(Sort1,2);
                if abs(Sort1(end,1)-Sort1(1,1)) < abs(Sort2(end,2)-Sort2(1,2))
                    rowSmooth = smooth(Sort2(:,1),SmoothFraction,'loess');
                    colSmooth = smooth(Sort2(:,2),SmoothFraction,'loess');
                else
                    rowSmooth = smooth(Sort1(:,1),SmoothFraction,'loess');
                    colSmooth = smooth(Sort1(:,2),SmoothFraction,'loess');
                end
%                 if loop == 3
%                     plot(rowSmooth,colSmooth,'r','LineWidth',2);
%                 end
                stats(ii).SmoothedData = [rowSmooth colSmooth];
            end

            % Sort array based on area
            statsCell = struct2cell(stats)';
            statsCell = sortrows(statsCell,-1);

            % Connect lines
            AngleMatrix = cell(size(statsCell,1),1);
            for ii = 1:size(statsCell,1)
                Sort1 = statsCell{ii,4};
                %figure; imshow(BWSingled); hold on;
                %plot(Sort1(:,1),Sort1(:,2),'r','LineWidth',2);

                % Extrapolate lines
                AddPoints = 20;             % Add how many datapoints to both ends to extrapolate
                x = AddPoints+1:size(Sort1,1)+AddPoints;
                xq = 1:0.2:size(Sort1,1)+2*AddPoints;
                %ExtrapPoints = [interp1(x,Sort1(:,1),xq,'pchip')' interp1(x,Sort1(:,2),xq,'pchip')'];
                ExtrapPoints = [interp1(x,Sort1(:,1),xq,'linear','extrap')' interp1(x,Sort1(:,2),xq,'linear','extrap')'];

                % Find lengths
                LineLength = sqrt((Sort1(1,1)-Sort1(end,1)).^2+(Sort1(1,2)-Sort1(end,2)).^2);
                NewLineLength1 = sqrt((ExtrapPoints(1,1)-Sort1(1,1)).^2+(ExtrapPoints(1,2)-Sort1(1,2)).^2);
                NewLineLength2 = sqrt((ExtrapPoints(end,1)-Sort1(end,1)).^2+(ExtrapPoints(end,2)-Sort1(end,2)).^2);

                % Re-extrapolate so that new line is longer than original by
                % factor of LengthFactor
                NewPoints1 = ceil(LineLength/NewLineLength1*AddPoints*LengthFactor);
                NewPoints2 = ceil(LineLength/NewLineLength2*AddPoints*LengthFactor);
                x = NewPoints1+1:size(Sort1,1)+NewPoints1;
                xq = 1:0.2:size(Sort1,1)+NewPoints1+NewPoints2;
                %ExtrapPoints = [interp1(x,Sort1(:,1),xq,'pchip')' interp1(x,Sort1(:,2),xq,'pchip')'];
                ExtrapPoints = [interp1(x,Sort1(:,1),xq,'linear','extrap')' interp1(x,Sort1(:,2),xq,'linear','extrap')'];

                %plot(ExtrapPoints(:,1),ExtrapPoints(:,2),'r','LineWidth',2);

                % Change lines to binary
                ExtrapPoints = round(ExtrapPoints);
                ExtrapPoints(find(ExtrapPoints(:,1)>=dim(1) | ExtrapPoints(:,1)<=1),:)=[];
                ExtrapPoints(find(ExtrapPoints(:,2)>=dim(2) | ExtrapPoints(:,2)<=1),:)=[];
                BinaryMatrix = ZeroMatrix;
                linearInd = sub2ind(size(BinaryMatrix),ExtrapPoints(:,1),ExtrapPoints(:,2));
                BinaryMatrix(linearInd) = 1;
                %figure; imshow(BinaryMatrix);

                % Thicken image
                se = strel('sphere',3);
                BinaryMatrix = imdilate(BinaryMatrix,se);
                %imshow(BinaryMatrix); hold on;
                %plot(Sort1(:,1),Sort1(:,2),'r','LineWidth',2);

                % Find Overlapping lines by looking at intersection of extended
                % line (BinaryMatrix) with labeled non-extended lines (BWLabeled)
                OverlapMatrix = uint16(BinaryMatrix).*BWLabeled;
                %figure; imshow(imadjust(OverlapMatrix));
                OverlapID = unique(OverlapMatrix(OverlapMatrix>0));
                OverlapID(OverlapID == statsCell{ii,3}) = [];

                % Determine if Overlapping lines is part of longer line
                for j = 1:length(OverlapID)
                    CurrentLine = statsCell{ii,4};
                    SearchedLine = stats(OverlapID(j)).SmoothedData;
                    IDX1 = knnsearch(CurrentLine,SearchedLine);
                    IDX1 = mode(IDX1);      % IDX1 is the index of current line closest to the searched line
                    IDX2 = knnsearch(SearchedLine,CurrentLine(IDX1,:)); % IDX2 is the index of the searched line closest to current line
                    Distance = sqrt((CurrentLine(IDX1,1)-SearchedLine(IDX2,1)).^2 + (CurrentLine(IDX1,2)-SearchedLine(IDX2,2)).^2);

                    % Find CurrentAngle
                    CurVect1 = CurrentLine(IDX1,:);     % Get x,y point of the current line closest to searched line
                    % Get second x,y point of a section of the current line closest to searched line
                    if size(CurrentLine,1) > LengthThreshold
                        if IDX1 > LengthThreshold
                            if IDX1-LengthThreshold > 0
                                CurVect2 = CurrentLine(IDX1-LengthThreshold,:);
                            else
                                CurVect2 = CurrentLine(1,:);
                            end
                        else
                            if IDX1+LengthThreshold <= size(CurrentLine,1)
                                CurVect2 = CurrentLine(IDX1+LengthThreshold,:);
                            else
                                CurVect2 = CurrentLine(size(CurrentLine,1),:);
                            end
                        end
                    else
                        if IDX1 > 1
                            CurVect2 = CurrentLine(1,:);
                        else
                            CurVect2 = CurrentLine(size(CurrentLine,1),:);
                        end
                    end
                    CurrentAngle = atand(-(CurVect2(1,2)-CurVect1(1,2))/(CurVect2(1,1)-CurVect1(1,1)));

                    % Find SearchedAngle
                    SearchedVect1 = SearchedLine(IDX2,:);   % Get x,y point of the point on the searched line closest to current line
                    % Get second x,y point of the section of the searched line closest to the current line
                    if size(SearchedLine,1) > LengthThreshold
                        if IDX2 > LengthThreshold
                            if IDX2-LengthThreshold > 0
                                SearchedVect2 = SearchedLine(IDX2-LengthThreshold,:);
                            else
                                SearchedVect2 = SearchedLine(1,:);
                            end
                        else
                            if IDX2+LengthThreshold <= size(SearchedLine,1)
                                SearchedVect2 = SearchedLine(IDX2+LengthThreshold,:);
                            else
                                SearchedVect2 = SearchedLine(size(SearchedLine,1),:);
                            end
                        end
                    else
                        if IDX2 > 1
                            SearchedVect2 = SearchedLine(1,:);
                        else
                            SearchedVect2 = SearchedLine(size(SearchedLine,1),:);
                        end
                    end
                    SearchedAngle = atand(-(SearchedVect2(1,2)-SearchedVect1(1,2))/(SearchedVect2(1,1)-SearchedVect1(1,1)));

                    % Find angle differences
                    AngleDiff = abs(CurrentAngle - SearchedAngle);
                    if AngleDiff > 90
                        Angles = [CurrentAngle SearchedAngle];
                        Angles = sort(Angles);
                        Angles(1) = 180+Angles(1);
                        AngleDiff = abs(Angles(1) - Angles(2)); 
                    end

                    % Store angles and codes
                    AngleMatrix{statsCell{ii,3}} = [AngleMatrix{statsCell{ii,3}};double(OverlapID(j)) AngleDiff];
                    AngleMatrix{statsCell{ii,3}} = sortrows(unique(AngleMatrix{statsCell{ii,3}},'rows'),2);
                    AngleMatrix{OverlapID(j)} = [AngleMatrix{OverlapID(j)};double(statsCell{ii,3}) AngleDiff];
                    AngleMatrix{OverlapID(j)} = sortrows(unique(AngleMatrix{OverlapID(j)},'rows'),2);
                end
            end

            if 0
            % Draw lines to be joined
            stats2 = regionprops(BWLabeled,BWLabeled,'MaxIntensity','Centroid');
            figure; imshow(BWSingled); hold on;
            for ii = 1:size(statsCell,1)
                 %for j = 1:size(AngleMatrix{ii},1)
                 for j = 1:size(AngleMatrix{ii},1)
                     if j < 3
                         if AngleMatrix{ii}(j,2) <= AngleThreshold
                             plot([stats2(ii).Centroid(1) stats2(AngleMatrix{ii}(j,1)).Centroid(1)],[stats2(ii).Centroid(2) stats2(AngleMatrix{ii}(j,1)).Centroid(2)],'y','LineWidth',1);
                         end
                     end
                 end
            end
            end

            % Join lines
            % Use 3 variables: BWLabeled to get ID, statsCell to get the pixelsID
            % and AngleMatrix to find out which lines to join
            % Make new stats where the pixelID are joined. Second column is
            % original while fourth column contains pixels after smoothing. The
            % smoothing needs to be redone considering newly joined pixels

            % First find out all pairs of lines to be joined
            PairedPoints = zeros(size(statsCell,1)*2,2);
            for ii = 1:size(statsCell,1)
                for j = 1:size(AngleMatrix{ii},1)
                    if j < 3
                        if AngleMatrix{ii}(j,2) <= AngleThreshold
                            PairedPoints(2*(ii-1)+j,:) = [ii AngleMatrix{ii}(j,1)];
                        end
                    end
                end
            end

            % Remove zeros in PairedPoints
            PairedPoints = PairedPoints(PairedPoints(:,1)>0,:);
            % Make the smaller index for PairedPoints to be on the left
            PairedPoints = sort(PairedPoints,2);
            % Sort based on the smaller index
            PairedPoints = sortrows(PairedPoints,[1 2]);
            % Get unique values
            PairedPoints = unique(PairedPoints,'rows');

            % Connect all PairedPoints and come out with lines of connections
            ConnectedPoints = cell(length(PairedPoints),1);
            ConnectedRow = 1;
            SearchIndex = 1;
            while ~isempty(PairedPoints)
                % Get the first pair of points for the ConnectedRow
                ConnectedPoints{ConnectedRow} = PairedPoints(1,:);
                PairedPoints(1,:) = [];     % Delete first column of PairedPoints already recorded
                while SearchIndex <= length(ConnectedPoints{ConnectedRow}) && ~isempty(PairedPoints)
                    % Search through first row of remaining PairedPoints to see if
                    % there is any similar to value searched in SearchIndex. Record if found and
                    % find unique values
                    if ~isempty(PairedPoints)
                        FindInd = find(PairedPoints(:,1)==ConnectedPoints{ConnectedRow}(SearchIndex));
                        if ~isempty(FindInd)
                            ConnectedPoints{ConnectedRow} = unique([ConnectedPoints{ConnectedRow} reshape(PairedPoints(FindInd,:),1,[])],'stable');
                            PairedPoints(FindInd,:) = [];     % Delete first column of PairedPoints already recorded
                        end
                    end

                    if ~isempty(PairedPoints)
                        % Repeat for second row of remaining PairedPoints
                        FindInd = find(PairedPoints(:,2)==ConnectedPoints{ConnectedRow}(SearchIndex));
                        if ~isempty(FindInd)
                            ConnectedPoints{ConnectedRow} = unique([ConnectedPoints{ConnectedRow} reshape(PairedPoints(FindInd,:),1,[])],'stable');
                            PairedPoints(FindInd,:) = [];     % Delete first column of PairedPoints already recorded
                        end
                    end

                    SearchIndex = SearchIndex + 1;
                end
                ConnectedPoints{ConnectedRow} = sort(ConnectedPoints{ConnectedRow});
                ConnectedRow = ConnectedRow + 1;
                SearchIndex = 1;
            end

            % Delete empty cells in ConnectedPoints
            ConnectedPoints = ConnectedPoints(~cellfun('isempty',ConnectedPoints));

            % Sort statsCell based on indices on third column
            statsCell = sortrows(statsCell,3);

            % Combine pixels (column 2) and get new area (column 1) for statsCell
            for j = 1:length(ConnectedPoints)
                FindInd = find([statsCell{:,3}]==ConnectedPoints{j}(1));
                for jj = 2:length(ConnectedPoints{j})
                    FindInd2 = find([statsCell{:,3}]==ConnectedPoints{j}(jj));
                    statsCell{FindInd,1} = statsCell{FindInd,1} + statsCell{FindInd2,1};
                    statsCell{FindInd,2} = sortrows([statsCell{FindInd,2};statsCell{FindInd2,2}],[1 2]);
                    statsCell(FindInd2,:) = [];
                end
            end
            statsCell = sortrows(statsCell,3);

            % Come out with IndexChange Matrix to construct new sets of indexes
            % to prepare for new stats and BWLabeled
            IndexChangeMat = zeros(size(stats,1),3);
            IndexChangeMat(:,1:2) = [(1:size(stats,1))' (1:size(stats,1))'];
            for j = 1:length(ConnectedPoints)
                IndexChangeMat(ConnectedPoints{j}(2:end),2)=ConnectedPoints{j}(1);
            end
            ChangeInd = unique(IndexChangeMat(:,2));
            for j = 1:length(ChangeInd)
                IndexChangeMat(IndexChangeMat(:,2)==ChangeInd(j),3)=j;
                statsCell{j,3}=j;       % Change indexes in third column of statsCell
            end

            % Construct stats structure from statsCell
            rowHeadings = {'Area','PixelList','MaxIntensity'};
            stats = cell2struct(statsCell(:,1:3),rowHeadings,2);

            % Change labels for BWLabeled
            for j = 1:max(max(BWLabeled))
                BWLabeled(BWLabeled==j)=IndexChangeMat(j,3);
            end
        end
        BinaryMatrix = ZeroMatrix;
        for ii = 1:size(stats,1)
            linearInd = sub2ind(size(BinaryMatrix),stats(ii).PixelList(:,2),stats(ii).PixelList(:,1));
            BinaryMatrix(linearInd) = 1;
        end
        dataBW(:,:,i) = BinaryMatrix;
        statsAll(i).stats = stats;
end

        if mod(i,10)==0
            disp(['Frame ' num2str(i) '/' num2str(num_images) ' fitted. Elapsed time: ' num2str(floor(toc)) ' s']);
        end
    end
    
    % Plot microtubule lengths and MicCoordinates
    figure;
    plot(MicLengths,'-m','LineWidth',2); hold on;
    plot(Frames,MicCoordinatesLengths,'*k','MarkerSize',10,'LineWidth',2);
    legend({'Auto Fit','Manual Fit'});
    xlabel('Frame');
    ylabel('Microtubule Length (nm)');
    title('Choose a few datapoints to fit to a spline');
    set(gca,'FontSize',12,'FontWeight','b');
    
    dcm_obj=datacursormode;
    h = msgbox('Click OK only when all cursors are chosen','Choose Cursors');
    set(h, 'Position', [80 500 170 60]);
    uiwait(h);
    Cursor=getCursorInfo(dcm_obj);
    Cursor = sortrows(reshape([Cursor.Position],2,[])',1);
    x = Cursor(:,1);
    xq = 1:num_images;
%     if size(Cursor,1) >=4
%         InterpPoints = [interp1(x,Cursor(:,1),xq,'spline','extrap')' interp1(x,Cursor(:,2),xq,'spline','extrap')'];
%     else
%         InterpPoints = [interp1(x,Cursor(:,1),xq,'linear','extrap')' interp1(x,Cursor(:,2),xq,'linear','extrap')'];
%     end
    InterpPoints = [interp1(x,Cursor(:,1),xq,'linear','extrap')' interp1(x,Cursor(:,2),xq,'linear','extrap')'];
    InterpPoints(InterpPoints(:,2)<0,2) = 0;
    dcm_obj.removeAllDataCursors();
    plot(InterpPoints(:,1),InterpPoints(:,2),'-b','LineWidth',2);

    % Save MicLengths and AllMicThere in EXCEL
    MicLengths = num2cell(InterpPoints(:,2)/1000);      % Assign MicLengths and change unit from nm to um
    AllMicThere = num2cell(AllMicThere);
    SaveMatrix = {'Microtubule Length (um)','All Mic. Present?'};
    SaveMatrix = [SaveMatrix; [MicLengths, AllMicThere]];
    
    % Write TimePoints to EXCEL file
    xlswrite([FileInitial ' (Mic. Length and Presence).xlsx'],SaveMatrix);
    
    % Find saveastiff
    DataPath = pwd;
    % Generate warning and ask for code path if we cannot find code in the active
    % folder and a few folder above
    if isempty(dir('saveastiff.m'))
        cd ..;      % Go to one folder up and look for SICstepFinder.m there   
        if isempty(dir('saveastiff.m'))
            cd ..;      % Go to one folder up and look for SICstepFinder.m there
            if isempty(dir('saveastiff.m'))
                cd ..       % Go to one folder up and look for SICstepFinder.m there
                if isempty(dir('saveastiff.m'))
                    % Generate warning and ask for the code path
                    h=warndlg('Cannot find saveastiff.m in active folder. Locate folder with saveastiff.m in the next window');
                    uiwait(h);
                    CodePath = uigetdir(DataPath,'Select folder containing saveastiff');
                    cd(CodePath);
                else; CodePath = pwd; 
                end;
            else; CodePath = pwd; 
            end;
        else; CodePath = pwd; 
        end;
    else; CodePath = pwd; 
    end;
    
    % Save dataBW
    cd(DataPath);
    if exist('Image Files','dir')~=7; mkdir('Image Files'); end
    cd(CodePath);
    if exist([DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif'],'file') > 0
        delete([DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif']);
        saveastiff(dataBW,[DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif']);
    else
        saveastiff(dataBW,[DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif']);
    end
    cd(DataPath);    
    
    % Display progress
    disp('Done getting microtubule length. Proceed to get hinder/ drive DistanceForces at all TimePoints...');
        %% (Repeat) Plot microtubule lengths and MicCoordinates
        figure;
        MicLengths = InterpPoints(:,2);
        plot(MicLengths,'-m','LineWidth',2); hold on;
        plot(Frames,MicCoordinatesLengths,'*k','MarkerSize',10,'LineWidth',2);
        legend({'Auto Fit','Manual Fit'});
        xlabel('Frame');
        ylabel('Microtubule Length (nm)');
        title('Choose a few datapoints to fit to a spline');
        set(gca,'FontSize',12,'FontWeight','b');

        dcm_obj=datacursormode;
        uiwait(msgbox('Click OK only when all cursors are chosen','Choose Cursors'));
        Cursor=getCursorInfo(dcm_obj);
        Cursor = sortrows(reshape([Cursor.Position],2,[])',1);
        x = Cursor(:,1);
        xq = 1:num_images;
        InterpPoints = [interp1(x,Cursor(:,1),xq,'linear','extrap')' interp1(x,Cursor(:,2),xq,'linear','extrap')'];
%         if size(Cursor,1) >=4
%             InterpPoints = [interp1(x,Cursor(:,1),xq,'spline','extrap')' interp1(x,Cursor(:,2),xq,'spline','extrap')'];
%         else
%             InterpPoints = [interp1(x,Cursor(:,1),xq,'linear','extrap')' interp1(x,Cursor(:,2),xq,'linear','extrap')'];
%         end
        InterpPoints(InterpPoints(:,2)<0,2) = 0;
        dcm_obj.removeAllDataCursors();
        plot(InterpPoints(:,1),InterpPoints(:,2),'-b','LineWidth',2);

        % Save MicLengths and AllMicThere in EXCEL
        MicLengths = num2cell(InterpPoints(:,2)/1000);      % Assign MicLengths and change unit from nm to um
        if ~iscell(AllMicThere)
            AllMicThere = num2cell(AllMicThere);
        end
        
        SaveMatrix = {'Microtubule Length (um)','All Mic. Present?'};
        SaveMatrix = [SaveMatrix; [MicLengths, AllMicThere]];

        % Write TimePoints to EXCEL file
        xlswrite([FileInitial ' (Mic. Length and Presence).xlsx'],SaveMatrix);

        % Find saveastiff
        DataPath = pwd;
        % Generate warning and ask for code path if we cannot find code in the active
        % folder and a few folder above
        if isempty(dir('saveastiff.m'))
            cd ..;      % Go to one folder up and look for SICstepFinder.m there   
            if isempty(dir('saveastiff.m'))
                cd ..;      % Go to one folder up and look for SICstepFinder.m there
                if isempty(dir('saveastiff.m'))
                    cd ..       % Go to one folder up and look for SICstepFinder.m there
                    if isempty(dir('saveastiff.m'))
                        % Generate warning and ask for the code path
                        h=warndlg('Cannot find saveastiff.m in active folder. Locate folder with saveastiff.m in the next window');
                        uiwait(h);
                        CodePath = uigetdir(DataPath,'Select folder containing saveastiff');
                        cd(CodePath);
                    else; CodePath = pwd; 
                    end;
                else; CodePath = pwd; 
                end;
            else; CodePath = pwd; 
            end;
        else; CodePath = pwd; 
        end;

        % Save dataBW
        cd(DataPath);
        if exist('Image Files','dir')~=7; mkdir('Image Files'); end
        cd(CodePath);
        if exist([DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif'],'file') > 0
            delete([DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif']);
            saveastiff(dataBW,[DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif']);
        else
            saveastiff(dataBW,[DataPath '\Image Files\' FileInitial ' (Mic-Binary).tif']);
        end
        cd(DataPath);    

        % Display progress
        disp('Done getting microtubule length. Proceed to extract duration, mean and maximum forces...');
    %% 4) Get hinder/ drive DistanceForces at all TimePoints
    % Import FinalResult3
    FileName = dir('*(Final Result3).xlsx');
    FileName = FileName.name;
    FinalResult3 = xlsread(FileName);
    FileInitial = strrep(FileName,' (Final Result3).xlsx','');
    
    % Get PixelSize and ExposureTime
    PixelSize = xlsread([FileInitial ' (Variables).xlsx']);
    ExposureTime = PixelSize(2);    % Exposure time in sec
    PixelSize = PixelSize(1);       % PixelSize in nm
    
    RegionNo = round((size(FinalResult3,2)-4)/7);
    FinalResultLength = size(FinalResult3,1);
    
    % Extract DistanceForces
    DistanceForces = zeros(FinalResultLength,RegionNo*2);
    for i = 1:RegionNo
        DistanceForces(:,i) = FinalResult3(:,7+7*(i-1));
        DistanceForces(:,RegionNo+i) = FinalResult3(:,10+7*(i-1));
    end
    
    % Get frame indexes and total frames for all microtubule and kinesins to initialize AlignMatrix
    FrameInd = cell(RegionNo+1,1);
    FrameInd{1} = nonzeros(uint16(FinalResult3(:,1)/ExposureTime));
    MaxFrame = max(FrameInd{1});
    for i = 1:RegionNo
        FrameInd{i+1} = nonzeros(uint16(FinalResult3(:,4+7*(i-1))/ExposureTime));
        MaxFrame = max(MaxFrame,max(FrameInd{i+1}));
    end

    % Initialize matrix to align microtubule velocity and DistanceForces
    AlignedMatrix = NaN(MaxFrame,RegionNo+1);
    AlignedMatrix(FrameInd{1},1) = FinalResult3(~isnan(FinalResult3(:,3)),3);
    AlignedMatrix(1:min(FrameInd{1})-1,1) = FinalResult3(1,3);   % Fill the first few values that is NaN with initial microtubule velocity
    for i = 1:RegionNo
        DistanceForce = sqrt(DistanceForces(:,i).^2+DistanceForces(:,RegionNo+i).^2).*sign(DistanceForces(:,i));
        DistanceForce = DistanceForce(~isnan(DistanceForce));
        DistanceForce(DistanceForce==0) = NaN;
        AlignedMatrix(FrameInd{i+1},i+1) = DistanceForce;
    end
    
    % Add frame column
    AlignedMatrix = [double(1:MaxFrame)' AlignedMatrix];
    
    % Delete values with all NaN for DistanceForces
    DeleteIndex = min(isnan(AlignedMatrix(:,3:end)),[],2);
    AlignedMatrix(DeleteIndex,:) = [];
    
    % Initialize TimePoints matrix
    TimePointsLength = size(AlignedMatrix,1);
    TimePoints = num2cell(NaN(TimePointsLength,RegionNo+8));
    TimePoints(:,1) = cellstr(FileInitial);     % Assign name
    TimePoints(:,7) = num2cell(AlignedMatrix(:,2));     % Assign microtubule velocity
    SortedForce = sort(AlignedMatrix(:,3:end),2);
    TimePoints(:,9:9+RegionNo-1) = num2cell(SortedForce);       % Assign all the driving and hindering DistanceForces in sorted order from driving to hindering
    SortedForce = sign(SortedForce);
    
    % Find out how many is driving and hindering
    DriveNo = SortedForce;
    HinderNo = SortedForce;
    DriveNo(DriveNo~=-1) = 0;
    HinderNo(HinderNo~=1) = 0;
    DriveNo = abs(sum(DriveNo,2));
    HinderNo = sum(HinderNo,2);
    TimePoints(:,2) = num2cell(DriveNo+HinderNo);       % Total kinesin driving/ Hindering
    TimePoints(:,3) = num2cell(DriveNo);       % Number of kinesin driving
    TimePoints(:,4) = num2cell(HinderNo);       % Number of kinesin hindering

    % Get microtubule lengths and presence
    MicLengthsPresence = xlsread([FileInitial ' (Mic. Length and Presence).xlsx']);
    TimePoints(:,5:6) = num2cell(MicLengthsPresence(AlignedMatrix(:,1),:));
    
    % Get number of kinesin points
    FileName = dir('*(KinTrackMateSpots).csv');
    FileName = FileName.name;
    Input = xlsread(FileName,'J:J');
    KinPerFrame = size(Input,1)/(max(Input)-min(Input));
    
    % Get area of tif file
    FileName = dir('*(Kin).tif');
    FileName = FileName.name;
    KinInputInfo = imfinfo(FileName);
    Area = KinInputInfo(1).Width*KinInputInfo(1).Height*(PixelSize/1000)^2;        % Area in um^2
    
    % Get kinesin density in number of kinesin per pixel^2
    KinDensity = KinPerFrame/Area;
    
    % Assign into TimePoints
    TimePoints(:,8) = num2cell(KinDensity);

    % Prepare cells for EXCEL spreadsheet
    FirstLine = cell(1,RegionNo+8);
    FirstLine(1,1:8) = {'File ID','Total Driving/ Hindering','Driving No.','Hindering No.','Microtubule Length (um)','Entire Mic Present?','Microtubule Vel. (nm/s)','Kinesin Density (Count/um^2)'};
    
    for i = 1:RegionNo
        % Define titles of each column
        FirstLine(1,8+i) = {['Point ' num2str(i) ' Distance Force (nm)']};
    end
    
    TimePoints = [FirstLine; TimePoints];

    % Write TimePoints to EXCEL file
    xlswrite([FileInitial ' (TimePoints).xlsx'],TimePoints);
    
    % Display progress
    disp('Done getting DistanceForces for all TimePoints...');
    
    % Close all figures and clear all variables
    fclose all;
    close all;
    clear all;    
        %% Close all figures and clear all variables
        fclose all;
        close all;
        clear all;
    %% Recompile from FinalResult3 and smooth velocity
    % Specify VariableOutput to save DurDistForces, MicVelocity and TimePoints
%     VariableOutput = 'Variables (All 5 Conc)';
    close all; clear all;
    VariableOutput = 'Variables (all) SmoothVel';
    
    if exist(['Aggregate Images\' VariableOutput '.mat'],'file')==2
        load(['Aggregate Images\' VariableOutput '.mat'])
    else
        % Get a list of all files and folders in this folder.
        FolderNames = dir('*)*');
        FolderNames = {FolderNames.name};
        MainFolder = pwd;

        % First find the sizes for DurationFile and TimePoints
        DurationFileLength = 0;
        TimePointsLength = 0;
        TimePointsWidth = 0;
        SubFoldersLength = 0;
        for i = 1:length(FolderNames)
            cd(FolderNames{i});
            subFolderNames = dir;
            dirFlags = [subFolderNames.isdir];
            subFolderNames = subFolderNames(dirFlags);
            subFolderNames = {subFolderNames.name};
            subFolderNames(1:2) = [];
            subFolderPath = pwd;

            for ii = 1: length(subFolderNames)
                cd(subFolderNames{ii});
                DurationFile = dir('*(Dur-DistForces).xlsx');
                if ~isempty(DurationFile)
                    DurationFile = DurationFile.name;
                    Input = xlsread(DurationFile);
                    DurationFileLength = DurationFileLength + size(Input,1);
                    SubFoldersLength = SubFoldersLength + 1;
                end

                TimePointsFile = dir('*(TimePoints).xlsx');
                if ~isempty(TimePointsFile)
                    TimePointsFile = TimePointsFile.name;
                    Input = xlsread(TimePointsFile);
                    TimePointsLength = TimePointsLength + size(Input,1);
                    TimePointsWidth = max(TimePointsWidth,size(Input,2));
                end

                % Go back to subFolderPath
                cd(subFolderPath);
            end

            % Go back to MainFolder
            cd(MainFolder);
        end

        % Initialize DurDistForces and TimePoints
        DurDistForces = NaN(DurationFileLength,8);
        MicVelocity = NaN(DurationFileLength,1);
        TimePoints = NaN(TimePointsLength,TimePointsWidth);
        TimePointsFile = cell(TimePointsLength,3);
        Displacements = cell(SubFoldersLength,3);
        DurationCount = 1;
        TimePointsCount = 1;
        SubFoldersCount = 1;

        % Loop to extract DurDistForces, MicVelocity and TimePoints
        for i = 1:length(FolderNames)
            cd(FolderNames{i});
            subFolderNames = dir('*)*');
%             subFolderNames = dir;
            dirFlags = [subFolderNames.isdir];
            subFolderNames = subFolderNames(dirFlags);
            subFolderNames = {subFolderNames.name};
            % Only get folders that have bracket next to initial number, and
            % prepare to sort folders based on number
            indexSubFolder = zeros(length(subFolderNames),1);
            for ii = 1:length(subFolderNames)
                % Find the strings before right bracket ')'
                StringBeforeBracket = strtok(subFolderNames{ii},')');
                if min(isstrprop(StringBeforeBracket,'digit')) == 1
                    indexSubFolder(ii) = str2num(StringBeforeBracket);
                end
            end
            % Remove folders with non-numeric string next to right bracket ')'
            % and sort folders
            subFolderNames(indexSubFolder==0)=[];
            indexSubFolder(indexSubFolder==0)=[];
            [indexSubFolder,SortOrder]=sort(indexSubFolder);
            subFolderNames = subFolderNames(SortOrder);
            % Record subfolder path
            subFolderPath = pwd;
            clear('ExposureTime');

            % Loop subfolders
            for ii = 1:length(subFolderNames)
                display(['Extracting data from ' num2str(i) '-' num2str(ii)]);
                cd(subFolderNames{ii});
                DurationFile = dir('*(Dur-DistForces).xlsx');
                FileName = dir('*(Final Result3).xlsx');
                
                if ~isempty(DurationFile) && ~isempty(DurationFile)
                    FileName = FileName.name;
                    FinalResult3 = xlsread(FileName);
                    FileInitial = strrep(FileName,' (Final Result3).xlsx','');

                    % Smoothing velocity
    %                 figure; plot(FinalResult3(~isnan(FinalResult3(:,3)),3),'b'); hold on;
    %                 plot(smooth(FinalResult3(~isnan(FinalResult3(:,3)),3),0.1,'loess'),'r');
                    FinalResult3(~isnan(FinalResult3(:,3)),3) = smooth(FinalResult3(~isnan(FinalResult3(:,3)),3),0.1,'loess');

                    % Get PixelSize and ExposureTime
                    PixelSize = xlsread([FileInitial ' (Variables).xlsx']);
                    ExposureTime = PixelSize(2);    % Exposure time in sec
                    PixelSize = PixelSize(1);       % PixelSize in nm

                    RegionNo = round((size(FinalResult3,2)-4)/7);
                    FinalResultLength = size(FinalResult3,1);

                    % Extract DistanceForces
                    DistanceForces = zeros(FinalResultLength,RegionNo*2);
                    for j = 1:RegionNo
                        DistanceForces(:,j) = FinalResult3(:,7+7*(j-1));
                        DistanceForces(:,RegionNo+j) = FinalResult3(:,10+7*(j-1));
                    end

                    % Get frame indexes and total frames for all microtubule and kinesins to initialize AlignMatrix
                    FrameInd = cell(RegionNo+1,1);
                    FrameInd{1} = nonzeros(uint16(FinalResult3(:,1)/ExposureTime));
                    MaxFrame = max(FrameInd{1});
                    for j = 1:RegionNo
                        FrameInd{j+1} = nonzeros(uint16(FinalResult3(:,4+7*(j-1))/ExposureTime));
                        MaxFrame = max(MaxFrame,max(FrameInd{j+1}));
                    end

                    % Initialize matrix to align microtubule velocity and DistanceForces
                    AlignedMatrix = NaN(MaxFrame,RegionNo+1);
                    AlignedMatrix(FrameInd{1},1) = FinalResult3(~isnan(FinalResult3(:,3)),3);
                    AlignedMatrix(1:min(FrameInd{1})-1,1) = FinalResult3(1,3);   % Fill the first few values that is NaN with initial microtubule velocity
                    for j = 1:RegionNo
                        DistanceForce = sqrt(DistanceForces(:,j).^2+DistanceForces(:,RegionNo+j).^2).*sign(DistanceForces(:,j));
                        DistanceForce = DistanceForce(~isnan(DistanceForce));
                        DistanceForce(DistanceForce==0) = NaN;
                        AlignedMatrix(FrameInd{j+1},j+1) = DistanceForce;
                    end

                    % Add frame column
                    AlignedMatrix = [double(1:MaxFrame)' AlignedMatrix];

                    % Delete values with all NaN for DistanceForces
                    DeleteIndex = min(isnan(AlignedMatrix(:,3:end)),[],2);
                    AlignedMatrix(DeleteIndex,:) = [];

                    % Initialize TimePoints matrix
                    TimePointsLength = size(AlignedMatrix,1);
                    TimePointsIn = num2cell(NaN(TimePointsLength,RegionNo+8));
                    TimePointsIn(:,1) = cellstr(FileInitial);     % Assign name
                    TimePointsIn(:,7) = num2cell(AlignedMatrix(:,2));     % Assign microtubule velocity
                    SortedForce = sort(AlignedMatrix(:,3:end),2);
                    TimePointsIn(:,9:9+RegionNo-1) = num2cell(SortedForce);       % Assign all the driving and hindering DistanceForces in sorted order from driving to hindering
                    SortedForce = sign(SortedForce);

                    % Find out how many is driving and hindering
                    DriveNo = SortedForce;
                    HinderNo = SortedForce;
                    DriveNo(DriveNo~=-1) = 0;
                    HinderNo(HinderNo~=1) = 0;
                    DriveNo = abs(sum(DriveNo,2));
                    HinderNo = sum(HinderNo,2);
                    TimePointsIn(:,2) = num2cell(DriveNo+HinderNo);       % Total kinesin driving/ Hindering
                    TimePointsIn(:,3) = num2cell(DriveNo);       % Number of kinesin driving
                    TimePointsIn(:,4) = num2cell(HinderNo);       % Number of kinesin hindering

                    % Get microtubule lengths and presence
                    MicLengthsPresence = xlsread([FileInitial ' (Mic. Length and Presence).xlsx']);
                    TimePointsIn(:,5:6) = num2cell(MicLengthsPresence(AlignedMatrix(:,1),:));

                    % Get number of kinesin points
                    FileName = dir('*(KinTrackMateSpots).csv');
                    FileName = FileName.name;
                    Input = xlsread(FileName,'J:J');
                    KinPerFrame = size(Input,1)/(max(Input)-min(Input));

                    % Get area of tif file
                    FileName = dir('*(Kin).tif');
                    FileName = FileName.name;
                    KinInputInfo = imfinfo(FileName);
                    Area = KinInputInfo(1).Width*KinInputInfo(1).Height*(PixelSize/1000)^2;        % Area in um^2

                    % Get kinesin density in number of kinesin per pixel^2
                    KinDensity = KinPerFrame/Area;

                    % Assign into TimePointsIn
                    TimePointsIn(:,8) = num2cell(KinDensity);

                    % Convert TimePointsIn from cell to matrix
                    TimePointsIn = cell2mat(TimePointsIn(:,2:end));
                    TimePoints(TimePointsCount:TimePointsCount+size(TimePointsIn,1)-1,1:size(TimePointsIn,2)) = TimePointsIn;
                    TimePointsFile(TimePointsCount:TimePointsCount+size(TimePointsIn,1)-1,1:2) = [num2cell(ones(size(TimePointsIn,1),1)*i) num2cell(ones(size(TimePointsIn,1),1)*indexSubFolder(ii))];
                    TimePointsFile(TimePointsCount:TimePointsCount+size(TimePointsIn,1)-1,3) = {subFolderNames{ii}};
                    
                    TimePointsCount = TimePointsCount + size(TimePointsIn,1);

                    DurationFile = DurationFile.name;
                    Input = xlsread(DurationFile);
                    DurDistForces(DurationCount:DurationCount+size(Input,1)-1,:) = [ones(size(Input,1),1)*i ones(size(Input,1),1)*indexSubFolder(ii) Input];

%                     FinalResultFile = FinalResultFile.name;
%                     FinalResult = xlsread(FinalResultFile);
%                     KinNo = (size(FinalResult,2)-4)/7;
                    FinalResult = FinalResult3(:,1:3);
                    FinalResult = FinalResult(~isnan(FinalResult(:,1)),:);
                    Displacements(SubFoldersCount,:) = {subFolderNames{ii} RegionNo max(FinalResult(:,2))};

                    if ~exist('ExposureTime','var')
                        ExposureTime = min(diff(FinalResult(:,1)));
                    end

                    % Interpolate time from ExposureTime to include all
                    % time-points used in DurDistForces
                    xInterp = ExposureTime:ExposureTime:max(Input(:,3))+20*ExposureTime;
%                     MicVel = interp1(FinalResult(:,1),FinalResult(:,3),xInterp','linear','extrap');
                    MicVel = NaN(round(max(Input(:,3))/ExposureTime+20),1);
                    MicVel(round(FinalResult(:,1)/ExposureTime)) = FinalResult(:,3);

                    % Get the microtubule velocity corresponding to each row of
                    % the DurDistForces variable
                    MicVelocity(DurationCount:DurationCount+size(Input,1)-1,1) = MicVel(round(Input(:,3)/ExposureTime));
                    DurationCount = DurationCount+size(Input,1);
                    SubFoldersCount = SubFoldersCount + 1;

    %                 TimePointsFile = dir('*(TimePoints).xlsx');
    %                 if ~isempty(TimePointsFile)
    %                     TimePointsFile = TimePointsFile.name;
    %                     Input = xlsread(TimePointsFile);
    %                     TimePoints(TimePointsCount:TimePointsCount+size(Input,1)-1,1:size(Input,2)) = Input;
    %                     TimePointsCount = TimePointsCount + size(Input,1);
    %                 end
                end
                % Go back to subFolderPath
                cd(subFolderPath);
            end

            % Go back to MainFolder
            cd(MainFolder);
        end

        % Delete DurDistForces which have NaN values
%         DurDistForces(isnan(DurDistForces(:,1)),:) = [];
%         MicVelocity(isnan(MicVelocity(:,1)),:) = [];
        DurDistForces(DurationCount:end,:) = [];
        MicVelocity(DurationCount:end,:) = [];
        
        % Sort DurDistForces and MicVelocity
        [DurDistForces,DurDistForcesInd] = sortrows(DurDistForces,[1 2 3 4 5]);
        MicVelocity = MicVelocity(DurDistForcesInd);

        % Delete TimePoints which have NaN values
        TimePointsFile(isnan(TimePoints(:,1)),:) = [];
        TimePoints(isnan(TimePoints(:,1)),:) = [];
        
        % Delete Displacements with empty cells
        Displacements = Displacements(find(~cellfun(@isempty,Displacements(:,1))),:);

        % Save DurDistForces, TimePoints, MicVelocity and Displacements
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        save(['Aggregate Images\' VariableOutput '.mat'],'DurDistForces','MicVelocity','TimePoints','TimePointsFile','Displacements','ExposureTime');
    end
    
    % TimePoints: 
    % 1) Total Driving/ Hindering, 2)Driving No., 3) Hindering No.
    % 4) Microtubule Length (um), 5) Entire Mic Present?, 6) Microtubule Vel. (nm/s)
    % 7) Kinesin Density (Count/um^2), 8) Point 1 Distance Force (nm),
    % 9) Point 2 Distance Force (nm), etc
    
    % DurDistForces:
    % 1) Folder number, 2) Sub-folder number
    % 3) RegionIndex, 4) LineIndex, 5) Time
    % 6) On-axis distance force, 7) Off-axis distance force
    % 8) Total distance force
    
    % Get ContactArea, TotalKin, DrivingKin and HinderingKin
    ContactArea = TimePoints(:,4).*TimePoints(:,7);
    TotalKin = TimePoints(:,1);
    DrivingKin = TimePoints(:,2);
    HinderingKin = TimePoints(:,3);
    DriveHinderDiff = DrivingKin - HinderingKin;
    
    % Find out first and last indexes for each drive/hinder DistanceForce
    [~,FirstInd,~] = unique(DurDistForces(:,[1 2 4]),'rows','first');
    [~,LastInd,~] = unique(DurDistForces(:,[1 2 4]),'rows','last');
    
    % Delete input with NaN values
    DeleteIndexes = false(length(MicVelocity),1);
    for i = 1:length(FirstInd)
        Indexes = FirstInd(i):LastInd(i);
        if sum(isnan(MicVelocity(Indexes)))
            DeleteIndexes(Indexes) = true;
        end
    end
    
    DurDistForces(DeleteIndexes,:) = [];
    MicVelocity(DeleteIndexes) = [];
    
    % Repeat to find first and last indexes for each drive/hinder DistanceForce
    [~,FirstInd,~] = unique(DurDistForces(:,[1 2 4]),'rows','first');
    [~,LastInd,~] = unique(DurDistForces(:,[1 2 4]),'rows','last');

    % Calculate durations and MeanDistForces for all drive/hinder
    AllDuration = DurDistForces(LastInd,5)-DurDistForces(FirstInd,5);
    MeanDistForces = zeros(length(FirstInd),1);
    MaxDistForces = zeros(length(FirstInd),1);      % Calculate max force of smoothed DistForces
    MeanRunLength = zeros(length(FirstInd),1);      % Mean run length in um
    DistForceDuration = zeros(length(FirstInd),1);
    for i = 1:length(MeanDistForces)
%         MeanDistForces(i) = mean(DurDistForces(FirstInd(i):LastInd(i),8));
%         SmoothedDistForces = smooth(DurDistForces(FirstInd(i):LastInd(i),8));
        MeanDistForces(i) = mean(DurDistForces(FirstInd(i):LastInd(i),6));
        SmoothedDistForces = smooth(DurDistForces(FirstInd(i):LastInd(i),6));
        [MaxDistForces(i),MaxDistForceInd] = max(abs(SmoothedDistForces));
        MaxDistForces(i) = sign(SmoothedDistForces(MaxDistForceInd))*MaxDistForces(i);
        TimeInterval = DurDistForces(FirstInd(i):LastInd(i),5);
        TimeInterval = TimeInterval(2:end) - TimeInterval(1:end-1);
        MeanRunLength(i) = sum(TimeInterval.*MicVelocity(FirstInd(i)+1:LastInd(i)))/1000;
        DistForceDuration(i) = DurDistForces(LastInd(i),5) - DurDistForces(FirstInd(i),5);
    end
    DriveDuration = AllDuration(MeanDistForces<0);
    HinderDuration = AllDuration(MeanDistForces>0);
    DriveRunLength = MeanRunLength(MeanDistForces<0);
    HinderRunLength = MeanRunLength(MeanDistForces>0);
    
    % Get all drive/ hinder DistanceForces
%     DriveDistForces = DurDistForces(DurDistForces(:,8)<0,8);
%     HinderDistForces = DurDistForces(DurDistForces(:,8)>0,8);
    DriveDistForces = DurDistForces(DurDistForces(:,6)<0,6);
    HinderDistForces = DurDistForces(DurDistForces(:,6)>0,6);
    DriveMaxDistForces = MaxDistForces(MaxDistForces<0);
    HinderMaxDistForces = MaxDistForces(MaxDistForces>0);
    
    % Classify each trace into drives, hinders and mixed. Plot Mode Fraction
    % Find out first and last indexes for each trace
    [~,TraceFirstInd,~] = unique(DurDistForces(:,[1 2 3]),'rows','first');
    [~,TraceLastInd,~] = unique(DurDistForces(:,[1 2 3]),'rows','last');

    % Initialize DriveMixHinder, MeanDriveDuration and MeanHinderDuration Array
    DriveMixHinder = zeros(length(TraceFirstInd),1);
    MeanDriveDuration = zeros(length(TraceFirstInd),1);
    MeanHinderDuration = zeros(length(TraceFirstInd),1);
    TransitionStats = zeros(length(TraceFirstInd),4);   % No of drive-drive, hinder-hinder, drive-hinder and hinder-drive for each trace
    TraceTotalDur = zeros(length(TraceFirstInd),1);     % Total duration from the first drive/hinder to last in a particular trace
    
    % Loop to get DriveMixHinder status (-1 if drive, 0 if there is a mix of drive and hinder.
    % 1 if hinder.) for all traces, and TransitionStats.
    for i = 1:length(TraceFirstInd)
        % Find out indexes with drive/ hinder belonging to that trace
        DriveHinderIndex = find(ismember(FirstInd,TraceFirstInd(i))):find(ismember(LastInd,TraceLastInd(i)));
        StartTimes = DurDistForces(FirstInd(DriveHinderIndex),5);
        [~,StartTimesInd] = sort(StartTimes);       % Sort based on start times of traces (may not be in order since we choose drive/ hinders randomly)
        DriveHinderSign = sign(MeanDistForces(DriveHinderIndex));
        DriveHinderSign = DriveHinderSign(StartTimesInd);   % Sort DriveHinderSign based on StartTimesInd
        
        if length(unique(DriveHinderSign)) == 1
            DriveMixHinder(i) = unique(DriveHinderSign);
        end
        
        if length(DriveHinderSign) > 1
            SignChange = DriveHinderSign(2:end) - DriveHinderSign(1:end-1);
            TransitionStats(i,:) = [length(find(SignChange==0 & DriveHinderSign(1:end-1)==-1)) length(find(SignChange==0 & DriveHinderSign(1:end-1)==1)) length(find(SignChange==2)) length(find(SignChange==-2))];
        end
        
        % Get average duration
        TraceDurations = AllDuration(DriveHinderIndex);
        MeanDriveDuration(i) = mean(TraceDurations(DriveHinderSign<0));
        MeanHinderDuration(i) = mean(TraceDurations(DriveHinderSign>0));
        
        % Get TraceTotalDur
        EndTimes = DurDistForces(LastInd(DriveHinderIndex),5);
        TraceTotalDur(i) = max(EndTimes) - min(StartTimes);
    end

    % Remove NaN from MeanDriveDuration and MeanHinderDuration
    MeanDriveDuration = MeanDriveDuration(~isnan(MeanDriveDuration));
    MeanHinderDuration = MeanHinderDuration(~isnan(MeanHinderDuration));
    
    % Convert DistanceForces to Force
    DistanceOffset = 20;        % DistanceOffset in nm, representing the additional length of DNA
    % 10 nm SA-QD/2, 5 nm Anti-Dig, 10 nm SA, 25 nm PEG
    AdjustedDriveDistance = abs(abs(DriveDistForces) - DistanceOffset);
    AdjustedHinderDistance = abs(abs(HinderDistForces) - DistanceOffset);
    AdjustedDriveMaxDistForces = abs(abs(DriveMaxDistForces) - DistanceOffset);
    AdjustedHinderMaxDistForces = abs(abs(HinderMaxDistForces) - DistanceOffset);
%     L0 = 516.5;   % Contour length in nm
    L0 = 532.1;   % Contour length in nm
%     kT = 4.114; % Boltzmann constant time temperature in pN nm
%     P = 45;     % Persistence length in nm

    % Make AdjustedDriveDistance and AdjustedHinderDistance above FractionCutOff*L0 to be FractionCutOff*L0
%     FractionCutOff = 0.968;
    FractionCutOff = 1;
%     RemovalCutoff = 526;    % Absolute distance above which forces will be removed
%     RemovalCutoff = 600;    % Absolute distance above which forces will be removed
    RemovalCutoff = 2000;    % Absolute distance above which forces will be removed
    DriveDistForces(AdjustedDriveDistance>RemovalCutoff) = [];
    HinderDistForces(AdjustedHinderDistance>RemovalCutoff) = [];
    AdjustedDriveDistance(AdjustedDriveDistance>RemovalCutoff) = [];
    AdjustedHinderDistance(AdjustedHinderDistance>RemovalCutoff) = [];
    AdjustedDriveDistance(AdjustedDriveDistance>FractionCutOff*L0)=FractionCutOff*L0;
    AdjustedHinderDistance(AdjustedHinderDistance>FractionCutOff*L0)=FractionCutOff*L0;
    DriveMaxDistForces(AdjustedDriveMaxDistForces>RemovalCutoff) = [];
    HinderMaxDistForces(AdjustedHinderMaxDistForces>RemovalCutoff) = [];
    AdjustedDriveMaxDistForces(AdjustedDriveMaxDistForces>RemovalCutoff)=[];
    AdjustedHinderMaxDistForces(AdjustedHinderMaxDistForces>RemovalCutoff)=[];
    AdjustedDriveMaxDistForces(AdjustedDriveMaxDistForces>FractionCutOff*L0)=FractionCutOff*L0;
    AdjustedHinderMaxDistForces(AdjustedHinderMaxDistForces>FractionCutOff*L0)=FractionCutOff*L0;

%     % Calculate DriveForce and HinderForce
%     DriveForces = (1/4*(1-(AdjustedDriveDistance/L0)).^(-2)-1/4+(AdjustedDriveDistance/L0)-0.8*(AdjustedDriveDistance/L0).^2.15)*kT/P;
%     HinderForces = (1/4*(1-(AdjustedHinderDistance/L0)).^(-2)-1/4+(AdjustedHinderDistance/L0)-0.8*(AdjustedHinderDistance/L0).^2.15)*kT/P;
%     MaxDriveForces = (1/4*(1-(AdjustedDriveMaxDistForces/L0)).^(-2)-1/4+(AdjustedDriveMaxDistForces/L0)-0.8*(AdjustedDriveMaxDistForces/L0).^2.15)*kT/P;
%     MaxHinderForces = (1/4*(1-(AdjustedHinderMaxDistForces/L0)).^(-2)-1/4+(AdjustedHinderMaxDistForces/L0)-0.8*(AdjustedHinderMaxDistForces/L0).^2.15)*kT/P;

    DriveForces = zeros(length(AdjustedDriveDistance),1);
    HinderForces = zeros(length(AdjustedHinderDistance),1);
    MaxDriveForces = zeros(length(AdjustedDriveMaxDistForces),1);
    MaxHinderForces = zeros(length(AdjustedHinderMaxDistForces),1);
    % Calculate DriveForce and HinderForce using XWLC
    for i = 1:length(AdjustedDriveDistance)
        DriveForces(i) = fzero(@(x)(AdjustedDriveDistance(i)/L0-XWLCContour(x)),1);
    end
    for i = 1:length(AdjustedHinderDistance)
        HinderForces(i) = fzero(@(x)(AdjustedHinderDistance(i)/L0-XWLCContour(x)),1);
    end
    for i = 1:length(AdjustedDriveMaxDistForces)
        MaxDriveForces(i) = fzero(@(x)(AdjustedDriveMaxDistForces(i)/L0-XWLCContour(x)),1);
    end
    for i = 1:length(AdjustedHinderMaxDistForces)
        MaxHinderForces(i) = fzero(@(x)(AdjustedHinderMaxDistForces(i)/L0-XWLCContour(x)),1);
    end
        %% (Repeat) Plot Duration and DistanceForce, Mode Fraction, and No. of Drive & Hinder vs concentration 
        % Specify VariableOutput to save DurDistForces, MicVelocity and TimePoints
        VariableOutput = 'Variables (100 nM)';

        if exist(['Aggregate Images\' VariableOutput '.mat'],'file')==2
            load(['Aggregate Images\' VariableOutput '.mat'])
        else
            % Get a list of all files and folders in this folder.
            FolderNames = dir('*)*');
            FolderNames = {FolderNames.name};
            MainFolder = pwd;

            % First find the sizes for DurationFile and TimePoints
            DurationFileLength = 0;
            TimePointsLength = 0;
            TimePointsWidth = 0;
            SubFoldersLength = 0;
            for i = 1:length(FolderNames)
                cd(FolderNames{i});
                subFolderNames = dir;
                dirFlags = [subFolderNames.isdir];
                subFolderNames = subFolderNames(dirFlags);
                subFolderNames = {subFolderNames.name};
                subFolderNames(1:2) = [];
                subFolderPath = pwd;

                for ii = 1: length(subFolderNames)
                    cd(subFolderNames{ii});
                    DurationFile = dir('*(Dur-DistForces).xlsx');
                    if ~isempty(DurationFile)
                        DurationFile = DurationFile.name;
                        Input = xlsread(DurationFile);
                        DurationFileLength = DurationFileLength + size(Input,1);
                        SubFoldersLength = SubFoldersLength + 1;
                    end

                    TimePointsFile = dir('*(TimePoints).xlsx');
                    if ~isempty(TimePointsFile)
                        TimePointsFile = TimePointsFile.name;
                        Input = xlsread(TimePointsFile);
                        TimePointsLength = TimePointsLength + size(Input,1);
                        TimePointsWidth = max(TimePointsWidth,size(Input,2));
                    end

                    % Go back to subFolderPath
                    cd(subFolderPath);
                end

                % Go back to MainFolder
                cd(MainFolder);
            end

            % Initialize DurDistForces and TimePoints
            DurDistForces = NaN(DurationFileLength,8);
            MicVelocity = NaN(DurationFileLength,1);
            TimePoints = NaN(TimePointsLength,TimePointsWidth);
            Displacements = cell(SubFoldersLength,3);
            DurationCount = 1;
            TimePointsCount = 1;
            SubFoldersCount = 1;

            % Loop to extract DurDistForces, MicVelocity and TimePoints
            for i = 1:length(FolderNames)
                cd(FolderNames{i});
                subFolderNames = dir('*)*');
    %             subFolderNames = dir;
                dirFlags = [subFolderNames.isdir];
                subFolderNames = subFolderNames(dirFlags);
                subFolderNames = {subFolderNames.name};
                % Only get folders that have bracket next to initial number, and
                % prepare to sort folders based on number
                indexSubFolder = zeros(length(subFolderNames),1);
                for ii = 1:length(subFolderNames)
                    % Find the strings before right bracket ')'
                    StringBeforeBracket = strtok(subFolderNames{ii},')');
                    if min(isstrprop(StringBeforeBracket,'digit')) == 1
                        indexSubFolder(ii) = str2num(StringBeforeBracket);
                    end
                end
                % Remove folders with non-numeric string next to right bracket ')'
                % and sort folders
                subFolderNames(indexSubFolder==0)=[];
                indexSubFolder(indexSubFolder==0)=[];
                [indexSubFolder,SortOrder]=sort(indexSubFolder);
                subFolderNames = subFolderNames(SortOrder);
                % Record subfolder path
                subFolderPath = pwd;
                clear('ExposureTime');

                for ii = 1:length(subFolderNames)
                    display(['Extracting data from ' num2str(i) '-' num2str(ii)]);
                    cd(subFolderNames{ii});
                    DurationFile = dir('*(Dur-DistForces).xlsx');
                    FinalResultFile = dir('*(Final Result3).xlsx');

                    if ~isempty(DurationFile)
                        DurationFile = DurationFile.name;
                        Input = xlsread(DurationFile);
                        DurDistForces(DurationCount:DurationCount+size(Input,1)-1,:) = [ones(size(Input,1),1)*i ones(size(Input,1),1)*indexSubFolder(ii) Input];

                        FinalResultFile = FinalResultFile.name;
                        FinalResult = xlsread(FinalResultFile);
                        KinNo = (size(FinalResult,2)-4)/7;
                        FinalResult = FinalResult(:,1:3);
                        FinalResult = FinalResult(~isnan(FinalResult(:,1)),:);
                        Displacements(SubFoldersCount,:) = {subFolderNames{ii} KinNo max(FinalResult(:,2))};

                        if ~exist('ExposureTime','var')
                            ExposureTime = min(diff(FinalResult(:,1)));
                        end

                        % Interpolate time from ExposureTime to include all
                        % time-points used in DurDistForces
                        xInterp = ExposureTime:ExposureTime:max(Input(:,3))+20*ExposureTime;
                        MicVel = interp1(FinalResult(:,1),FinalResult(:,3),xInterp','linear','extrap');

                        % Get the microtubule velocity corresponding to each row of
                        % the DurDistForces variable
                        MicVelocity(DurationCount:DurationCount+size(Input,1)-1,1) = MicVel(round(Input(:,3)/ExposureTime));
                        DurationCount = DurationCount+size(Input,1);
                        SubFoldersCount = SubFoldersCount + 1;
                    end

                    TimePointsFile = dir('*(TimePoints).xlsx');
                    if ~isempty(TimePointsFile)
                        TimePointsFile = TimePointsFile.name;
                        Input = xlsread(TimePointsFile);
                        TimePoints(TimePointsCount:TimePointsCount+size(Input,1)-1,1:size(Input,2)) = Input;
                        TimePointsCount = TimePointsCount + size(Input,1);
                    end

                    % Go back to subFolderPath
                    cd(subFolderPath);
                end

                % Go back to MainFolder
                cd(MainFolder);
            end

            % Delete DurDistForces which have NaN values
            DurDistForces(isnan(DurDistForces(:,1)),:) = [];
            MicVelocity(isnan(MicVelocity(:,1)),:) = [];

            % Delete TimePoints which have NaN values
            TimePoints(isnan(TimePoints(:,1)),:) = [];

            % Delete Displacements with empty cells
            Displacements = Displacements(find(~cellfun(@isempty,Displacements(:,1))),:);

            % Save DurDistForces, TimePoints, MicVelocity and Displacements
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            save(['Aggregate Images\' VariableOutput],'DurDistForces','MicVelocity','TimePoints','Displacements');
        end

        % Get ContactArea, TotalKin, DrivingKin and HinderingKin
        ContactArea = TimePoints(:,4).*TimePoints(:,7);
        TotalKin = TimePoints(:,1);
        DrivingKin = TimePoints(:,2);
        HinderingKin = TimePoints(:,3);
        DriveHinderDiff = DrivingKin - HinderingKin;

        % Find out first and last indexes for each drive/hinder DistanceForce
        [~,FirstInd,~] = unique(DurDistForces(:,[1 2 4]),'rows','first');
        [~,LastInd,~] = unique(DurDistForces(:,[1 2 4]),'rows','last');

        % Calculate durations and MeanDistForces for all drive/hinder
        AllDuration = DurDistForces(LastInd,5)-DurDistForces(FirstInd,5);
        MeanDistForces = zeros(length(FirstInd),1);
        MeanRunLength = zeros(length(FirstInd),1);      % Mean run length in um
        for i = 1:length(MeanDistForces)
%             MeanDistForces(i) = mean(DurDistForces(FirstInd(i):LastInd(i),8));
            MeanDistForces(i) = mean(DurDistForces(FirstInd(i):LastInd(i),6));
            TimeInterval = DurDistForces(FirstInd(i):LastInd(i),5);
            TimeInterval = TimeInterval(2:end) - TimeInterval(1:end-1);
            MeanRunLength(i) = sum(TimeInterval.*MicVelocity(FirstInd(i)+1:LastInd(i)))/1000;
        end
        DriveDuration = AllDuration(MeanDistForces<0);
        HinderDuration = AllDuration(MeanDistForces>0);

        % Get all drive/ hinder DistanceForces
%         DriveDistForces = DurDistForces(DurDistForces(:,8)<0,8);
%         HinderDistForces = DurDistForces(DurDistForces(:,8)>0,8);
        DriveDistForces = DurDistForces(DurDistForces(:,6)<0,6);
        HinderDistForces = DurDistForces(DurDistForces(:,6)>0,6);

        % Classify each trace into drives, hinders and mixed. Plot Mode Fraction
        % Find out first and last indexes for each trace
        [~,TraceFirstInd,~] = unique(DurDistForces(:,[1 2 3]),'rows','first');
        [~,TraceLastInd,~] = unique(DurDistForces(:,[1 2 3]),'rows','last');

        % Initialize DriveMixHinder, MeanDriveDuration and MeanHinderDuration Array
        DriveMixHinder = zeros(length(TraceFirstInd),1);
        MeanDriveDuration = zeros(length(TraceFirstInd),1);
        MeanHinderDuration = zeros(length(TraceFirstInd),1);

        % Loop to get DriveMixHinder status for all traces. -1 if drive, 0 if
        % there is a mix of drive and hinder. 1 if hinder.
        for i = 1:length(TraceFirstInd)
            % Find out indexes with drive/ hinder belonging to that trace
            DriveHinderIndex = find(ismember(FirstInd,TraceFirstInd(i))):find(ismember(LastInd,TraceLastInd(i)));
            DriveHinderSign = sign(MeanDistForces(DriveHinderIndex));
            if length(unique(DriveHinderSign)) == 1
                DriveMixHinder(i) = unique(DriveHinderSign);
            end

            % Get average duration
            TraceDurations = AllDuration(DriveHinderIndex);
            MeanDriveDuration(i) = mean(TraceDurations(DriveHinderSign<0));
            MeanHinderDuration(i) = mean(TraceDurations(DriveHinderSign>0));
        end

        % Remove NaN from MeanDriveDuration and MeanHinderDuration
        MeanDriveDuration = MeanDriveDuration(~isnan(MeanDriveDuration));
        MeanHinderDuration = MeanHinderDuration(~isnan(MeanHinderDuration));

        % Convert DistanceForces to Force
        DistanceOffset = 25;        % DistanceOffset in nm, representing the additional length of DNA
        % 10 nm SA-QD/2, 10 nm Anti-Dig, 5 nm SA, 5 nm PEG
        AdjustedDriveDistance = abs(abs(DriveDistForces) - DistanceOffset);
        AdjustedHinderDistance = abs(abs(HinderDistForces) - DistanceOffset);
        L0 = 530;   % Contour length in nm
        kT = 4.114; % Boltzmann constant time temperature in pN nm
        P = 45;     % Persistence length in nm

        % Make AdjustedDriveDistance and AdjustedHinderDistance above 0.98*L0 to be 0.98*L0
        FractionCutOff = 0.965;
        AdjustedDriveDistance(AdjustedDriveDistance>FractionCutOff*L0)=FractionCutOff*L0;
        AdjustedHinderDistance(AdjustedHinderDistance>FractionCutOff*L0)=FractionCutOff*L0;

        % Calculate DriveForce and HinderForce
        DriveForces = (1/4*(1-(AdjustedDriveDistance/L0)).^(-2)-1/4+(AdjustedDriveDistance/L0)-0.8*(AdjustedDriveDistance/L0).^2.15)*kT/P;
        HinderForces = (1/4*(1-(AdjustedHinderDistance/L0)).^(-2)-1/4+(AdjustedHinderDistance/L0)-0.8*(AdjustedHinderDistance/L0).^2.15)*kT/P;
        %% (Repeat) Convert DistanceForces to Force
        DistanceOffset = 25;        % DistanceOffset in nm, representing the additional length of DNA
        % 10 nm SA-QD/2, 10 nm Anti-Dig, 5 nm SA, 5 nm PEG
        AdjustedDriveDistance = abs(abs(DriveDistForces) - DistanceOffset);
        AdjustedHinderDistance = abs(abs(HinderDistForces) - DistanceOffset);
        L0 = 530;   % Contour length in nm
        kT = 4.114; % Boltzmann constant time temperature in pN nm
        P = 45;     % Persistence length in nm

        % Make AdjustedDriveDistance and AdjustedHinderDistance above FractionCutOff*L0 to be FractionCutOff*L0
        FractionCutOff = 0.945;
        AdjustedDriveDistance(AdjustedDriveDistance>FractionCutOff*L0)=FractionCutOff*L0;
        AdjustedHinderDistance(AdjustedHinderDistance>FractionCutOff*L0)=FractionCutOff*L0;

        % Calculate DriveForce and HinderForce (from Duration files)
        DriveForces = (1/4*(1-(AdjustedDriveDistance/L0)).^(-2)-1/4+(AdjustedDriveDistance/L0)-0.8*(AdjustedDriveDistance/L0).^2.15)*kT/P;
        HinderForces = (1/4*(1-(AdjustedHinderDistance/L0)).^(-2)-1/4+(AdjustedHinderDistance/L0)-0.8*(AdjustedHinderDistance/L0).^2.15)*kT/P;
        
        % Get Forces (from TimePoints file)
        Forces = TimePoints(:,8:size(TimePoints,2));
        ForcesSign = sign(Forces);
        Forces = abs(abs(Forces) - DistanceOffset);
        Forces(Forces>FractionCutOff*L0)=FractionCutOff*L0;
        Forces(Forces<-FractionCutOff*L0)=-FractionCutOff*L0;
        Forces = (1/4*(1-(Forces/L0)).^(-2)-1/4+(Forces/L0)-0.8*(Forces/L0).^2.15)*kT/P.*ForcesSign;
        %% a) Classify each trace into drives, hinders and mixed. Plot Mode Fraction
        SampleNo = length(DriveMixHinder);
        CI = zeros(3,1);
        for i = -1:1
            Probability = length(find(DriveMixHinder==i))/SampleNo;
            CI(i+2) = 1.96*sqrt(Probability*(1-Probability)/SampleNo);  % z = 1.96 for 95% CI. z is the normalized standard deviation. 95% is the percent area under the normalized graph
        end
        % Source to calculate confidence interval for binomial distribution:
        % http://davidmlane.com/hyperstat/B9168.html
        % Ours is multinomial, but according to this forum, for a single
        % variable it is the same as binomial: https://stats.stackexchange.com/questions/175756/multinomial-proportion-confidence-interval
        % Good Youtube explanation: https://www.youtube.com/watch?v=3ReWri_jh3M
        % Or https://www.youtube.com/watch?v=W_M1mH5Tqys

        % Plot histograms for DriveMixHinder
        C = categorical(DriveMixHinder,[-1 0 1],{'Drive','Mixed','Hinder'});
        figure;
        h = histogram(C,'BarWidth',0.5,'Normalization','probability','BarWidth',0.9); hold on;
        errorbar(1:3,h.Values,CI,'LineStyle','none','Color','k');
        % Create text objects at each location
        text(1:3,h.Values+CI',num2str(round(h.Values'*100)/100),'FontSize',18','FontWeight','b','horizontalalignment','center','verticalalignment','bottom');
        text(0.8,0.85,['N = ' num2str(length(DriveMixHinder))],'FontSize',18','FontWeight','b','horizontalalignment','center','verticalalignment','bottom','Units','Normalized');
        set(gca,'FontSize',18','FontWeight','b','TickDir','out');
        ylabel('Fraction');
        title('Fraction of Traces in Various Modes');
        ylim([0 0.7]);

        % Save figures for Duration Force diagram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Fraction Drive-Mixed-Hinder.jpg');
        savefig('Aggregate Images\Fraction Drive-Mixed-Hinder.fig');
        
        % Find out fraction of time kinesin is hindering
        SumDriveDuration = sum(DriveDuration);
        SumHinderDuration = sum(HinderDuration);
        TimeFractionOfHinder = SumHinderDuration/(SumDriveDuration+SumHinderDuration);
        figure;
        bar(TimeFractionOfHinder);
        xlim([0 2]);
        ylim([0 1]);
        title('Fraction of Time Hindering');
        ylabel('Fraction');
        text(0.92,TimeFractionOfHinder+0.05,num2str(round(TimeFractionOfHinder*100)/100),'FontSize',16,'fontweight','b');
        set(gca,'FontSize',16,'FontWeight','b');
        
        % Save figures for Fraction of Time Hindering
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\FractionOfTimeHindering.jpg');
        savefig('Aggregate Images\FractionOfTimeHindering.fig');
        %% b) Plot Average Transitions
        OverallTransitionProb = sum(TransitionStats,1);
        AverageTransition = mean(TransitionStats,1);
        StdTransition = std(TransitionStats,[],1);
        SEMTransition = StdTransition/sqrt(size(TransitionStats,1));
        OverallTransitionProb = OverallTransitionProb/sum(OverallTransitionProb);
        
        if 0
        % Plot OverallTransitionProb
        figure; bar(1:4,OverallTransitionProb,'c');
        ylabel('Transition Probability');
        title('Transition Probabilities');
        xticklabels({'Drive-Drive','Hinder-Hinder','Drive-Hinder','Hinder-Drive'})
        xtickangle(30);
        text((1:length(OverallTransitionProb))',(OverallTransitionProb+0.05)',num2str(round(OverallTransitionProb*100)'/100),'FontSize',14,'FontWeight','b','HorizontalAlignment','center');
        set(gca,'FontSize',14,'FontWeight','b');
        ylim([0 0.6]);
        
        % Save figures for OverallTransitionProb
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\OverallTransitionProb.jpg');
        savefig('Aggregate Images\OverallTransitionProb.fig');
        end
        
        % Plot AverageTransition
        figure; bar(AverageTransition,'c'); hold on;
        errorbar(1:4,AverageTransition,SEMTransition/2,'.','color','black');
        ylabel('Average Number of Transitions');
        title('Average Transition/ Trace');
        xticklabels({'Drive-Drive','Hinder-Hinder','Drive-Hinder','Hinder-Drive'})
        xtickangle(30);
        text((1:length(AverageTransition))',(AverageTransition+0.2)',cellfun(@(x,y) {x,['(' y '%)']},cellstr(num2str(round(AverageTransition*100)'/100)),cellstr(num2str(round(OverallTransitionProb'*100))),'UniformOutput',false),'FontSize',14,'FontWeight','b','HorizontalAlignment','center');
        text(0.8,0.85,['N = ' num2str(length(TransitionStats))],'FontSize',18','FontWeight','b','horizontalalignment','center','verticalalignment','bottom','Units','Normalized');
        set(gca,'FontSize',14,'FontWeight','b');
        ylim([0 1.6]);
        
        % Save figures for AverageTransition
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\AverageTransition.jpg');
        savefig('Aggregate Images\AverageTransition.fig');
            %% Plot Average Transitions normalized by TraceTotalDur
            % Convert TraceTotalDur from second to min
            TraceTotalDurMin = TraceTotalDur/60;
            NormalizedTransitionStats = TransitionStats./TraceTotalDurMin;  % Number of transition per minute
            OverallTransitionProb = sum(NormalizedTransitionStats,1);
            AverageTransition = mean(NormalizedTransitionStats,1);
            StdTransition = std(NormalizedTransitionStats,[],1);
            SEMTransition = StdTransition/sqrt(size(NormalizedTransitionStats,1));
            OverallTransitionProb = OverallTransitionProb/sum(OverallTransitionProb);

            if 0
            % Plot OverallTransitionProb
            figure; bar(1:4,OverallTransitionProb,'c');
            ylabel('Transition Probability');
            title('Transition Probabilities');
            xticklabels({'Drive-Drive','Hinder-Hinder','Drive-Hinder','Hinder-Drive'})
            xtickangle(30);
            text((1:length(OverallTransitionProb))',(OverallTransitionProb+0.05)',num2str(round(OverallTransitionProb*100)'/100),'FontSize',14,'FontWeight','b','HorizontalAlignment','center');
            set(gca,'FontSize',14,'FontWeight','b');
            ylim([0 0.6]);

            % Save figures for OverallTransitionProb
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\OverallTransitionProb.jpg');
            savefig('Aggregate Images\OverallTransitionProb.fig');
            end

            % Plot AverageTransition
            figure; bar(AverageTransition,'FaceColor',[0.5 0.5 1]); hold on;
            errorbar(1:4,AverageTransition,SEMTransition/2,'.','color','black');
            ylabel('Average No. of Transitions/ min');
            title('Average Transition/ min');
            xticklabels({'Drive-Drive','Hinder-Hinder','Drive-Hinder','Hinder-Drive'})
            xtickangle(30);
            text((1:length(AverageTransition))',(AverageTransition+max(AverageTransition)*0.15)',cellfun(@(x,y) {x,['(' y '%)']},cellstr(num2str(round(AverageTransition*100)'/100)),cellstr(num2str(round(OverallTransitionProb'*100))),'UniformOutput',false),'FontSize',14,'FontWeight','b','HorizontalAlignment','center');
            text(0.8,0.85,['N = ' num2str(length(NormalizedTransitionStats))],'FontSize',18','FontWeight','b','horizontalalignment','center','verticalalignment','bottom','Units','Normalized');
            set(gca,'FontSize',14,'FontWeight','b');
            ylim([0 max(AverageTransition)*1.3]);

            % Save figures for AverageTransition
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\AverageTransition(Normalized).jpg');
            savefig('Aggregate Images\AverageTransition(Normalized).fig');
            
            % Save variables ready in EXCEL
            TransitionType = {'Transition Type';'Drive-Drive';'Hinder-Hinder';'Drive-Hinder';'Hinder-Drive'};
            AverageTransitionEXCEL = [{'Average No. of Transition/ min'};num2cell(AverageTransition')];
            ErrorTransitionEXCEL = [{'Transition SEM (No./ min)'};num2cell(SEMTransition')];
            SampleNo = [{'Sample No.'}; num2cell(length(NormalizedTransitionStats)); {'';'';''}];
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            xlswrite('Aggregate Images\NormalizedAverageTransition.xlsx',[TransitionType,AverageTransitionEXCEL,ErrorTransitionEXCEL,SampleNo]);
        %% c) Plot histograms of Drive and Hinder Duration, Run Length and DistanceForces
        % Note: we calculate the mean duration for a trace to get a single
        % datapoint here. So one trace only give one hinder/ drive/ both
        % datapoint for duration
        DurationBinSpacing = 0.8;
        xText = 0.22;
        yText = 0.75;
        FontSize = 12;

        figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
        ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
        axes(ha(1));
        edges = 0:DurationBinSpacing:max(DriveDuration);
%             centers = edges(2:end)-DurationBinSpacing/2;
        histogram(DriveDuration,edges,'FaceColor',[0 0 0.4]); hold on;
        stdev = std(DriveDuration);
        sem = stdev/sqrt(length(DriveDuration));
        text(xText,yText,{['Mean: ' num2str(round(mean(DriveDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
            ['Std = ' num2str(round(stdev*10)/10) ' s'],...
            ['N = ' num2str(length(DriveDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
        % Write titles and axes labels
        title('Drive/ Hinder Duration ');
        set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
        ylabel('Drive Count'); xlabel('');
        xlim([0 35]);

        axes(ha(2));
        edges = 0:DurationBinSpacing:max(HinderDuration);
        centers = edges(2:end)-DurationBinSpacing/2;
        Hist = histogram(HinderDuration,edges,'FaceColor',[1 0.4 0]); hold on;
        set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
        stdev = std(HinderDuration);
        sem = stdev/sqrt(length(HinderDuration));
        text(xText,yText,{['Mean: ' num2str(round(mean(HinderDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
            ['Std = ' num2str(round(stdev*10)/10) ' s'],...
            ['N = ' num2str(length(HinderDuration))]},'FontSize',12,'fontweight','b','Units','normalized');

        % Write titles and axes labels
        a = gca; set(a,'box','off','color','none');
        b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
        axes(a); linkaxes([a b]);
        ylabel('Hinder Count');
        xlim([0 35]);
        xlabel('Time (s)');

        % Save figures for Duration Force diagram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Duration(By Mode).jpg');
        savefig('Aggregate Images\Duration(By Mode).fig');
        
        
        % Find out drive and hinder run lengths
        RunLengthBinSpacing = 1;
        xText = 0.2;
        yText = 0.75;
        FontSize = 12;

        figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
        ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
        axes(ha(1));
        edges = 0:RunLengthBinSpacing:max(DriveRunLength);
%         centers = edges(2:end)-RunLengthBinSpacing/2;
        histogram(DriveRunLength,edges,'FaceColor',[0 0 0.4]); hold on;
        stdev = std(DriveRunLength);
        sem = stdev/sqrt(length(DriveRunLength));
        text(xText,yText,{['Mean: ' num2str(round(mean(DriveRunLength)*10)/10) '  ' num2str(round(sem*10)/10) ' \mum'],...
            ['Std = ' num2str(round(stdev*10)/10) ' s'],...
            ['N = ' num2str(length(DriveRunLength))]},'FontSize',12,'fontweight','b','Units','normalized');
        % Write titles and axes labels
        title('Drive/ Hinder Run Length');
        set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
        ylabel('Drive Count'); xlabel('');
        xlim([0 25]);

        axes(ha(2));
        edges = 0:RunLengthBinSpacing:max(HinderRunLength);
%         centers = edges(2:end)-DurationBinSpacing/2;
        histogram(HinderRunLength,edges,'FaceColor',[1 0.4 0]); hold on;
        set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
        stdev = std(HinderRunLength);
        sem = stdev/sqrt(length(HinderRunLength));
        text(xText,yText,{['Mean: ' num2str(round(mean(HinderRunLength)*10)/10) '  ' num2str(round(sem*10)/10) ' \mum'],...
            ['Std = ' num2str(round(stdev*10)/10) ' s'],...
            ['N = ' num2str(length(HinderRunLength))]},'FontSize',12,'fontweight','b','Units','normalized');
        % Write titles and axes labels
        a = gca; set(a,'box','off','color','none');
        b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
        axes(a); linkaxes([a b]);
        ylabel('Hinder Count');
        xlim([0 25]);
        xlabel('Run Length (?m)');

        % Save figures for Duration Force diagram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\RunLength(By Mode).jpg');
        savefig('Aggregate Images\RunLength(By Mode).fig');        
        
        
        % Plot histograms of Drive and Hinder DistanceForces
        % Note: we calculate the mean duration for a trace to get a single
        % datapoint here. So one trace only give one hinder/ drive/ both
        % datapoint for duration
        DistForceBinSpacing = 20;
        xText = 0.05;
        yText = 0.8;
        FontSize = 12;

        figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
        ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);

        axes(ha(1));
    %     subplot(2,2,2);
        edges = 0:DistForceBinSpacing:max(abs(DriveDistForces));
        hist=histogram(abs(DriveDistForces),edges,'FaceColor','r');
        title('Distance Force');
        stdev = std(DriveDistForces);
        sem = stdev/sqrt(length(DriveDistForces));
        text(xText,yText,{['Mean: ' num2str(abs(round(mean(DriveDistForces)))) '  ' num2str(round(sem)) ' nm'],...
            ['Std = ' num2str(round(stdev)) ' nm'],...
            ['N = ' num2str(length(DriveDistForces))]},'FontSize',FontSize,'fontweight','b','Units','normalized');
        set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[]);
        ylabel('Drive Count'); xlabel('');
        xlim([-10 780]);
        ylim([0 max(hist.Values)*1.6]);


        axes(ha(2));
        edges = 0:DistForceBinSpacing:max(HinderDistForces);
        hist=histogram(HinderDistForces,edges,'FaceColor',[0.9 0.9 0]);
        stdev = std(HinderDistForces);
        sem = stdev/sqrt(length(HinderDistForces));
        text(xText,yText,{['Mean: ' num2str(abs(round(mean(HinderDistForces)))) '  ' num2str(round(sem)) ' nm'],...
            ['Std = ' num2str(round(stdev)) ' nm'],...
            ['N = ' num2str(length(HinderDistForces))]},'FontSize',FontSize,'fontweight','b','Units','normalized');
        a = gca; set(a,'box','off','color','none');
        b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
        axes(a); linkaxes([a b]);
        set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
        xlim([-10 780]);
        ylim([0 max(hist.Values)*1.6]);
        ylabel('Drive Count');
        xlabel('Distance Force (nm)');

        % Save figures for Distance Force diagram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\DistanceForce(By Mode).jpg');
        savefig('Aggregate Images\DistanceForce(By Mode).fig');
            %% Plot histograms of DriveDuration and HinderDuration and plot exponential
            BinSpacingRange = 0.1:0.05:1.0;
            ExpFitPointsRemovedRange = 1:3;
            ConfidenceInterval = 0.95;
            
            % Prepare to optimize binning and point removed for drive duration
            gofs = zeros(length(BinSpacingRange),length(ExpFitPointsRemovedRange));
            for i = 1:length(BinSpacingRange)
                for j = 1:length(ExpFitPointsRemovedRange)
                    edges = 0:BinSpacingRange(i):max(DriveDuration);
                    centers = edges(2:end)-BinSpacingRange(i)/2;
                    Hist = histcounts(DriveDuration,edges);
                    % Fit exponential
                    [~,gof1] = fit(centers(ExpFitPointsRemovedRange(j)+1:end)',Hist(ExpFitPointsRemovedRange(j)+1:end)','exp1','StartPoint', [max(DriveDuration)*2 -1/mean(DriveDuration)]);
                    
                    edges = 0:BinSpacingRange(i):max(HinderDuration);
                    centers = edges(2:end)-BinSpacingRange(i)/2;
                    Hist = histcounts(HinderDuration,edges);
                    % Fit exponential
                    [~,gof2] = fit(centers(ExpFitPointsRemovedRange(j)+1:end)',Hist(ExpFitPointsRemovedRange(j)+1:end)','exp1','StartPoint', [max(HinderDuration)*2 -1/mean(HinderDuration)]);
                    gofs(i,j) = gof1.rsquare + gof2.rsquare;
                end
            end
            [~,I] = max(gofs(:));
            [I1,I2] = ind2sub(size(gofs),I);

            BinSpacing = BinSpacingRange(I1);
            ExpFitPointsRemoved = ExpFitPointsRemovedRange(I2);     % No of points at the start to remove before fitting exponential
            xText = 0.22;
            yText = 0.75;
            FontSize = 12;

            figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
            ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
            axes(ha(1));
            edges = 0:BinSpacing:max(DriveDuration);
            centers = edges(2:end)-BinSpacing/2;
            Hist = histogram(DriveDuration,edges,'FaceColor',[0 0 0.4]); hold on;
            % Fit exponential
            [f,gof] = fit(centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)','exp1','StartPoint', [max(DriveDuration)*2 -1/mean(DriveDuration)]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Dwell: ' num2str(round(mean(CIMean)*100)/100) '  ' num2str(round(Error*100)/100) ' s'],...
                ['N = ' num2str(length(DriveDuration))],['R^2 = ' num2str(round(gof.rsquare*1000)/1000)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            title('Drive/ Hinder Duration ');
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
            ylabel('Drive Count'); xlabel('');
            xlim([0 35]);
            
            % Get variables ready for EXCEL
            EXCELTitle = {'Drive/Hinder','Duration Mean (s)','Duration Error (s)','Confidence Interval','Sample No.'};
            DriveValues = {'Drive',CIMean,Error,ConfidenceInterval,length(DriveDuration)};

            axes(ha(2));
            edges = 0:BinSpacing:max(HinderDuration);
            centers = edges(2:end)-BinSpacing/2;
            Hist = histogram(HinderDuration,edges,'FaceColor',[1 0.4 0]); hold on;
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
            % Fit exponential
            [f,gof] = fit(centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)','exp1','StartPoint', [max(HinderDuration)*2 -1/mean(HinderDuration)]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Dwell: ' num2str(round(mean(CIMean)*100)/100) '  ' num2str(round(Error*100)/100) ' s'],...
                ['N = ' num2str(length(HinderDuration))],['R^2 = ' num2str(round(gof.rsquare*1000)/1000)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            ylabel('Hinder Count');
            xlim([0 35]);
            xlabel('Time (s)');

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\DurationExpFit(By Mode).jpg');
            savefig('Aggregate Images\DurationExpFit(By Mode).fig');
            
            % Save duration in EXCEL
            HinderValues = {'Hinder',CIMean,Error,ConfidenceInterval,length(HinderDuration)};
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            xlswrite('Aggregate Images\DurationExpFit.xlsx',[EXCELTitle;DriveValues;HinderValues]);
            %% Plot histograms of Drive and Hinder run lengths and plot exponential
            RunLengthBinSpacingRange = 0.1:0.05:1.4;
            ExpFitPointsRemovedRange = 1:3;
            ConfidenceInterval = 0.95;
            
            % Prepare to optimize binning and point removed for drive run length
            gofs = zeros(length(RunLengthBinSpacingRange),length(ExpFitPointsRemovedRange));
            for i = 1:length(RunLengthBinSpacingRange)
                for j = 1:length(ExpFitPointsRemovedRange)
                    edges = 0:RunLengthBinSpacingRange(i):max(DriveRunLength);
                    centers = edges(2:end)-RunLengthBinSpacingRange(i)/2;
                    Hist = histcounts(DriveRunLength,edges);
                    % Fit exponential
                    [~,gof1] = fit(centers(ExpFitPointsRemovedRange(j)+1:end)',Hist(ExpFitPointsRemovedRange(j)+1:end)','exp1','StartPoint', [max(DriveRunLength)*2 -1/mean(DriveRunLength)]);
                    
                    edges = 0:RunLengthBinSpacingRange(i):max(HinderRunLength);
                    centers = edges(2:end)-RunLengthBinSpacingRange(i)/2;
                    Hist = histcounts(HinderRunLength,edges);
                    % Fit exponential
                    [~,gof2] = fit(centers(ExpFitPointsRemovedRange(j)+1:end)',Hist(ExpFitPointsRemovedRange(j)+1:end)','exp1','StartPoint', [max(HinderRunLength)*2 -1/mean(HinderRunLength)]);
                    gofs(i,j) = gof1.rsquare + gof2.rsquare;
                end
            end
            [~,I] = max(gofs(:));
            [I1,I2] = ind2sub(size(gofs),I);

            RunLengthBinSpacing = RunLengthBinSpacingRange(I1);
            ExpFitPointsRemoved = ExpFitPointsRemovedRange(I2);     % No of points at the start to remove before fitting exponential
            xText = 0.22;
            yText = 0.75;
            FontSize = 12;

            figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
            ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
            axes(ha(1));
            edges = 0:RunLengthBinSpacing:max(DriveRunLength);
            centers = edges(2:end)-RunLengthBinSpacing/2;
            Hist = histogram(DriveRunLength,edges,'FaceColor',[0 0 0.4]); hold on;
            % Fit exponential
            [f,gof] = fit(centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)','exp1','StartPoint', [max(DriveRunLength)*2 -1/mean(DriveRunLength)]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Run: ' num2str(round(mean(CIMean)*100)/100) '  ' num2str(round(Error*100)/100) ' \mum'],...
                ['N = ' num2str(length(DriveRunLength))],['R^2 = ' num2str(round(gof.rsquare*1000)/1000)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            title('Drive/ Hinder Run Length');
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
            ylabel('Drive Count'); xlabel('');
            xlim([0 25]);
            
            % Get variables ready for EXCEL
            EXCELTitle = {'Drive/Hinder','Run Length Mean (um)','Run Length Error (um)','Confidence Interval','No. of Sample'};
            DriveValues = {'Drive',CIMean,Error,ConfidenceInterval,length(DriveRunLength)};

            axes(ha(2));
            edges = 0:RunLengthBinSpacing:max(HinderRunLength);
            centers = edges(2:end)-RunLengthBinSpacing/2;
            Hist = histogram(HinderRunLength,edges,'FaceColor',[1 0.4 0]); hold on;
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
            % Fit exponential
            [f,gof] = fit(centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)','exp1','StartPoint', [max(HinderRunLength)*2 -1/mean(HinderRunLength)]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Run: ' num2str(round(mean(CIMean)*100)/100) '  ' num2str(round(Error*100)/100) ' \mum'],...
                ['N = ' num2str(length(HinderRunLength))],['R^2 = ' num2str(round(gof.rsquare*1000)/1000)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            ylabel('Hinder Count');
            xlim([0 25]);
            xlabel('Run Length (\mum)');

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\RunLengthExpFit(By Mode).jpg');
            savefig('Aggregate Images\RunLengthExpFit(By Mode).fig');
            
            % Save run length in EXCEL
            HinderValues = {'Hinder',CIMean,Error,ConfidenceInterval,length(HinderRunLength)};
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            xlswrite('Aggregate Images\RunLengthExpFit.xlsx',[EXCELTitle;DriveValues;HinderValues]);
            %% Plot histograms of Drive and Hinder max forces
            % Note: we calculate the mean duration for a trace to get a single
            % datapoint here. So one trace only give one hinder/ drive/ both
            % datapoint for duration
            ForceBinSpacing = 2;
            xText = 0.28;
            yText = 0.78;
            UseLogY = 1;
            FontSize = 14;
            xmax = 25;

            figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
            ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
            axes(ha(1));
            edges = 0:ForceBinSpacing:max(abs(MaxDriveForces));
            histogram(abs(MaxDriveForces),edges,'FaceColor',[0 0 0.4]);
            title('Drive/ Hinder Force');
            stdev = std(MaxDriveForces);
            sem = stdev/sqrt(length(MaxDriveForces));
            text(xText-0.1,yText,{['Mean: ' num2str(abs(round(mean(MaxDriveForces)*100)/100)) '  ' num2str(round(sem*100)/100) ' pN'],...
                ['Std = ' num2str(round(stdev*100)/100) ' nm'],...
                ['N = ' num2str(length(MaxDriveForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[]);
            xlim([0 xmax]);
            if UseLogY == 1
                set(gca,'yscale','log');
            end
            ylabel('Drive Count'); xlabel('');
                
            % Get variables ready for EXCEL
            EXCELTitle = {'Drive/Hinder','Force Mean (pN)','Force Error (pN)','No. of Sample'};
            DriveValues = {'Drive',mean(MaxDriveForces),sem,length(abs(MaxDriveForces))};
            
            axes(ha(2));
            edges = 0:ForceBinSpacing:max(MaxHinderForces);
            histogram(MaxHinderForces,edges,'FaceColor',[1 0.4 0]);
            stdev = std(MaxHinderForces);
            sem = stdev/sqrt(length(MaxHinderForces));
            text(xText-0.1,yText,{['Mean: ' num2str(abs(round(mean(MaxHinderForces)*100)/100)) '  ' num2str(round(sem*100)/100) ' pN'],...
                ['Std = ' num2str(round(stdev*100)/100) ' nm'],...
                ['N = ' num2str(length(MaxHinderForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
            xlim([0 xmax]);
            ylabel('Hinder Count');
            xlabel('Force (pN)');
            if UseLogY == 1
                set(gca,'yscale','log');
            end

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\MaxForce.jpg');
            savefig('Aggregate Images\MaxForce.fig');
            
            % Save force in EXCEL
            HinderValues = {'Hinder',mean(MaxHinderForces),sem,length(abs(MaxHinderForces))};
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            xlswrite('Aggregate Images\ForceMean.xlsx',[EXCELTitle;DriveValues;HinderValues]);
            %% Plot histograms of Drive and Hinder max forces and plot exponential
            % Note: we calculate the mean duration for a trace to get a single
            % datapoint here. So one trace only give one hinder/ drive/ both
            % datapoint for duration
            % Optimizing BinSpacing and ExpFitPointsRemoved
            BinSpacingRange = 0.1:0.05:0.2;
%             ExpFitPointsRemovedRange = 1:3;
            ExpFitPointsRemovedRange = 0:2;
            
            % Set parameters
%             ForceBinSpacing = 0.1;
            xmax = 3;
            
            % Set Confidence Interval
            ConfidenceInterval = 0.95;
            
            % Prepare to optimize binning and point removed for drive duration
            gofs = zeros(length(BinSpacingRange),length(ExpFitPointsRemovedRange));
            for i = 1:length(BinSpacingRange)
                for j = 1:length(ExpFitPointsRemovedRange)
                    edges = 0:BinSpacingRange(i):max(abs(MaxDriveForces));
                    centers = edges(2:end)-BinSpacingRange(i)/2;
                    Hist = histcounts(abs(MaxDriveForces),edges);
                    % Fit exponential
                    [~,gof1] = fit(centers(ExpFitPointsRemovedRange(j)+1:end)',Hist(ExpFitPointsRemovedRange(j)+1:end)','exp1','StartPoint', [max(abs(MaxDriveForces))*2 -1/mean(abs(MaxDriveForces))]);
                    
                    edges = 0:BinSpacingRange(i):max(abs(MaxHinderForces));
                    centers = edges(2:end)-BinSpacingRange(i)/2;
                    Hist = histcounts(abs(MaxHinderForces),edges);
                    % Fit exponential
                    [~,gof2] = fit(centers(ExpFitPointsRemovedRange(j)+1:end)',Hist(ExpFitPointsRemovedRange(j)+1:end)','exp1','StartPoint', [max(abs(MaxHinderForces))*2 -1/mean(abs(MaxHinderForces))]);
                    gofs(i,j) = gof1.rsquare + gof2.rsquare;
                end
            end
            [~,I] = max(gofs(:));
            [I1,I2] = ind2sub(size(gofs),I);

            BinSpacing = BinSpacingRange(I1);
            ExpFitPointsRemoved = ExpFitPointsRemovedRange(I2);     % No of points at the start to remove before fitting exponential
            xText = 0.10;
            yText = 0.75;
            FontSize = 12;

            figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
            ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
            axes(ha(1));
            edges = 0:BinSpacing:max(abs(MaxDriveForces));
            centers = edges(2:end)-BinSpacing/2;
            Hist = histogram(abs(MaxDriveForces),edges,'FaceColor',[0 0 0.4]); hold on;
            % Fit exponential
%             ft = fittype('a*exp(-b*x)+c*exp(-d*x)');
%             [f,gof] = fit(centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)',ft);
            [f,gof] = fit(centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)','exp1','StartPoint', [max(abs(MaxDriveForces))*2 -1/mean(abs(MaxDriveForces))]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Force: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' pN'],...
                ['N = ' num2str(length(abs(MaxDriveForces)))],['R^2 = ' num2str(round(gof.rsquare*1000)/1000)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            title('Drive/ Hinder Force   ');
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
            ylabel('Drive Count'); xlabel('');
            xlim([0 xmax]);
            
            % Get variables ready for EXCEL
            EXCELTitle = {'Drive/Hinder','Force Mean (pN)','Force Error (pN)','Confidence Interval','No. of Sample'};
            DriveValues = {'Drive',CIMean,Error,ConfidenceInterval,length(abs(MaxDriveForces))};
            
            axes(ha(2));
            edges = 0:BinSpacing:max(abs(MaxHinderForces));
            centers = edges(2:end)-BinSpacing/2;
            Hist = histogram(abs(MaxHinderForces),edges,'FaceColor',[1 0.4 0]); hold on;
            % Fit exponential
            [f,gof] = fit(centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)','exp1','StartPoint', [max(abs(MaxHinderForces))*2 -1/mean(abs(MaxHinderForces))]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(ExpFitPointsRemoved+1:end)',Hist.Values(ExpFitPointsRemoved+1:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Force: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' pN'],...
                ['N = ' num2str(length(abs(MaxHinderForces)))],['R^2 = ' num2str(round(gof.rsquare*1000)/1000)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
            ylabel('Hinder Count'); xlabel('');
            xlim([0 xmax]);
            xlabel('Force (pN)');

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\MaxForceExp.jpg');
            savefig('Aggregate Images\MaxForceExp.fig');
            
            % Save force in EXCEL
            HinderValues = {'Hinder',CIMean,Error,ConfidenceInterval,length(abs(MaxHinderForces))};
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            xlswrite('Aggregate Images\ForceExpFit.xlsx',[EXCELTitle;DriveValues;HinderValues]);
            %% Plot histograms of Drive and Hinder duration and DistanceForces
            DurationBinSpacing = 0.5;
            DistForceBinSpacing = 10;
            xText = 0.4;
            yText = 0.79;

            figure;
            ha = tight_subplot(2,2,[.02 .05],[.13 .10],[.08 .03]);
            axes(ha(1));
        %     subplot(2,2,1);
            edges = 0:DurationBinSpacing:max(DriveDuration);
            histogram(DriveDuration,edges,'FaceColor','r');
            title('Duration');
            stdev = std(DriveDuration);
            sem = stdev/sqrt(length(DriveDuration));
            text(xText,yText,{['Mean: ' num2str(round(mean(DriveDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
                ['Std = ' num2str(round(stdev*10)/10) ' s'],...
                ['N = ' num2str(length(DriveDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
            ylabel('Drive');
            xlim([0 32]);

            axes(ha(2));
        %     subplot(2,2,2);
            edges = 0:DistForceBinSpacing:max(abs(DriveDistForces));
            histogram(abs(DriveDistForces),edges,'FaceColor','r');
            title('Force');
            stdev = std(DriveDistForces);
            sem = stdev/sqrt(length(DriveDistForces));
            text(xText,yText,{['Mean: ' num2str(abs(round(mean(DriveDistForces)))) '  ' num2str(round(sem)) ' nm'],...
                ['Std = ' num2str(round(stdev)) ' nm'],...
                ['N = ' num2str(length(DriveDistForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'XTick',[]);
            xlim([-10 780]);

            axes(ha(3));
        %     subplot(2,2,3);
            edges = 0:DurationBinSpacing:max(HinderDuration);
            histogram(HinderDuration,edges,'FaceColor',[0.9 0.9 0]);
            stdev = std(HinderDuration);
            sem = stdev/sqrt(length(HinderDuration));
            text(xText,yText,{['Mean: ' num2str(round(mean(HinderDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
                ['Std = ' num2str(round(stdev*10)/10) ' s'],...
                ['N = ' num2str(length(HinderDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(round(mean(HinderDuration)*10)/10) ' s'],['N = ' num2str(length(HinderDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'TickDir','out');
            ylabel('Hinder');
            xlim([0 32]);
            xlabel('Time (s)');

            axes(ha(4));
        %     subplot(2,2,4);
            edges = 0:DistForceBinSpacing:max(HinderDistForces);
            histogram(HinderDistForces,edges,'FaceColor',[0.9 0.9 0]);
            stdev = std(HinderDistForces);
            sem = stdev/sqrt(length(HinderDistForces));
            text(xText,yText,{['Mean: ' num2str(abs(round(mean(HinderDistForces)))) '  ' num2str(round(sem)) ' nm'],...
                ['Std = ' num2str(round(stdev)) ' nm'],...
                ['N = ' num2str(length(HinderDistForces))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(abs(round(mean(HinderDistForces)))) ' nm'],['N = ' num2str(length(HinderDistForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'TickDir','out');
            xlim([-10 780]);
            xlabel('Distance Force (nm)');

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\DurationForce(Per Drive or Hinder).jpg');
            savefig('Aggregate Images\DurationForce(Per Drive or Hinder).fig');                
            %% Plot histograms of MeanDriveDuration, MeanHinderDuration and drive and hinder DistanceForces
            % Note: we calculate the mean duration for a trace to get a single
            % datapoint here. So one trace only give one hinder/ drive/ both
            % datapoint for duration
            DurationBinSpacing = 2;
            DistForceBinSpacing = 50;
            xText = 0.4;
            yText = 0.79;

            figure;
            ha = tight_subplot(2,2,[.02 .05],[.13 .10],[.08 .03]);
            axes(ha(1));
        %     subplot(2,2,1);
            edges = 0:DurationBinSpacing:max(MeanDriveDuration);
            histogram(MeanDriveDuration,edges,'FaceColor','r');
            title('Duration');
            stdev = std(MeanDriveDuration);
            sem = stdev/sqrt(length(MeanDriveDuration));
            text(xText,yText,{['Mean: ' num2str(round(mean(MeanDriveDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
                ['Std = ' num2str(round(stdev*10)/10) ' s'],...
                ['N = ' num2str(length(MeanDriveDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(round(mean(MeanDriveDuration)*10)/10) ' s'],['N = ' num2str(length(MeanDriveDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
            ylabel('Drive');
            xlim([0 32]);

            axes(ha(2));
        %     subplot(2,2,2);
            edges = 0:DistForceBinSpacing:max(abs(DriveDistForces));
            histogram(abs(DriveDistForces),edges,'FaceColor','r');
            title('Force');
            stdev = std(DriveDistForces);
            sem = stdev/sqrt(length(DriveDistForces));
            text(xText,yText,{['Mean: ' num2str(abs(round(mean(DriveDistForces)))) '  ' num2str(round(sem)) ' nm'],...
                ['Std = ' num2str(round(stdev)) ' nm'],...
                ['N = ' num2str(length(DriveDistForces))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(abs(round(mean(DriveDistForces)))) ' nm'],['N = ' num2str(length(DriveDistForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'XTick',[]);
            xlim([-10 780]);

            axes(ha(3));
        %     subplot(2,2,3);
            edges = 0:DurationBinSpacing:max(MeanHinderDuration);
            histogram(MeanHinderDuration,edges,'FaceColor',[0.9 0.9 0]);
            stdev = std(MeanHinderDuration);
            sem = stdev/sqrt(length(MeanHinderDuration));
            text(xText,yText,{['Mean: ' num2str(round(mean(MeanHinderDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
                ['Std = ' num2str(round(stdev*10)/10) ' s'],...
                ['N = ' num2str(length(MeanHinderDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(round(mean(MeanHinderDuration)*10)/10) ' s'],['N = ' num2str(length(MeanHinderDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'TickDir','out');
            ylabel('Hinder');
            xlim([0 32]);
            xlabel('Time (s)');

            axes(ha(4));
        %     subplot(2,2,4);
            edges = 0:DistForceBinSpacing:max(HinderDistForces);
            histogram(HinderDistForces,edges,'FaceColor',[0.9 0.9 0]);
            stdev = std(HinderDistForces);
            sem = stdev/sqrt(length(HinderDistForces));
            text(xText,yText,{['Mean: ' num2str(abs(round(mean(HinderDistForces)))) '  ' num2str(round(sem)) ' nm'],...
                ['Std = ' num2str(round(stdev)) ' nm'],...
                ['N = ' num2str(length(HinderDistForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'TickDir','out');
            xlim([-10 780]);
            xlabel('Distance Force (nm)');

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\DurationForce(By Trace).jpg');
            savefig('Aggregate Images\DurationForce(By Trace).fig');
            %% Plot histograms of MeanDriveDuration, MeanHinderDuration and drive and hinder Forces
            % Note: we calculate the mean duration for a trace to get a single
            % datapoint here. So one trace only give one hinder/ drive/ both
            % datapoint for duration
            DurationBinSpacing = 2;
            ForceBinSpacing = 0.5;
            xText = 0.4;
            yText = 0.8;
            UseLogY = 1;

            figure;
            ha = tight_subplot(2,2,[.02 .05],[.13 .10],[.08 .03]);
            axes(ha(1));
        %     subplot(2,2,1);
            edges = 0:DurationBinSpacing:max(MeanDriveDuration);
            histogram(MeanDriveDuration,edges,'FaceColor','r');
            title('Duration');
            stdev = std(MeanDriveDuration);
            sem = stdev/sqrt(length(MeanDriveDuration));
            text(xText,yText,{['Mean: ' num2str(round(mean(MeanDriveDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
                ['Std = ' num2str(round(stdev*10)/10) ' s'],...
                ['N = ' num2str(length(MeanDriveDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(round(mean(MeanDriveDuration)*10)/10) ' s'],['N = ' num2str(length(MeanDriveDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
            ylabel('Drive');
            xlim([0 32]);

            axes(ha(2));
        %     subplot(2,2,2);
            edges = 0:ForceBinSpacing:max(abs(DriveForces));
            histogram(abs(DriveForces),edges,'FaceColor','r');
            title('Force');
            stdev = std(DriveForces);
            sem = stdev/sqrt(length(DriveForces));
            text(xText-0.1,yText,{['Mean: ' num2str(abs(round(mean(DriveForces)*100)/100)) '  ' num2str(round(sem*100)/100) ' pN'],...
                ['Std = ' num2str(round(stdev*100)/100) ' nm'],...
                ['N = ' num2str(length(DriveForces))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(abs(round(mean(DriveDistForces)))) ' nm'],['N = ' num2str(length(DriveDistForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'XTick',[]);
            xlim([0 10]);
            if UseLogY == 1
                set(gca,'yscale','log');
            end

            axes(ha(3));
        %     subplot(2,2,3);
            edges = 0:DurationBinSpacing:max(MeanHinderDuration);
            histogram(MeanHinderDuration,edges,'FaceColor',[0.9 0.9 0]);
            stdev = std(MeanHinderDuration);
            sem = stdev/sqrt(length(MeanHinderDuration));
            text(xText,yText,{['Mean: ' num2str(round(mean(MeanHinderDuration)*10)/10) '  ' num2str(round(sem*10)/10) ' s'],...
                ['Std = ' num2str(round(stdev*10)/10) ' s'],...
                ['N = ' num2str(length(MeanHinderDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
        %     text(xText,yText,{['Mean: ' num2str(round(mean(MeanHinderDuration)*10)/10) ' s'],['N = ' num2str(length(MeanHinderDuration))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'TickDir','out');
            ylabel('Hinder');
            xlim([0 32]);
            xlabel('Time (s)');

            axes(ha(4));
        %     subplot(2,2,4);
            edges = 0:ForceBinSpacing:max(HinderForces);
            histogram(HinderForces,edges,'FaceColor',[0.9 0.9 0]);
            stdev = std(HinderForces);
            sem = stdev/sqrt(length(HinderForces));
            text(xText-0.1,yText,{['Mean: ' num2str(abs(round(mean(HinderForces)*100)/100)) '  ' num2str(round(sem*100)/100) ' pN'],...
                ['Std = ' num2str(round(stdev*100)/100) ' nm'],...
                ['N = ' num2str(length(HinderForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',12,'fontweight','b','YTick',[],'TickDir','out');
            xlim([0 10]);
            xlabel('Force (pN)');
            if UseLogY == 1
                set(gca,'yscale','log');
            end

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\DurationForce(By Trace).jpg');
            savefig('Aggregate Images\DurationForce(By Trace).fig');
            %% Plot histograms of MeanDriveDuration and MeanHinderDuration and plot exponential fit
            % Note: we calculate the mean duration for a trace to get a single
            % datapoint here. So one trace only give one hinder/ drive/ both
            % datapoint for duration
            DurationBinSpacing = 0.0001;
            xText = 0.22;
            yText = 0.75;
            FontSize = 14;
            ConfidenceInterval = 0.95;

            figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
            ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
            axes(ha(1));
            edges = 0:DurationBinSpacing:max(MeanDriveDuration);
            centers = edges(2:end)-DurationBinSpacing/2;
            Hist = histogram(MeanDriveDuration,edges,'FaceColor',[0 0 0.4]); hold on;
            % Fit exponential
            f = fit(centers(2:end)',Hist.Values(2:end)','exp1','StartPoint', [max(MeanDriveDuration)*2 -1/mean(MeanDriveDuration)]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(2:end)',Hist.Values(2:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Dwell: ' num2str(round(mean(CIMean)*10)/10) '  ' num2str(round(Error*10)/10) ' s'],...
                ['N = ' num2str(length(MeanDriveDuration))]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            title('Drive/ Hinder Duration ');
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[],'TickDir','out');
            ylabel('Drive Count'); xlabel('');
            xlim([0 42]);

            axes(ha(2));
            edges = 0:DurationBinSpacing:max(MeanHinderDuration);
            centers = edges(2:end)-DurationBinSpacing/2;
            Hist = histogram(MeanHinderDuration,edges,'FaceColor',[1 0.4 0]); hold on;
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
            % Fit exponential
            f = fit(centers(2:end)',Hist.Values(2:end)','exp1','StartPoint', [max(MeanHinderDuration)*2 -1/mean(MeanHinderDuration)]);
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,centers(2:end)',Hist.Values(2:end)');
            set(h,'LineWidth',1.2,'Color','black');
            legend('off');
            text(xText,yText,{['Dwell: ' num2str(round(mean(CIMean)*10)/10) '  ' num2str(round(Error*10)/10) ' s'],...
                ['N = ' num2str(length(MeanHinderDuration))]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            % Write titles and axes labels
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            ylabel('Hinder Count');
            xlim([0 42]);
            xlabel('Time (s)');

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\MeanDurationExpFit(By Trace).jpg');
            savefig('Aggregate Images\MeanDurationExpFit(By Trace).fig');
            %% Plot histograms of Drive and Hinder MaxDistForces
            DistForceBinSpacing = 20;
            xText = 0.22;
            yText = 0.75;
            FontSize = 12;

            figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
            ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);

            axes(ha(1));
            edges = 0:DistForceBinSpacing:max(abs(DriveMaxDistForces));
            hist=histogram(abs(DriveMaxDistForces),edges,'FaceColor','r');
            title('Distance Force');
            stdev = std(DriveMaxDistForces);
            sem = stdev/sqrt(length(DriveMaxDistForces));
            text(xText,yText,{['Mean: ' num2str(abs(round(mean(DriveMaxDistForces)))) '  ' num2str(round(sem)) ' nm'],...
                ['Std = ' num2str(round(stdev)) ' nm'],...
                ['N = ' num2str(length(DriveMaxDistForces))]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[]);
            ylabel('Drive Count'); xlabel('');
            xlim([-10 780]);
            ylim([0 max(hist.Values)*1.6]);

            axes(ha(2));
            edges = 0:DistForceBinSpacing:max(HinderMaxDistForces);
            hist=histogram(HinderMaxDistForces,edges,'FaceColor',[0.9 0.9 0]);
            stdev = std(HinderMaxDistForces);
            sem = stdev/sqrt(length(HinderMaxDistForces));
            text(xText,yText,{['Mean: ' num2str(abs(round(mean(HinderMaxDistForces)))) '  ' num2str(round(sem)) ' nm'],...
                ['Std = ' num2str(round(stdev)) ' nm'],...
                ['N = ' num2str(length(HinderMaxDistForces))]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
            xlim([-10 780]);
            ylim([0 max(hist.Values)*1.6]);
            ylabel('Hinder Count');
            xlabel('Distance Force (nm)');

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\MaxDistanceForce.jpg');
            savefig('Aggregate Images\MaxDistanceForce.fig');  
            %% Plot histograms of Drive and Hinder forces (all forces)
            % Note: we calculate the mean duration for a trace to get a single
            % datapoint here. So one trace only give one hinder/ drive/ both
            % datapoint for duration
            ForceBinSpacing = 0.4;
            xText = 0.28;
            yText = 0.78;
            UseLogY = 0;
            FontSize = 14;
            xmax = 8;

            figure('units','normalized','outerposition',[0.2 0.2 0.2 0.48])
            ha = tight_subplot(2,1,[.01 .05],[.16 .10],[.13 .03]);
            axes(ha(1));
            edges = 0:ForceBinSpacing:max(abs(DriveForces));
            histogram(abs(DriveForces),edges,'FaceColor',[0 0 0.4]);
            title('Drive/ Hinder Force');
            stdev = std(DriveForces);
            sem = stdev/sqrt(length(DriveForces));
            text(xText-0.1,yText,{['Mean: ' num2str(abs(round(mean(DriveForces)*100)/100)) '  ' num2str(round(sem*100)/100) ' pN'],...
                ['Std = ' num2str(round(stdev*100)/100) ' nm'],...
                ['N = ' num2str(length(DriveForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'XTick',[]);
            xlim([0 xmax]);
            if UseLogY == 1
                set(gca,'yscale','log');
            end
            ylabel('Drive Count'); xlabel('');
            
            % Get variables ready for EXCEL
            EXCELTitle = {'Drive/Hinder','Force Mean (pN)','Force Error (pN)','No. of Sample'};
            DriveValues = {'Drive',mean(DriveForces),sem,length(abs(DriveForces))};

            axes(ha(2));
            edges = 0:ForceBinSpacing:max(HinderForces);
            histogram(HinderForces,edges,'FaceColor',[1 0.4 0]);
            stdev = std(HinderForces);
            sem = stdev/sqrt(length(HinderForces));
            text(xText-0.1,yText,{['Mean: ' num2str(abs(round(mean(HinderForces)*100)/100)) '  ' num2str(round(sem*100)/100) ' pN'],...
                ['Std = ' num2str(round(stdev*100)/100) ' nm'],...
                ['N = ' num2str(length(HinderForces))]},'FontSize',12,'fontweight','b','Units','normalized');
            a = gca; set(a,'box','off','color','none');
            b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
            axes(a); linkaxes([a b]);
            set(gca,'FontSize',FontSize,'fontweight','b','YTick',[],'TickDir','out');
            xlim([0 xmax]);
            ylabel('Hinder Count');
            xlabel('Force (pN)');
            if UseLogY == 1
                set(gca,'yscale','log');
            end

            % Save figures for Duration Force diagram
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\Force.jpg');
            savefig('Aggregate Images\Force.fig');
            
             % Save force in EXCEL
            HinderValues = {'Hinder',mean(HinderForces),sem,length(abs(HinderForces))};
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            xlswrite('Aggregate Images\AllForceMean.xlsx',[EXCELTitle;DriveValues;HinderValues]);
        %% d) Plot TimePoints
        xBin = 40;
        yBin = 8;

        figure('units','normalized','outerposition',[0.2 0.2 0.5 0.7]);
        ha = tight_subplot(4,1,[.02 .05],[.14 .06],[.10 .03]);
        axes(ha(1));
%         plot(ContactArea,TotalKin,'.k');
        n = hist3([ContactArea TotalKin],[xBin max(TotalKin)-min(TotalKin)]);
        n1 = n';
        n1(size(n,2) + 1, size(n,1) + 1) = 0;
        xb = linspace(min(ContactArea),max(ContactArea),size(n,1)+1);
        yb = linspace(min(TotalKin),max(TotalKin),size(n,2)+1);
        h = pcolor(xb,yb,n1);
        colormap(hot) % heat map
%         ylim([0 max(TimePoints(:,1))*1.2]);
        ylabel('Total');
        title('Total, Drive, Hinder, and Net Drive vs Motor Availability');
        set(gca,'FontSize',13,'fontweight','b','XTick',[]);
        caxis([min(min(n1)) max(max(n1))/20]);

        axes(ha(2));
%         plot(ContactArea,DrivingKin,'.b'); hold on;
        n = hist3([ContactArea DrivingKin],[xBin max(DrivingKin)-min(DrivingKin)]);
        n1 = n';
        n1(size(n,2) + 1, size(n,1) + 1) = 0;
        xb = linspace(min(ContactArea),max(ContactArea),size(n,1)+1);
        yb = linspace(min(DrivingKin),max(DrivingKin),size(n,2)+1);
        h = pcolor(xb,yb,n1);
        colormap(hot) % heat map
        ylabel('Drive');
        set(gca,'FontSize',13,'fontweight','b','XTick',[]);
        caxis([min(min(n1)) max(max(n1))/20]);
        
        axes(ha(3));
%         plot(ContactArea,HinderingKin,'.r'); hold on;
        n = hist3([ContactArea HinderingKin],[xBin max(HinderingKin)-min(HinderingKin)]);
        n1 = n';
        n1(size(n,2) + 1, size(n,1) + 1) = 0;
        xb = linspace(min(ContactArea),max(ContactArea),size(n,1)+1);
        yb = linspace(min(HinderingKin),max(HinderingKin),size(n,2)+1);
        h = pcolor(xb,yb,n1);
        colormap(hot) % heat map
        ylabel('Hinder');
        set(gca,'FontSize',13,'fontweight','b','XTick',[]);
        caxis([min(min(n1)) max(max(n1))/20]);

        axes(ha(4));
%         plot(ContactArea,DriveHinderDiff,'.b'); hold on;
        n = hist3([ContactArea DriveHinderDiff],[xBin max(DriveHinderDiff)-min(DriveHinderDiff)]);
        n1 = n';
        n1(size(n,2) + 1, size(n,1) + 1) = 0;
        xb = linspace(min(ContactArea),max(ContactArea),size(n,1)+1);
        yb = linspace(min(DriveHinderDiff),max(DriveHinderDiff),size(n,2)+1);
        h = pcolor(xb,yb,n1);
        colormap(hot) % heat map
%         ylim([0 max(TimePoints(:,1))*1.2]);
        xlabel('Motor Availability (Mic. Length x Kin. Density) (nm^-^1)');
        ylabel('Net Drive');
        a = gca; set(a,'box','off','color','none');
        b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
        axes(a); linkaxes([a b]);
        set(gca,'FontSize',13,'fontweight','b','TickDir','out');
        caxis([min(min(n1)) max(max(n1))/20]);
        
        % Save figures for Duration Force diagram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\All Graphs vs Motor Availability.jpg');
        savefig('Aggregate Images\All Graphs vs Motor Availability.fig');
        %% e) Plot histograms for a certain range of ContactArea
        GroupNo = 1;
        MaxContact = max(ContactArea);
        ContactDivision = ceil(MaxContact/GroupNo*10)/10;
        AxesFontSize = 16;      % Determine axes font size
        DrawText = 0;           % 1 if want to write text. 0 if don't want
        yLabelHeight = 1;
        xLimitReduction = 1.5;  % Reduce the limit for x
        xLimitMax = 8.5;          % Use this to determine what the xLimitMax is (if use this then xLimitReduction is not used. Need to manually change)

        % Start figures for TotalKin, Driving and HinderingKin, and DriveHinderDiff
%         h1 = figure; 
        h1 = figure('units','normalized','outerposition',[0 0.2 0.8 0.4]);
        edges1 = -0.5:1:max(TotalKin)+0.5;
        ha1 = tight_subplot(GroupNo,1,[.005 .01],[.16 .08],[.15 .03]);
        h2 = figure; edges2 = -0.5:1:max([DrivingKin; HinderingKin])+0.5;
        ha2 = tight_subplot(GroupNo,2,[.005 .01],[.16 .08],[.15 .03]);
        h3 = figure; edges3 = -0.5+min(DriveHinderDiff):1:max(DriveHinderDiff)+0.5;
        ha3 = tight_subplot(GroupNo,1,[.005 .01],[.16 .08],[.15 .03]);

        % Get the bin centers
        center1 = 0:1:max(TotalKin);
        center2 = 0:1:max([DrivingKin; HinderingKin]);
        center3 = min(DriveHinderDiff):1:max(DriveHinderDiff);

        % Initialize each matrix
        Matrix1 = zeros(length(center1)+1,GroupNo*2+1);
        Matrix2a = zeros(length(center2)+1,GroupNo*2+1);
        Matrix3 = zeros(length(center3)+1,GroupNo*2+1);

        % Put in bin centers for first row and ContactArea centers for first column
        ContactSorted = sort(ContactArea);
        ContactEnds = ContactSorted((1:GroupNo)*floor(length(ContactSorted)/GroupNo));
        ContactEnds(end) = ContactSorted(end);
        ContactStarts = [0; ContactEnds(1:end-1)];
        ContactCenters = (1:GroupNo)*ContactDivision-ContactDivision/2;
        Matrix1(2:end,1) = center1; Matrix1(1,2:end) = [ContactCenters ContactCenters];
        Matrix2a(2:end,1) = center2; Matrix2a(1,2:end) = [ContactCenters ContactCenters];
        Matrix2b = Matrix2a;
        Matrix3(2:end,1) = center3; Matrix3(1,2:end) = [ContactCenters ContactCenters];
        xMax = max(TotalKin);

        for i = 1:GroupNo
            %ContactEnd = ContactStart + ContactDivision;
            ContactIndex = find(ContactArea>ContactStarts(i) & ContactArea<=ContactEnds(i));

            figure(h1);
            axes(ha1(GroupNo-i+1));
            N = histcounts(TotalKin(ContactIndex),edges1);
            Fraction = N/sum(N);
            Fraction(Fraction == 0) = NaN;
            bar(edges1(2:end)-0.5,Fraction,1);
            text(0.75,0.8,['N = ' num2str(length(TotalKin(ContactIndex)))],'FontSize',AxesFontSize,'fontweight','b','Units','normalized');
            text(0.75,0.6,['Mean = ' num2str(mean(TotalKin(ContactIndex)))],'FontSize',AxesFontSize,'fontweight','b','Units','normalized');
            if DrawText == 1
                text(center1,Fraction,num2str(round(Fraction'*100)/100),'FontSize',AxesFontSize,'FontWeight','b','horizontalalignment','center','verticalalignment','bottom');
            end
%             ylabel(num2str(round(ContactCenters(i))));
            ylabel('0.2 nM');
            xlim([0 9]);
            ylim([0 0.9]);
%             xlim([0 xLimitMax]);
%             ylim([0 0.55]);
            
            if i == 1
%                 set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
%                 title('Total Attached Motors as Availability Rises');
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',1:8,'YTick',0.2:0.2:0.8);
                title('Total Attached Motors');
            elseif i == GroupNo
                a = gca; set(a,'box','off','color','none');
                b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
                axes(a); linkaxes([a b]);
                set(gca,'FontSize',AxesFontSize,'fontweight','b','YTick',[],'TickDir','out');
                xlabel('Number of Kinesins');
                text(-0.15,yLabelHeight,'Motor Availability','FontSize',AxesFontSize+2,'fontweight','b','Units','normalized','Rotation',90);
                ylim([0 0.55]);
            else
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
            end

            % Record Matrix1. First few columns are counts. Next few columns
            % are percentage
            Matrix1(2:end,i+1) = N;
            Matrix1(2:end,i+GroupNo+1) = Fraction;

            figure(h2);
            axes(ha2(2*(GroupNo-i)+1));
            KinDrive = DrivingKin(ContactIndex);
            KinDrive = KinDrive(KinDrive>0);
            N = histcounts(KinDrive,edges2);
%             N(N == 0) = NaN;
            Fraction = N/sum(N);
            Fraction(Fraction == 0) = NaN;
%             bar(edges2(2:end)-0.5,N,1); hold on;
            bar(edges2(2:end)-0.5,Fraction,1); hold on;
            text(0.5,0.7,['N_D = ' num2str(length(KinDrive))],'FontSize',AxesFontSize,'fontweight','b','Units','normalized');
            if DrawText == 1
                text(center2,Fraction,num2str(Fraction'),'Color','b','FontSize',AxesFontSize,'FontWeight','b','horizontalalignment','center','verticalalignment','bottom');
            end
            ylabel(num2str(round(ContactCenters(i))));
%             xlim([0 xMax-xLimitReduction]);
            xlim([0 xLimitMax]);
%             ylim([0 length(KinDrive)]);
            ylim([0 1]);

            % Record Matrix2a
            Matrix2a(2:end,i+1) = N;
            Matrix2a(2:end,i+GroupNo+1) = Fraction;
            
            if i == GroupNo
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
                title('                                    No. of Drive/ Hinder as Availability Rises');
            elseif i == 1
                % Remove upper ticks
                a = gca; set(a,'box','off','color','none');
                b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
                axes(a); linkaxes([a b]);
                set(gca,'FontSize',AxesFontSize,'fontweight','b','YTick',[],'TickDir','out');
                xlabel('                                 Number of Kinesins');
                text(-0.3,yLabelHeight,'Motor Availability','FontSize',AxesFontSize+2,'fontweight','b','Units','normalized','Rotation',90);
                ylim([0 1]);
            else
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
            end

            axes(ha2(2*(GroupNo-i)+2));
            KinHinder = HinderingKin(ContactIndex);
            KinHinder = KinHinder(KinHinder>0);
            N = histcounts(KinHinder,edges2);
%             N(N == 0) = NaN;
            Fraction = N/sum(N);
            Fraction(Fraction == 0) = NaN;
            bar(edges2(2:end)-0.5,Fraction,1,'r');
            text(0.5,0.7,['N_H = ' num2str(length(KinHinder))],'FontSize',AxesFontSize,'fontweight','b','Units','normalized');
            if DrawText == 1
                text(center2,Fraction,num2str(Fraction'),'Color','r','FontSize',AxesFontSize,'FontWeight','b','horizontalalignment','center','verticalalignment','bottom');
            end
            %text(center2,Fraction,num2str(round(Fraction'*100)/100),'FontSize',14','FontWeight','b','horizontalalignment','center','verticalalignment','bottom');
%             ylabel(num2str(ContactCenters(i)));
%             xlim([0 xMax-xLimitReduction]);
            xlim([0 xLimitMax]);
            ylim([0 1]);

            if i == GroupNo
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
%                 title('No. of Drive/ Hinder as Availability Rises');
            elseif i == 1
                % Remove upper ticks
                a = gca; set(a,'box','off','color','none');
                b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
                axes(a); linkaxes([a b]);
                set(gca,'FontSize',AxesFontSize,'fontweight','b','YTick',[],'TickDir','out');
%                 xlabel('Number of Kinesins');
%                 text(-0.15,yLabelHeight,'Motor Availability','FontSize',AxesFontSize+2,'fontweight','b','Units','normalized','Rotation',90);
                ylim([0 1]);
            else
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
            end

            % Record Matrix2b
            Matrix2b(2:end,i+1) = N;
            Matrix2b(2:end,i+GroupNo+1) = Fraction;

            figure(h3);
            axes(ha3(GroupNo-i+1));
            N = histcounts(DriveHinderDiff(ContactIndex),edges3);
            Fraction = N/sum(N);
            Fraction(Fraction == 0) = NaN;
            bar(edges3(2:end)-0.5,Fraction,1);
            text(0.75,0.8,['N = ' num2str(length(DriveHinderDiff(ContactIndex)))],'FontSize',AxesFontSize,'fontweight','b','Units','normalized');
            if DrawText == 1
                text(center3,Fraction,num2str(round(Fraction'*100)/100),'FontSize',AxesFontSize,'FontWeight','b','horizontalalignment','center','verticalalignment','bottom');
            end
            ylabel(num2str(round(ContactCenters(i))));
%             xlim([min(DriveHinderDiff)+0.5 max(DriveHinderDiff)-0.5]);
            xlim([-3.5 xLimitMax]);
            ylim([0 0.5]);

            if i == GroupNo
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
                title('Net Driving Motors as Availability Rises');
            elseif i == 1
                a = gca; set(a,'box','off','color','none');
                b = axes('Position',get(a,'Position'),'box','on','xtick',[],'ytick',[]);
                axes(a); linkaxes([a b]);
                set(gca,'FontSize',AxesFontSize,'fontweight','b','YTick',[],'TickDir','out');
                xlabel('Number of Kinesins');
                text(-0.15,yLabelHeight,'Motor Availability','FontSize',AxesFontSize+2,'fontweight','b','Units','normalized','Rotation',90);
                ylim([0 0.5]);
            else
                set(gca,'FontSize',AxesFontSize,'fontweight','b','XTick',[],'YTick',[]);
            end

            % Record Matrix3
            Matrix3(2:end,i+1) = N;
            Matrix3(2:end,i+GroupNo+1) = Fraction;

    %         ContactStart = ContactEnd;
    %         ContactEnd = ContactStart + ContactDivision;
        end
    
        % Save figures for Dur-DistForces
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        figure(h1);
        print('-djpeg','-r300','Aggregate Images\Total Attached Motors as Availability Rises.jpg');
        savefig('Aggregate Images\Total Attached Motors as Availability Rises.fig');
        figure(h2);
        print('-djpeg','-r300','Aggregate Images\No. of Drive or Hinder as Availability Rises.jpg');
        savefig('Aggregate Images\No. of Drive or Hinder as Availability Rises.fig');
        figure(h3);
        print('-djpeg','-r300','Aggregate Images\Net Driving Motors as Availability Rises.jpg');
        savefig('Aggregate Images\Net Driving Motors as Availability Rises.fig');
        %% f) Plot average DistanceForce vs Time
        DriveIndexes = find(MeanDistForces<0);
        HinderIndexes = find(MeanDistForces>0);
        VelocityThreshold = 5000;
        CutTail = 1;        % 1 if we want to cut the tail based on TailVelocityThreshold
        TailCutMaxNumber = 5;
        TailVelocityThreshold = 800;
        GroupNo = 1;
        UseSmooth = 1;
        SmoothSpan = 5;
        PlotAll = 1;
        UseRelativeVelocity = 11;    % 1 if we want to take into account kinesin velocity. 0 if not. 5 to plot kinesin velocity. 10 to plot DistanceForces, 11 to plot Force
        xLimitDistForces = [0 50];
        yLimitDistForces = [0 7];
        UseSemilogX = 1;        % 0 to not use semilogx. 1 or others to use
        
        % Sort DriveIndexes and HinderIndexes based on duration
        [~,SortedIndex] = sort(AllDuration(DriveIndexes));
        DriveIndexes = DriveIndexes(SortedIndex);
        [~,SortedIndex] = sort(AllDuration(HinderIndexes));
        HinderIndexes = HinderIndexes(SortedIndex);
        
        % Plot all DriveDistanceForces
%         figure;
        % Initialize AllDriveDistanceForces, MicVelsDrive and KinVelsDrive
        AllDriveDistanceForces = NaN(round(max(AllDuration)/ExposureTime)+1,length(DriveIndexes));
        MicRelativeVelsDrive = NaN(round(max(AllDuration)/ExposureTime)+1,length(DriveIndexes));
        % Loop to get all DriveDistanceForces
        for i = 1:length(DriveIndexes)
            % Get the DistanceForce
            DistanceForce = DurDistForces(FirstInd(DriveIndexes(i)):LastInd(DriveIndexes(i)),[5 8]);
            DistanceForce(:,1) = DistanceForce(:,1)-DistanceForce(1,1)+ExposureTime;
            FrameIndex = round(DistanceForce(:,1)/ExposureTime);
            
            % Smooth DistanceForce if the UseSmooth option is 1
            if UseSmooth == 1
                DistanceForce(:,2) = smooth(DistanceForce(:,2),SmoothSpan);
            end
            
            % Store DistanceForce in AllDriveDistanceForces
            AllDriveDistanceForces(FrameIndex,i) = DistanceForce(:,2);
            
            % Get MicVelsDrive
            MicVelsDrive = MicVelocity(FirstInd(DriveIndexes(i))+1:LastInd(DriveIndexes(i)));
            
            % Calculate KinVelsDrive
            KinVelsDrive = diff(DistanceForce(:,2))./diff(DistanceForce(:,1));
            KinDriveDistanceForce = DistanceForce(1:end-1,2);
            
            % Evaluate tails to cut those with very high velocity if
            % CutTail = 1
            if CutTail == 1
                % Determine how many points at tail to evaluate
                TailPoints = max(min(TailCutMaxNumber,length(KinVelsDrive)-1),1);  % At least one, at most 5, or length(KinVelsDrive)-1
                CutIndex = find(KinVelsDrive(end-TailPoints+1:end)>TailVelocityThreshold,1);
                if ~isempty(CutIndex)
                    CutIndex = length(KinVelsDrive)-TailPoints+CutIndex;
                    KinVelsDrive(CutIndex:end)=NaN;
                    KinDriveDistanceForce(CutIndex:end)=NaN;
                end
            end
            
            % Make KinVelsDrive beyond a certain VelocityThreshold to be NaN
            KinVelsDrive(KinVelsDrive>VelocityThreshold)=NaN;
            KinDriveDistanceForce(KinVelsDrive>VelocityThreshold)=NaN;
            
            % Calculate Relative Velocity of Microtubule if
            % UseRelativeVelocity = 1
            if UseRelativeVelocity == 1
                MicRelativeVelsDrive(FrameIndex(1:end-1),i) = MicVelsDrive - KinVelsDrive;
            elseif UseRelativeVelocity == 5
                MicRelativeVelsDrive(FrameIndex(1:end-1),i) = KinVelsDrive;
            elseif UseRelativeVelocity == 10
                MicRelativeVelsDrive(FrameIndex(1:end-1),i) = KinDriveDistanceForce;
            elseif UseRelativeVelocity == 11
                % Convert DistanceForces to Force
                AdjustedKinDriveDistanceForce = abs(KinDriveDistanceForce) - DistanceOffset;
%                 FractionCutOff = 1;
                AdjustedKinDriveDistanceForce(AdjustedKinDriveDistanceForce>FractionCutOff*L0)=FractionCutOff*L0;
                KinDriveForces = NaN(length(AdjustedKinDriveDistanceForce),1);
                for j = 1:length(AdjustedKinDriveDistanceForce)
                    if ~isnan(AdjustedKinDriveDistanceForce(j))
                        KinDriveForces(j) = fzero(@(x)(AdjustedKinDriveDistanceForce(j)/L0-XWLCContour(x)),1);
                    end
                end
                MicRelativeVelsDrive(FrameIndex(1:end-1),i) = KinDriveForces;
%                 MicRelativeVelsDrive(FrameIndex(1:end-1),i) = sign(KinDriveDistanceForce).*(1/4*(1-(AdjustedKinDriveDistanceForce/L0)).^(-2)-1/4+(AdjustedKinDriveDistanceForce/L0)-0.8*(AdjustedKinDriveDistanceForce/L0).^2.15)*kT/P;
%                 MicRelativeVelsDrive(FrameIndex(1:end-1),i) = (1/4*(1-(AdjustedKinDriveDistanceForce/L0)).^(-2)-1/4+(AdjustedKinDriveDistanceForce/L0)-0.8*(AdjustedKinDriveDistanceForce/L0).^2.15)*kT/P;
            else
                MicRelativeVelsDrive(FrameIndex(1:end-1),i) = MicVelsDrive;
            end

            % Plot DistanceForce
%             plot(FrameIndex,DistanceForce(:,2)); hold on;
        end
        % Get the MeanDriveDistanceForces and the SEM
        MeanDriveDistanceForces = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        SEMDriveDistanceForces = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        MeanMicRelativeVelsDrive = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        SEMMicRelativeVelsDrive = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        DriveGroupLength = floor(length(DriveIndexes)/GroupNo);
        for i = 1:GroupNo
            % Get the MeanDriveDistanceForces and the SEM
            GroupDistanceForces = AllDriveDistanceForces(:,(i-1)*DriveGroupLength+1:i*DriveGroupLength);
            MeanDriveDistanceForces(:,i) = nanmean(GroupDistanceForces,2);
            SEMDriveDistanceForces(:,i) = nanstd(GroupDistanceForces,0,2)./sqrt(sum(~isnan(GroupDistanceForces),2));
            % Get the MeanMicRelativeVelsDrive and the SEM
            GroupMicRelativeVels = MicRelativeVelsDrive(:,(i-1)*DriveGroupLength+1:i*DriveGroupLength);
            MeanMicRelativeVelsDrive(:,i) = nanmean(GroupMicRelativeVels,2);
            SEMMicRelativeVelsDrive(:,i) = nanstd(GroupMicRelativeVels,0,2)./sqrt(sum(~isnan(GroupMicRelativeVels),2));
        end
        
        % Plot all HinderDistanceForces
%         figure;
        % Initialize AllHinderDistanceForces and MicRelativeVelsHinder
        AllHinderDistanceForces = NaN(round(max(AllDuration)/ExposureTime)+1,length(HinderIndexes));
        MicRelativeVelsHinder = NaN(round(max(AllDuration)/ExposureTime)+1,length(HinderIndexes));
        % Loop to get all HinderDistanceForces
        for i = 1:length(HinderIndexes)
            % Get the DistanceForce
            DistanceForce = DurDistForces(FirstInd(HinderIndexes(i)):LastInd(HinderIndexes(i)),[5 8]);
            DistanceForce(:,1) = DistanceForce(:,1)-DistanceForce(1,1)+ExposureTime;
            FrameIndex = round(DistanceForce(:,1)/ExposureTime);
            
            % Smooth DistanceForce if the UseSmooth option is 1
            if UseSmooth == 1
                DistanceForce(:,2) = smooth(DistanceForce(:,2),SmoothSpan);
            end
            
            % Store DistanceForce in AllHinderDistanceForces
            AllHinderDistanceForces(FrameIndex,i) = DistanceForce(:,2);
            
            % Get MicVelsHinder
            MicVelsHinder = MicVelocity(FirstInd(HinderIndexes(i))+1:LastInd(HinderIndexes(i)));
            
            % Calculate KinVelsHinder
            KinVelsHinder = diff(DistanceForce(:,2))./diff(DistanceForce(:,1));
            KinHinderDistanceForce = DistanceForce(1:end-1,2);
            
            % Evaluate tails to cut those with very high velocity if
            % CutTail = 1
            if CutTail == 1
                % Determine how many points at tail to evaluate
                TailPoints = max(min(TailCutMaxNumber,length(KinVelsHinder)-1),1);  % At least one, at most 5, or length(KinVelsDrive)-1
                CutIndex = find(KinVelsHinder(end-TailPoints+1:end)<-TailVelocityThreshold,1);
                if ~isempty(CutIndex)
                    CutIndex = length(KinVelsHinder)-TailPoints+CutIndex;
                    KinVelsHinder(CutIndex:end)=NaN;
                    KinHinderDistanceForce(CutIndex:end)=NaN;
                end
            end
            
            % Make KinVelsHinder below a certain VelocityThreshold to be NaN
            KinVelsHinder(KinVelsHinder<-VelocityThreshold)=NaN;
            KinHinderDistanceForce(KinVelsHinder<-VelocityThreshold)=NaN;
            
            % Calculate Relative Velocity of Microtubule if
            % UseRelativeVelocity = 1
            if UseRelativeVelocity == 1
                MicRelativeVelsHinder(FrameIndex(1:end-1),i) = MicVelsHinder - KinVelsHinder;
            elseif UseRelativeVelocity == 5
                MicRelativeVelsHinder(FrameIndex(1:end-1),i) = KinVelsHinder;
            elseif UseRelativeVelocity == 10
                MicRelativeVelsHinder(FrameIndex(1:end-1),i) = KinHinderDistanceForce;
            elseif UseRelativeVelocity == 11
                % Convert DistanceForces to Force
                AdjustedKinHinderDistanceForce = KinHinderDistanceForce - DistanceOffset;
                AdjustedKinHinderDistanceForce(AdjustedKinHinderDistanceForce>FractionCutOff*L0)=FractionCutOff*L0;
%                 MicRelativeVelsHinder(FrameIndex(1:end-1),i) = sign(KinHinderDistanceForce).*(1/4*(1-(AdjustedKinHinderDistanceForce/L0)).^(-2)-1/4+(AdjustedKinHinderDistanceForce/L0)-0.8*(AdjustedKinHinderDistanceForce/L0).^2.15)*kT/P;
                KinHinderForces = NaN(length(AdjustedKinHinderDistanceForce),1);
                for j = 1:length(AdjustedKinHinderDistanceForce)
                    if ~isnan(AdjustedKinHinderDistanceForce(j))
                        KinHinderForces(j) = fzero(@(x)(AdjustedKinHinderDistanceForce(j)/L0-XWLCContour(x)),1);
                    end
                end
                MicRelativeVelsHinder(FrameIndex(1:end-1),i) = KinHinderForces;
            else
                MicRelativeVelsHinder(FrameIndex(1:end-1),i) = MicVelsHinder;
            end

            % Plot DistanceForce
%             plot(FrameIndex,DistanceForce(:,2)); hold on;
        end
        % Get the MeanHinderDistanceForces and the SEM
        MeanHinderDistanceForces = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        SEMHinderDistanceForces = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        MeanMicRelativeVelsHinder = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        SEMMicRelativeVelsHinder = NaN(round(max(AllDuration)/ExposureTime)+1,GroupNo);
        HinderGroupLength = floor(length(HinderIndexes)/GroupNo);
        for i = 1:GroupNo
            % Get the MeanHinderDistanceForces and the SEM
            GroupDistanceForces = AllHinderDistanceForces(:,(i-1)*HinderGroupLength+1:i*HinderGroupLength);
            MeanHinderDistanceForces(:,i) = nanmean(GroupDistanceForces,2);
            SEMHinderDistanceForces(:,i) = nanstd(GroupDistanceForces,0,2)./sqrt(sum(~isnan(GroupDistanceForces),2));
            % Get the MeanMicRelativeVelsHinder and the SEM
            GroupMicRelativeVels = MicRelativeVelsHinder(:,(i-1)*HinderGroupLength+1:i*HinderGroupLength);
            MeanMicRelativeVelsHinder(:,i) = nanmean(GroupMicRelativeVels,2);
            SEMMicRelativeVelsHinder(:,i) = nanstd(GroupMicRelativeVels,0,2)./sqrt(sum(~isnan(GroupMicRelativeVels),2));
        end
        
        
        
%         % Plot both MeanDriveDistanceForces and MeanHinderDistanceForces
%         % on the same figure
%         Transparency = 0;
%         figure;
%         shadedErrorBar((1:length(MeanDriveDistanceForces))*ExposureTime,abs(MeanDriveDistanceForces),SEMDriveDistanceForces,'b',Transparency); hold on;
%         shadedErrorBar((1:length(MeanHinderDistanceForces))*ExposureTime,MeanHinderDistanceForces,SEMHinderDistanceForces,'r',Transparency);
%         plot((1:length(MeanDriveDistanceForces))*ExposureTime,abs(MeanDriveDistanceForces),'b','LineWidth',2); hold on;
%         plot((1:length(MeanHinderDistanceForces))*ExposureTime,abs(MeanHinderDistanceForces),'r','LineWidth',2);
%         
%         % Plot both MeanMicRelativeVelsDrive and MeanMicRelativeVelsHinder
%         % on the same figure
%         Transparency = 0;
%         figure;
%         shadedErrorBar((1:length(MeanMicRelativeVelsDrive))*ExposureTime,MeanMicRelativeVelsDrive,SEMMicRelativeVelsDrive,'b',Transparency); hold on;
%         shadedErrorBar((1:length(MeanMicRelativeVelsHinder))*ExposureTime,MeanMicRelativeVelsHinder,SEMMicRelativeVelsHinder,'r',Transparency);
        
        
        Transparency = 0.5;
        figure;
        if UseSemilogX == 0
            plot((1:length(MeanMicRelativeVelsDrive(:,i)))*ExposureTime,MeanMicRelativeVelsDrive(:,i),'b','LineWidth',2); hold on;
            plot((1:length(MeanMicRelativeVelsHinder(:,i)))*ExposureTime,MeanMicRelativeVelsHinder(:,i),'r','LineWidth',2);
        else
            semilogx((1:length(MeanMicRelativeVelsDrive(:,i)))*ExposureTime,MeanMicRelativeVelsDrive(:,i),'b','LineWidth',2); hold on;
            semilogx((1:length(MeanMicRelativeVelsHinder(:,i)))*ExposureTime,MeanMicRelativeVelsHinder(:,i),'r','LineWidth',2);
        end
        legend({'Driving','Hindering'});
        if PlotAll == 1
            for i = 1:GroupNo
                shadedErrorBar((1:length(MeanMicRelativeVelsDrive(:,i)))*ExposureTime,MeanMicRelativeVelsDrive(:,i),SEMMicRelativeVelsDrive(:,i),'b',Transparency); hold on;
                shadedErrorBar((1:length(MeanMicRelativeVelsHinder(:,i)))*ExposureTime,MeanMicRelativeVelsHinder(:,i),SEMMicRelativeVelsHinder(:,i),'r',Transparency); hold on;

                plot((1:length(MeanMicRelativeVelsDrive(:,i)))*ExposureTime,MeanMicRelativeVelsDrive(:,i),'b','LineWidth',2); hold on;
                plot((1:length(MeanMicRelativeVelsHinder(:,i)))*ExposureTime,MeanMicRelativeVelsHinder(:,i),'r','LineWidth',2);
            end
        else
            for i = GroupNo
                shadedErrorBar((1:length(MeanMicRelativeVelsDrive(:,i)))*ExposureTime,MeanMicRelativeVelsDrive(:,i),SEMMicRelativeVelsDrive(:,i),'b',Transparency); hold on;
                shadedErrorBar((1:length(MeanMicRelativeVelsHinder(:,i)))*ExposureTime,MeanMicRelativeVelsHinder(:,i),SEMMicRelativeVelsHinder(:,i),'r',Transparency); hold on;

                plot((1:length(MeanMicRelativeVelsDrive(:,i)))*ExposureTime,MeanMicRelativeVelsDrive(:,i),'b','LineWidth',2); hold on;
                plot((1:length(MeanMicRelativeVelsHinder(:,i)))*ExposureTime,MeanMicRelativeVelsHinder(:,i),'r','LineWidth',2);
            end
        end
        xlim(xLimitDistForces);
        ylim(yLimitDistForces);
        title('Average Kin. Vel. Relative to Mic. Over Time');
        set(gca,'FontSize',16,'FontWeight','b');
        xlabel('Time (s)');
        ylabel('Relative Velocity (nm/s)');
        
        % Save figures for KinVelocityOverTime
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Kin Velocity Over Time.jpg');
        savefig('Aggregate Images\Kin Velocity Over Time.fig');
        %% g) Plot Number Hindering vs Total Kinesins attached
        % Want all microtubule present?
        WantAllMicPresent = 0;      % 1 if we want only points with all microtubule present. 0 otherwise
        
        % Get parameters
        xText = 0.1;
        yText = 0.9;
        FontSize = 16;
        
        % Set Confidence Interval
        ConfidenceInterval = 0.95;
        
        % Get AllMicPresent parameter
        AllMicPresent = TimePoints(:,5);
        
        % Initialize MeanStdevSEM and FractionHinder
        MeanStdevSEM = zeros(max(TotalKin),3);
        FractionHinder = zeros(max(TotalKin),3);
        TotalCounts = zeros(max(TotalKin),1);
        
        for i = 1:max(TotalKin)
            if WantAllMicPresent == 0
                CurrentHinderingKin = HinderingKin(TotalKin==i);
                MeanStdevSEM(i,1) = mean(CurrentHinderingKin);
                MeanStdevSEM(i,2) = std(CurrentHinderingKin);
                MeanStdevSEM(i,3) = std(CurrentHinderingKin)/sqrt(length(CurrentHinderingKin));
                FractionHinder(i,1) = MeanStdevSEM(i,1)/i;
                FractionHinder(i,2) = MeanStdevSEM(i,2)/i;
                FractionHinder(i,3) = MeanStdevSEM(i,3)/i;
                TotalCounts(i) = length(CurrentHinderingKin);
            else
                CurrentHinderingKin = HinderingKin(TotalKin==i & AllMicPresent==1);
                MeanStdevSEM(i,1) = mean(CurrentHinderingKin);
                MeanStdevSEM(i,2) = std(CurrentHinderingKin);
                MeanStdevSEM(i,3) = std(CurrentHinderingKin)/sqrt(length(CurrentHinderingKin));
                FractionHinder(i,1) = MeanStdevSEM(i,1)/i;
                FractionHinder(i,2) = MeanStdevSEM(i,2)/i;
                FractionHinder(i,3) = MeanStdevSEM(i,3)/i;
                TotalCounts(i) = length(CurrentHinderingKin);
            end
        end
        
        % Plot number of hindering kinesin vs total kinesin
        figure;
        bar(1:max(TotalKin),MeanStdevSEM(:,1),'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
        errorbar(1:max(TotalKin),MeanStdevSEM(:,1),MeanStdevSEM(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
        title('No. of Hindering Kin. vs Total Kin.');
        ylabel('No. of Hindering Kinesin');
        xlabel('Total Kinesin');
        set(gca,'FontWeight','b','FontSize',16);
        TotalCount = sum(TotalCounts);
        text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
        % Save figures for number of hindering kinesin vs total kinesin
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Hinder No vs Total.jpg');
        savefig('Aggregate Images\Hinder No vs Total.fig');
        
        % Plot fraction of hindering kinesin vs total kinesin
        figure;
        bar(1:max(TotalKin),FractionHinder(:,1),'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
        errorbar(1:max(TotalKin),FractionHinder(:,1),FractionHinder(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
        title('Fraction of Hindering Kin. vs Total Kin.');
        ylabel('Fraction of Hindering Kinesin');
        xlabel('Total Kinesin');
        set(gca,'FontWeight','b','FontSize',16);
        text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
        % Save figures for fraction of hindering kinesin vs total kinesin
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Hinder Fraction vs Total.jpg');
        savefig('Aggregate Images\Hinder Fraction vs Total.fig');
        
        % Plot TotalCounts
        figure;
        bar(1:max(TotalKin),TotalCounts,'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
        %plot(1:max(TotalKin),TotalCounts,'LineWidth',1.5);
        title('No. of samples vs Total Kin.');
        set(gca,'FontWeight','b','FontSize',16);
        % Fit exponential
        f = fit((1:max(TotalKin))',TotalCounts,'exp1');
        CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
        Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
        h = plot(f,(1:max(TotalKin))',TotalCounts);
        set(h,'LineWidth',1.2,'Color','r');
        legend('off');
        text(0.25,yText,{['Kin. const.: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' kin'],...
            ['N = ' num2str(TotalCount)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
        text((1:max(TotalKin)),TotalCounts+max(TotalCounts)*0.05,num2str(TotalCounts),'FontSize',FontSize,'fontweight','b','HorizontalAlignment','center');
        ylabel('No. of samples');
        xlabel('Total Kinesin');
        
        % Save TotalCounts histogram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\No of Samples vs Total.jpg');
        savefig('Aggregate Images\No of Samples vs Total.fig');
            %% Simulation to estimate effect of unidentified kinesins
            % Get parameters
            FontSize = 16;
            xText = 0.1;
            yText = 0.9;
            
            % Generate simulation with a certain FractionHinder
%             TotalCounts = [15023;6450;2130;871;503;197;124;27;5]*5;
%             TotalCounts = (50:-3:1)*10000;
            expDwell = 2.5;
            x = (1:13)*1/expDwell;
            TotalCounts = round(exp(-x)*20500);
            SimulatedTimePoints = NaN(sum(TotalCounts),length(TotalCounts));
            CountIndex = 1;
            FractHinder = 0.25;
            for i = 1:length(TotalCounts)
                for j = 1:TotalCounts(i)
                    if i == 1
                        SimulatedTimePoints(CountIndex,1:i) = 1;
                    else
                        SimulatedTimePoints(CountIndex,1:i) = rand(1,i)>FractHinder;
                    end
                    CountIndex = CountIndex + 1;
                end
            end
            
            % Set Confidence Interval
            ConfidenceInterval = 0.95;
            
            % Calculate SimulatedDriving, Hindering and Total for initial
            % simulation
            SimulatedDriving = nansum(SimulatedTimePoints,2);
            SimulatedHindering = nansum(SimulatedTimePoints==0,2);
            SimulatedTotal = SimulatedDriving + SimulatedHindering;
            
            % Initialize MeanStdevSEM and FractionHinder
            MeanStdevSEM = zeros(max(SimulatedTotal),3);
            FractionHinder = zeros(max(SimulatedTotal),3);
            TotalCounts = zeros(max(SimulatedTotal),1);

            for i = 1:max(SimulatedTotal)
                CurrentHinderingKin = SimulatedHindering(SimulatedTotal==i);
                MeanStdevSEM(i,1) = mean(CurrentHinderingKin);
                MeanStdevSEM(i,2) = std(CurrentHinderingKin);
                MeanStdevSEM(i,3) = std(CurrentHinderingKin)/sqrt(length(CurrentHinderingKin));
                FractionHinder(i,1) = MeanStdevSEM(i,1)/i;
                FractionHinder(i,2) = MeanStdevSEM(i,2)/i;
                FractionHinder(i,3) = MeanStdevSEM(i,3)/i;
                TotalCounts(i) = length(CurrentHinderingKin);
            end
            TotalCount = sum(TotalCounts);
            
            % Plot fraction of hindering kinesin vs total kinesin
            figure;
            bar(1:max(SimulatedTotal),FractionHinder(:,1),'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
            errorbar(1:max(SimulatedTotal),FractionHinder(:,1),FractionHinder(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
            title('Fraction of Hindering Kin. vs Total Kin.');
            ylabel('Fraction of Hindering Kinesin');
            xlabel('Total Kinesin');
            set(gca,'FontWeight','b','FontSize',16);
            text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
            % Save figures for fraction of hindering kinesin vs total kinesin
            if exist('Aggregate Images\Simulation','dir')~=7; mkdir('Aggregate Images\Simulation'); end
            print('-djpeg','-r300','Aggregate Images\Simulation\Hinder Fraction vs Total (Initial).jpg');
            savefig('Aggregate Images\Simulation\Hinder Fraction vs Total (Initial).fig');

            % Plot TotalCounts
            figure;
            bar(1:max(SimulatedTotal),TotalCounts,'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
            %plot(1:max(TotalKin),TotalCounts,'LineWidth',1.5);
            title('No. of samples vs Total Kin.');
            set(gca,'FontWeight','b','FontSize',16);
            % Fit exponential
            f = fit((1:max(SimulatedTotal))',TotalCounts,'exp1');
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,(1:max(SimulatedTotal))',TotalCounts);
            set(h,'LineWidth',1.2,'Color','r');
            legend('off');
            text(0.25,yText,{['Kin. const.: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' kin'],...
                ['N = ' num2str(TotalCount)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            text((1:max(SimulatedTotal)),TotalCounts+max(TotalCounts)*0.05,num2str(TotalCounts),'FontSize',FontSize,'fontweight','b','HorizontalAlignment','center');
            ylim([0 max(TotalCounts)*1.1]);
            ylabel('No. of samples');
            xlabel('Total Kinesin');
            % Save plot
            if exist('Aggregate Images\Simulation','dir')~=7; mkdir('Aggregate Images\Simulation'); end
            print('-djpeg','-r300','Aggregate Images\Simulation\Total Count (Initial).jpg');
            savefig('Aggregate Images\Simulation\Total Count (Initial).fig');
            
            
            % Simulate lost kinesins
            FractionLost = 0.65;
            LostMatrix = double(rand(sum(TotalCounts),length(TotalCounts))>FractionLost);
            LostMatrix(LostMatrix==0) = NaN;
            SimulatedTimePoints = SimulatedTimePoints.*LostMatrix;
            SimulatedDriving = nansum(SimulatedTimePoints,2);
            SimulatedHindering = nansum(SimulatedTimePoints==0,2);
            SimulatedTotal = SimulatedDriving + SimulatedHindering;
            
            % Initialize MeanStdevSEM and FractionHinder
            MeanStdevSEM = zeros(max(SimulatedTotal),3);
            FractionHinder = zeros(max(SimulatedTotal),3);
            TotalCounts = zeros(max(SimulatedTotal),1);

            for i = 1:max(SimulatedTotal)
                CurrentHinderingKin = SimulatedHindering(SimulatedTotal==i);
                MeanStdevSEM(i,1) = mean(CurrentHinderingKin);
                MeanStdevSEM(i,2) = std(CurrentHinderingKin);
                MeanStdevSEM(i,3) = std(CurrentHinderingKin)/sqrt(length(CurrentHinderingKin));
                FractionHinder(i,1) = MeanStdevSEM(i,1)/i;
                FractionHinder(i,2) = MeanStdevSEM(i,2)/i;
                FractionHinder(i,3) = MeanStdevSEM(i,3)/i;
                TotalCounts(i) = length(CurrentHinderingKin);
            end
            
            % Plot number of hindering kinesin vs total kinesin
            figure;
            bar(1:max(SimulatedTotal),MeanStdevSEM(:,1),'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
            errorbar(1:max(SimulatedTotal),MeanStdevSEM(:,1),MeanStdevSEM(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
            title('No. of Hindering Kin. vs Total Kin.');
            ylabel('No. of Hindering Kinesin');
            xlabel('Total Kinesin');
            set(gca,'FontWeight','b','FontSize',16);
            TotalCount = sum(TotalCounts);
            text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
            % Save figures for number of hindering kinesin vs total kinesin
            if exist('Aggregate Images\Simulation','dir')~=7; mkdir('Aggregate Images\Simulation'); end
            print('-djpeg','-r300','Aggregate Images\Simulation\Hinder No vs Total (Final).jpg');
            savefig('Aggregate Images\Simulation\Hinder No vs Total (Final).fig');

            % Plot fraction of hindering kinesin vs total kinesin
            figure;
            bar(1:max(SimulatedTotal),FractionHinder(:,1),'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
            errorbar(1:max(SimulatedTotal),FractionHinder(:,1),FractionHinder(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
            title('Fraction of Hindering Kin. vs Total Kin.');
            ylabel('Fraction of Hindering Kinesin');
            xlabel('Total Kinesin');
            set(gca,'FontWeight','b','FontSize',16);
            text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
            % Save figures for fraction of hindering kinesin vs total kinesin
            if exist('Aggregate Images\Simulation','dir')~=7; mkdir('Aggregate Images\Simulation'); end
            print('-djpeg','-r300','Aggregate Images\Simulation\Hinder Fraction vs Total (Final).jpg');
            savefig('Aggregate Images\Simulation\Hinder Fraction vs Total (Final).fig');

            % Plot TotalCounts
            figure;
            bar(1:max(SimulatedTotal),TotalCounts,'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
            %plot(1:max(TotalKin),TotalCounts,'LineWidth',1.5);
            title('No. of samples vs Total Kin.');
            set(gca,'FontWeight','b','FontSize',16);
            % Fit exponential
            f = fit((1:max(SimulatedTotal))',TotalCounts,'exp1');
            CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
            Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
            h = plot(f,(1:max(SimulatedTotal))',TotalCounts);
            set(h,'LineWidth',1.2,'Color','r');
            legend('off');
            text(0.25,yText,{['Kin. const.: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' kin'],...
                ['N = ' num2str(TotalCount)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
            text((1:max(SimulatedTotal)),TotalCounts+max(TotalCounts)*0.05,num2str(TotalCounts),'FontSize',FontSize,'fontweight','b','HorizontalAlignment','center');
            ylim([0 max(TotalCounts)*1.1]);
            ylabel('No. of samples');
            xlabel('Total Kinesin');
            % Save plot
            if exist('Aggregate Images\Simulation','dir')~=7; mkdir('Aggregate Images\Simulation'); end
            print('-djpeg','-r300','Aggregate Images\Simulation\Total Count (Final).jpg');
            savefig('Aggregate Images\Simulation\Total Count (Final).fig');
        %% h) Plot velocity vs total, driving and hindering kinesins
        % Want all microtubule present?
        WantAllMicPresent = 0;      % 1 if we want only points with all microtubule present. 0 otherwise
        
        % Get parameters
        xText = 0.1;
        yText = 0.95;
        FontSize = 16;
        binSpacing = 50;
        xLimitVelHist = [-300 1600];
%         HistogramColor = [0.3 0.3 0.8];
%         HistogramColor = [0.3 0.8 0.3];
%         HistogramColor = [0.8 0.8 0.3];
%         HistogramColor = [0.5 0.3 0.6];
        HistogramColor = [0.6 0.5 0.3];
        textFont = 14;
        UseGaussianFit = 1;     % 1 to use GaussianFit for bulk mic velocity plot, 2 for MLE
        centerGauss = [0;800];  % Gaussian center if UseGaussianFit == 1
        stdGauss = [100;100];   % Gaussian std if UseGaussianFit == 1
        OverridePeakNo = 3;     % (For MLE) 0 to let code find the number of peaks. > 0  to define a fixed number of peaks
        UseStdev = 1;           % 1 to use std. Otherwise to use SEM    
        
        % Plot red line or red curve for center of Gaussian or Gaussian
        % components?
        PlotRedLine = 0;
        PlotRedCurve = 0;
        
        % Set Confidence Interval
        ConfidenceInterval = 0.95;
        
        % Get Velocity and AllMicPresent parameter
        Velocity = TimePoints(:,6);
        AllMicPresent = TimePoints(:,5);
        
        % Initialize MeanStdevSEM and FractionHinder
        VelTotMeanStdevSEM = zeros(max(TotalKin),3);
        VelDriveMeanStdevSEM = zeros(max(DrivingKin),3);
        VelHinderMeanStdevSEM = zeros(max(HinderingKin),3);
        TotTotalCounts = zeros(max(TotalKin),1);
        DriveTotalCounts = zeros(max(DrivingKin),1);
        HinderTotalCounts = zeros(max(HinderingKin),1);
        
        if WantAllMicPresent == 1
            figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
            ha1 = tight_subplot(max(TotalKin(AllMicPresent==1)),1,[.003 .003],[.08 .05],[.1 .03]);
            edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
            figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
            ha2 = tight_subplot(max(DrivingKin(AllMicPresent==1)),1,[.003 .003],[.08 .05],[.1 .03]);
            figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
            ha3 = tight_subplot(max(HinderingKin(AllMicPresent==1)),1,[.003 .003],[.08 .05],[.1 .03]);
        else
            figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
            ha1 = tight_subplot(max(TotalKin),1,[.003 .003],[.08 .05],[.1 .03]);
            edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
            figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
            ha2 = tight_subplot(max(DrivingKin),1,[.003 .003],[.08 .05],[.1 .03]);
            figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
            ha3 = tight_subplot(max(HinderingKin),1,[.003 .003],[.08 .05],[.1 .03]);
        end
        
        for i = 1:max(TotalKin)
            if WantAllMicPresent == 0
                CurrentVel = Velocity(TotalKin==i);
                CurrentVel = CurrentVel(~isnan(CurrentVel));
                VelTotMeanStdevSEM(i,1) = mean(CurrentVel);
                VelTotMeanStdevSEM(i,2) = std(CurrentVel);
                VelTotMeanStdevSEM(i,3) = std(CurrentVel)/sqrt(length(CurrentVel));
                TotTotalCounts(i) = length(CurrentVel);
                axes(ha1(i));
                histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
                text(0.2,0.7,{['Mean: ' num2str(round(mean(VelTotMeanStdevSEM(i,1)))) '  ' num2str(round(VelTotMeanStdevSEM(i,3))) ' nm/s'],...
                    ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
                xlim(xLimitVelHist);
                ylimit = ylim; plot([VelTotMeanStdevSEM(i,1) VelTotMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
                ylim(ylimit);
                ylabel(i);
                if i == max(TotalKin)
                    set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                    xlabel('Velocity (nm/s)');
                elseif i == 1
                    set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    title('Vel. Hist. at Varying Total Kin.');
                else
                    set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                end
                
                if i <= max(DrivingKin)
                    CurrentVel = Velocity(DrivingKin==i);
                    VelDriveMeanStdevSEM(i,1) = nanmean(CurrentVel);
                    VelDriveMeanStdevSEM(i,2) = nanstd(CurrentVel);
                    VelDriveMeanStdevSEM(i,3) = nanstd(CurrentVel)/sqrt(length(~isnan(CurrentVel)));
                    DriveTotalCounts(i) = length(CurrentVel);
                    
                    axes(ha2(i));
                    histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
                    text(0.2,0.7,{['Mean: ' num2str(round(mean(VelDriveMeanStdevSEM(i,1)))) '  ' num2str(round(VelDriveMeanStdevSEM(i,3))) ' nm/s'],...
                        ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
                    xlim(xLimitVelHist);
                    ylimit = ylim; plot([VelDriveMeanStdevSEM(i,1) VelDriveMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
                    ylim(ylimit);
                    ylabel(i);
                    if i == max(DrivingKin)
                        set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                        xlabel('Velocity (nm/s)');
                    elseif i == 1
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                        title('Vel. Hist. at Varying Driving Kin.');
                    else
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    end
                end
                
                if i <= max(HinderingKin)
                    CurrentVel = Velocity(HinderingKin==i);
                    VelHinderMeanStdevSEM(i,1) = nanmean(CurrentVel);
                    VelHinderMeanStdevSEM(i,2) = nanstd(CurrentVel);
                    VelHinderMeanStdevSEM(i,3) = nanstd(CurrentVel)/sqrt(length(~isnan(CurrentVel)));
                    HinderTotalCounts(i) = length(CurrentVel);
                    
                    axes(ha3(i));
                    histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
                    text(0.2,0.7,{['Mean: ' num2str(round(mean(VelHinderMeanStdevSEM(i,1)))) '  ' num2str(round(VelHinderMeanStdevSEM(i,3))) ' nm/s'],...
                        ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
                    xlim(xLimitVelHist);
                    ylimit = ylim; plot([VelHinderMeanStdevSEM(i,1) VelHinderMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
                    ylim(ylimit);
                    ylabel(i);
                    if i == max(HinderingKin)
                        set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                        xlabel('Velocity (nm/s)');
                    elseif i == 1
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                        title('Vel. Hist. at Varying Hindering Kin.');
                    else
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    end
                end
            else
                CurrentVel = Velocity(TotalKin==i & AllMicPresent==1);
                CurrentVel = CurrentVel(~isnan(CurrentVel));
                VelTotMeanStdevSEM(i,1) = mean(CurrentVel);
                VelTotMeanStdevSEM(i,2) = std(CurrentVel);
                VelTotMeanStdevSEM(i,3) = std(CurrentVel)/sqrt(length(CurrentVel));
                TotTotalCounts(i) = length(CurrentVel);
                
                axes(ha1(i));
                histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
                text(0.2,0.7,{['Mean: ' num2str(round(mean(VelTotMeanStdevSEM(i,1)))) '  ' num2str(round(VelTotMeanStdevSEM(i,3))) ' nm/s'],...
                    ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
                xlim(xLimitVelHist);
                ylimit = ylim; plot([VelTotMeanStdevSEM(i,1) VelTotMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
                ylim(ylimit);
                ylabel(i);
                if i == max(TotalKin(AllMicPresent==1))
                    set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                    xlabel('Velocity (nm/s)');
                elseif i == 1
                    set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    title('Vel. Hist. at Varying Total Kin.');
                else
                    set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                end
                
                if i <= max(DrivingKin(AllMicPresent==1))
                    CurrentVel = Velocity(DrivingKin==i & AllMicPresent==1);
                    VelDriveMeanStdevSEM(i,1) = nanmean(CurrentVel);
                    VelDriveMeanStdevSEM(i,2) = nanstd(CurrentVel);
                    VelDriveMeanStdevSEM(i,3) = nanstd(CurrentVel)/sqrt(length(~isnan(CurrentVel)));
                    DriveTotalCounts(i) = length(CurrentVel);
                    
                    axes(ha2(i));
                    histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
                    text(0.2,0.7,{['Mean: ' num2str(round(mean(VelDriveMeanStdevSEM(i,1)))) '  ' num2str(round(VelDriveMeanStdevSEM(i,3))) ' nm/s'],...
                        ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
                    xlim(xLimitVelHist);
                    ylimit = ylim; plot([VelDriveMeanStdevSEM(i,1) VelDriveMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
                    ylim(ylimit);
                    ylabel(i);
                    if i == max(DrivingKin)
                        set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                        xlabel('Velocity (nm/s)');
                    elseif i == 1
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                        title('Vel. Hist. at Varying Driving Kin.');
                    else
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    end
                end
                
                if i <= max(HinderingKin(AllMicPresent==1))
                    CurrentVel = Velocity(HinderingKin==i & AllMicPresent==1);
                    VelHinderMeanStdevSEM(i,1) = nanmean(CurrentVel);
                    VelHinderMeanStdevSEM(i,2) = nanstd(CurrentVel);
                    VelHinderMeanStdevSEM(i,3) = nanstd(CurrentVel)/sqrt(length(~isnan(CurrentVel)));
                    HinderTotalCounts(i) = length(CurrentVel);
                    
                    axes(ha3(i));
                    histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
                    text(0.2,0.7,{['Mean: ' num2str(round(mean(VelHinderMeanStdevSEM(i,1)))) '  ' num2str(round(VelHinderMeanStdevSEM(i,3))) ' nm/s'],...
                        ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
                    xlim(xLimitVelHist);
                    ylimit = ylim; plot([VelHinderMeanStdevSEM(i,1) VelHinderMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
                    ylim(ylimit);
                    ylabel(i);
                    if i == max(HinderingKin)
                        set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                        xlabel('Velocity (nm/s)');
                    elseif i == 1
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                        title('Vel. Hist. at Varying Hindering Kin.');
                    else
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    end
                end
            end
        end
        
        % Save ha1, ha2 and ha3
        axes(ha1);
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Total Kin.jpg');
        savefig('Aggregate Images\Velocity histogram at Varying Total Kin.fig');
        axes(ha2);
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Driving Kin.jpg');
        savefig('Aggregate Images\Velocity histogram at Varying Driving Kin.fig');
        axes(ha3);
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Hindering Kin.jpg');
        savefig('Aggregate Images\Velocity histogram at Varying Hindering Kin.fig');
        
        % Plot bulk velocity histogram
        if 1
        if WantAllMicPresent == 0
            CurrentVel = Velocity;
            CurrentVel = CurrentVel(~isnan(CurrentVel));
            figure('units','normalized','outerposition',[0.05 0.5 0.6 0.3]);
            hHist = histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
            binCenter = (edges(2:end)+edges(1:end-1))/2;
            HistData = hHist.Values;
            
            if UseGaussianFit == 1
%                 centerGauss = [0;800];
%                 stdGauss = [100;100];
                [~,IGauss] = min(abs(centerGauss-binCenter),[],2);
                peakGauss = HistData(IGauss)';

                p1 = [peakGauss centerGauss stdGauss];
                f = @(p,x)sum(p(:,1).*exp(-((x-p(:,2))./(sqrt(2)*p(:,3))).^2),1);
                pfit = lsqcurvefit(f,p1,binCenter,HistData); 
                f1gauss = @(p,x)p(1)*exp(-((x-p(2))/(sqrt(2)*p(3))).^2);
                xfit = floor(edges(1)):0.5:ceil(edges(end));
                yfit = zeros(size(p1,1),length(xfit));
                AreaGauss = zeros(size(p1,1),1);
                for ii = 1:size(p1,1)
                    yfit(ii,:) = f1gauss(pfit(ii,:),xfit);
                    plot(xfit,yfit(ii,:),'--k','LineWidth',2);
                    AreaGauss(ii) = sum(f1gauss(pfit(ii,:),binCenter));
                end

                sem = pfit(:,3)./sqrt(AreaGauss);
                if UseStdev == 1; Err = round(pfit(:,3)*10)/10; else; Err = round(sem*10)/10; end; 

                ylimit = ylim; ylimit = [ylimit(1) ylimit(2)*1.4];

                xAnnotateOffset = 0;       % x Offset for texts in step-size histogram
                yAnnotateOffset = 0;       % y Offset for texts in step-size histogram
                ReportStdev = 1;    % 1 to report Stdev. Any other to report all peaks but no Stdev
                UseStdev = 1;       % 1 if we want to use Stdev. 0 if we want to use Standard Error of Mean (SEM)
                AnnotationSize = 14;
                for iter = 1:size(p1,1)
                    if ReportStdev == 1
                        Text = str2mat([sprintf('%1.0f',pfit(iter,2))  '  ' num2str(round(Err(iter))) ' (' num2str(round(AreaGauss(iter)/sum(AreaGauss)*100)) '%)']);
                        text(pfit(iter,2)+xAnnotateOffset,0.2*ylimit(2)+pfit(iter,1)+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    else
                        Text = str2mat([sprintf('%1.0f',pfit(iter,2)) ' (' num2str(round(AreaGauss(iter)/sum(AreaGauss)*100)) '%)']);
                        text(pfit(iter,2)+xAnnotateOffset,0.2*ylimit(2)+pfit(iter,1)+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    end
                end 
            else
                % Maximum likelihood fit (With repeated fit for one parameter number and choose lowest BIC)
    %             OverridePeakNo = 3;     % 0 to let code find the number of peaks. > 0  to define a fixed number of peaks
                MinPeakNo = 2;      % Decide the minimum peak number if there is no override
                MaxPeakNo = 3;      % Decide the minimum peak number if there is no override
                RepeatFit = 4;      % Number of repetition for each peak values for MLE
                DifferenceThreshold = 150;    % The difference between center of first and second peak below which the fit is rejected. This is to prevent peaks with the same center
                xAnnotateOffset = 0;       % x Offset for texts in step-size histogram
                yAnnotateOffset = 0;       % y Offset for texts in step-size histogram
                ReportStdev = 1;    % 1 to report Stdev. Any other to report all peaks but no Stdev
                UseStdev = 1;       % 1 if we want to use Stdev. 0 if we want to use Standard Error of Mean (SEM)
                AnnotationSize = 14;
                xfit = edges(1):edges(end);
                Length = length(CurrentVel');

                if OverridePeakNo == 0
                    FitLengths = (MaxPeakNo-MinPeakNo+1)*RepeatFit;   % Total number of fits
                    BIC = zeros(FitLengths,2);
                    obj2 = cell(1,FitLengths);
                    FitIndex = 0;
                    for PeakNo = MinPeakNo:MaxPeakNo
                        for Repeat = 1:RepeatFit
                            try
                                FitIndex = FitIndex + 1;
                                obj2{FitIndex} = gmdistribution.fit(CurrentVel,PeakNo);
                                BIC(FitIndex,:)= [obj2{FitIndex}.BIC PeakNo];
                            catch
                                obj2{FitIndex} = [];
                                BIC(FitIndex,:)= [Inf PeakNo];
                            end
                        end
                    end
                    [BICSorted,SortIndex] = sort(BIC(:,1));
                    objSorted = obj2(SortIndex);
                    PeakSorted = BIC(SortIndex,2);
                else
                    BIC = zeros(RepeatFit,2);
                    obj2 = cell(1,RepeatFit);
                    FitIndex = 0;
                    for Repeat = 1:RepeatFit
                        try
                            FitIndex = FitIndex + 1;
                            obj2{FitIndex} = gmdistribution.fit(CurrentVel,OverridePeakNo);
                            BIC(FitIndex,:)= [obj2{FitIndex}.BIC OverridePeakNo];
                        catch
                            obj2{FitIndex} = [];
                            BIC(FitIndex,:)= [Inf OverridePeakNo];
                        end
                    end
                    [BICSorted,SortIndex] = sort(BIC(:,1));
                    objSorted = obj2(SortIndex);
                    PeakSorted = BIC(SortIndex,2);
                end

                if min(BIC(:,1))==Inf
                    BIC = zeros(RepeatFit,2);
                    obj2 = cell(1,RepeatFit);
                    FitIndex = 0;
                    for Repeat = 1:RepeatFit
                        try
                            FitIndex = FitIndex + 1;
                            obj2{FitIndex} = gmdistribution.fit(CurrentVel,1);
                            BIC(FitIndex,:)= [obj2{FitIndex}.BIC 1];
                        catch
                            obj2{FitIndex} = [];
                            BIC(FitIndex,:)= [Inf 1];
                        end
                    end
                    [BICSorted,SortIndex] = sort(BIC(:,1));
                    objSorted = obj2(SortIndex);
                    PeakSorted = BIC(SortIndex,2);
                end

                PeakToTry = 1;
                NoError = 0;
                while NoError == 0
                    try
                        obj = objSorted{PeakToTry};
                        Amplitude = zeros(PeakSorted(PeakToTry),1);
                        for iter = 1:PeakSorted(PeakToTry)
                            Amplitude(iter) = pdf(obj,obj.mu(iter))*Length;
                        end

                        [~,PeakMax] = max(Amplitude);
                        [Amplitude,AmpInd]=sort(Amplitude,'descend');
                        Peaks = obj.mu; Peaks = Peaks(AmpInd);
                        Stdev = permute(obj.Sigma,[3 2 1]); Stdev = Stdev(AmpInd);

                        if PeakSorted(PeakToTry) > 1
                            if abs(Peaks(2)-Peaks(1)) > DifferenceThreshold
                                NoError = 1;
                            else
                                PeakToTry = PeakToTry + 1;
                            end
                        else
                            NoError = 1;
                        end
                    catch
                        PeakToTry = PeakToTry + 1;
                    end
                end

                Peaks = obj.mu;
                Stdev = permute(sqrt(obj.Sigma),[3 2 1]);
                Amplitude = obj.PComponents*Length./(Stdev'*sqrt(2*pi))*binSpacing;
                [Amplitude,AmpInd]=sort(Amplitude,'descend');
                Peaks = Peaks(AmpInd);
                Stdev = Stdev(AmpInd);

                yfit = (pdf(obj,xfit')*Length)';
                f1gauss = @(p,x)p(1)*exp(-((x-p(2))/(sqrt(2)*p(3))).^2);
                NoOfPeaks = length(Peaks);
                nGauss = Amplitude'.*Stdev*sqrt(2*pi);
                sem = Stdev./sqrt(nGauss);

                % Append with zeros if there are less than 3 peaks
                if NoOfPeaks < 3
                    Peaks = [Peaks; zeros(3-NoOfPeaks,1)];
                    nGauss = [nGauss; zeros(3-NoOfPeaks,1)];
                    Stdev = [Stdev; zeros(3-NoOfPeaks,1)];
                    sem = [sem; zeros(3-NoOfPeaks,1)];
                end

                plot(xfit,yfit*binSpacing,'--k','LineWidth',1);
                yfitgauss = zeros(length(xfit),NoOfPeaks);
                if UseStdev == 1; Err = round(Stdev*10)/10; else; Err = round(sem*10)/10; end; 
                ylimit = ylim; ylimit = [ylimit(1) ylimit(2)*1.3];
                for iter = 1:NoOfPeaks
                    yfitgauss(:,iter) = f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],xfit);
                    if PlotRedCurve == 1
                        plot(xfit,yfitgauss(:,iter),'--r','LineWidth',1);
                    end
                    if PlotRedLine == 1
                        plot([Peaks(iter) Peaks(iter)],ylimit,'r','LineWidth',1);
                    end
                    if ReportStdev == 1
                        Text = str2mat([sprintf('%1.0f',Peaks(iter))  '  ' num2str(round(Err(iter))) ' (' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                        text(Peaks(iter)+xAnnotateOffset,0.24*ylimit(2)+f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    else
                        Text = str2mat([sprintf('%1.0f',Peaks(iter)) ' (' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                        text(Peaks(iter)+xAnnotateOffset,0.24*ylimit(2)+f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    end
                end
            end
            text(0.8,0.8,['N = ' num2str(length(CurrentVel))],'FontSize',textFont,'fontweight','b','Units','normalized');
            xlim(xLimitVelHist);
            ylim(ylimit);
            ylabel('Count');
            set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
            xlabel('Velocity (nm/s)');
            title('Bulk Microtubule Velocity Histogram');
        else
            CurrentVel = Velocity;
            CurrentVel = CurrentVel(~isnan(CurrentVel));
            figure('units','normalized','outerposition',[0.05 0.5 0.6 0.3]);
            hHist = histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
            binCenter = (edges(2:end)+edges(1:end-1))/2;
            HistData = hHist.Values;
            
            if UseGaussianFit == 1
%                 centerGauss = [0;800];
%                 stdGauss = [100;100];
                [~,IGauss] = min(abs(centerGauss-binCenter),[],2);
                peakGauss = HistData(IGauss)';

                p1 = [peakGauss centerGauss stdGauss];
                f = @(p,x)sum(p(:,1).*exp(-((x-p(:,2))./(sqrt(2)*p(:,3))).^2),1);
                pfit = lsqcurvefit(f,p1,binCenter,HistData); 
                f1gauss = @(p,x)p(1)*exp(-((x-p(2))/(sqrt(2)*p(3))).^2);
                xfit = floor(edges(1)):0.5:ceil(edges(end));
                yfit = zeros(size(p1,1),length(xfit));
                AreaGauss = zeros(size(p1,1),1);
                for ii = 1:size(p1,1)
                    yfit(ii,:) = f1gauss(pfit(ii,:),xfit);
                    plot(xfit,yfit(ii,:),'--k','LineWidth',2);
                    AreaGauss(ii) = sum(f1gauss(pfit(ii,:),binCenter));
                end

                sem = pfit(:,3)./sqrt(AreaGauss);
                if UseStdev == 1; Err = round(pfit(:,3)*10)/10; else; Err = round(sem*10)/10; end; 

                ylimit = ylim; ylimit = [ylimit(1) ylimit(2)*1.4];

                xAnnotateOffset = 0;       % x Offset for texts in step-size histogram
                yAnnotateOffset = 0;       % y Offset for texts in step-size histogram
                ReportStdev = 1;    % 1 to report Stdev. Any other to report all peaks but no Stdev
                UseStdev = 1;       % 1 if we want to use Stdev. 0 if we want to use Standard Error of Mean (SEM)
                AnnotationSize = 14;
                for iter = 1:size(p1,1)
                    if ReportStdev == 1
                        Text = str2mat([sprintf('%1.0f',pfit(iter,2))  '  ' num2str(round(Err(iter))) ' (' num2str(round(AreaGauss(iter)/sum(AreaGauss)*100)) '%)']);
                        text(pfit(iter,2)+xAnnotateOffset,0.2*ylimit(2)+pfit(iter,1)+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    else
                        Text = str2mat([sprintf('%1.0f',pfit(iter,2)) ' (' num2str(round(AreaGauss(iter)/sum(AreaGauss)*100)) '%)']);
                        text(pfit(iter,2)+xAnnotateOffset,0.2*ylimit(2)+pfit(iter,1)+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    end
                end 
            else
    %             OverridePeakNo = 2;     % 0 to let code find the number of peaks. > 0  to define a fixed number of peaks
                MinPeakNo = 2;      % Decide the minimum peak number if there is no override
                MaxPeakNo = 3;      % Decide the minimum peak number if there is no override
                RepeatFit = 4;      % Number of repetition for each peak values for MLE
                DifferenceThreshold = 150;    % The difference between center of first and second peak below which the fit is rejected. This is to prevent peaks with the same center
                xAnnotateOffset = 0;       % x Offset for texts in step-size histogram
                yAnnotateOffset = 0;       % y Offset for texts in step-size histogram
                ReportStdev = 1;    % 1 to report Stdev. Any other to report all peaks but no Stdev
                UseStdev = 1;       % 1 if we want to use Stdev. 0 if we want to use Standard Error of Mean (SEM)
                AnnotationSize = 14;
                xfit = edges(1):edges(end);
                Length = length(CurrentVel');

                if OverridePeakNo == 0
                    FitLengths = (MaxPeakNo-MinPeakNo+1)*RepeatFit;   % Total number of fits
                    BIC = zeros(FitLengths,2);
                    obj2 = cell(1,FitLengths);
                    FitIndex = 0;
                    for PeakNo = MinPeakNo:MaxPeakNo
                        for Repeat = 1:RepeatFit
                            try
                                FitIndex = FitIndex + 1;
                                obj2{FitIndex} = gmdistribution.fit(CurrentVel,PeakNo);
                                BIC(FitIndex,:)= [obj2{FitIndex}.BIC PeakNo];
                            catch
                                obj2{FitIndex} = [];
                                BIC(FitIndex,:)= [Inf PeakNo];
                            end
                        end
                    end
                    [BICSorted,SortIndex] = sort(BIC(:,1));
                    objSorted = obj2(SortIndex);
                    PeakSorted = BIC(SortIndex,2);
                else
                    BIC = zeros(RepeatFit,2);
                    obj2 = cell(1,RepeatFit);
                    FitIndex = 0;
                    for Repeat = 1:RepeatFit
                        try
                            FitIndex = FitIndex + 1;
                            obj2{FitIndex} = gmdistribution.fit(CurrentVel,OverridePeakNo);
                            BIC(FitIndex,:)= [obj2{FitIndex}.BIC OverridePeakNo];
                        catch
                            obj2{FitIndex} = [];
                            BIC(FitIndex,:)= [Inf OverridePeakNo];
                        end
                    end
                    [BICSorted,SortIndex] = sort(BIC(:,1));
                    objSorted = obj2(SortIndex);
                    PeakSorted = BIC(SortIndex,2);
                end

                if min(BIC(:,1))==Inf
                    BIC = zeros(RepeatFit,2);
                    obj2 = cell(1,RepeatFit);
                    FitIndex = 0;
                    for Repeat = 1:RepeatFit
                        try
                            FitIndex = FitIndex + 1;
                            obj2{FitIndex} = gmdistribution.fit(CurrentVel,1);
                            BIC(FitIndex,:)= [obj2{FitIndex}.BIC 1];
                        catch
                            obj2{FitIndex} = [];
                            BIC(FitIndex,:)= [Inf 1];
                        end
                    end
                    [BICSorted,SortIndex] = sort(BIC(:,1));
                    objSorted = obj2(SortIndex);
                    PeakSorted = BIC(SortIndex,2);
                end

                PeakToTry = 1;
                NoError = 0;
                while NoError == 0
                    try
                        obj = objSorted{PeakToTry};
                        Amplitude = zeros(PeakSorted(PeakToTry),1);
                        for iter = 1:PeakSorted(PeakToTry)
                            Amplitude(iter) = pdf(obj,obj.mu(iter))*Length;
                        end

                        [~,PeakMax] = max(Amplitude);
                        [Amplitude,AmpInd]=sort(Amplitude,'descend');
                        Peaks = obj.mu; Peaks = Peaks(AmpInd);
                        Stdev = permute(obj.Sigma,[3 2 1]); Stdev = Stdev(AmpInd);

                        if PeakSorted(PeakToTry) > 1
                            if abs(Peaks(2)-Peaks(1)) > DifferenceThreshold
                                NoError = 1;
                            else
                                PeakToTry = PeakToTry + 1;
                            end
                        else
                            NoError = 1;
                        end
                    catch
                        PeakToTry = PeakToTry + 1;
                    end
                end

                Peaks = obj.mu;
                Stdev = permute(sqrt(obj.Sigma),[3 2 1]);
                Amplitude = obj.PComponents*Length./(Stdev'*sqrt(2*pi))*binSpacing;
                [Amplitude,AmpInd]=sort(Amplitude,'descend');
                Peaks = Peaks(AmpInd);
                Stdev = Stdev(AmpInd);

                yfit = (pdf(obj,xfit')*Length)';
                f1gauss = @(p,x)p(1)*exp(-((x-p(2))/(sqrt(2)*p(3))).^2);
                NoOfPeaks = length(Peaks);
                nGauss = Amplitude'.*Stdev*sqrt(2*pi);
                sem = Stdev./sqrt(nGauss);

                % Append with zeros if there are less than 3 peaks
                if NoOfPeaks < 3
                    Peaks = [Peaks; zeros(3-NoOfPeaks,1)];
                    nGauss = [nGauss; zeros(3-NoOfPeaks,1)];
                    Stdev = [Stdev; zeros(3-NoOfPeaks,1)];
                    sem = [sem; zeros(3-NoOfPeaks,1)];
                end

                plot(xfit,yfit*binSpacing,'--k','LineWidth',1);
                yfitgauss = zeros(length(xfit),NoOfPeaks);
                if UseStdev == 1; Err = round(Stdev*10)/10; else; Err = round(sem*10)/10; end; 
                ylimit = ylim; ylimit = [ylimit(1) ylimit(2)*1.3];
                for iter = 1:NoOfPeaks
                    yfitgauss(:,iter) = f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],xfit);
                    if PlotRedCurve == 1
                        plot(xfit,yfitgauss(:,iter),'--r','LineWidth',1);
                    end
                    if PlotRedLine == 1
                        plot([Peaks(iter) Peaks(iter)],ylimit,'r','LineWidth',1);
                    end
                    if ReportStdev == 1
                        Text = str2mat([sprintf('%1.0f',Peaks(iter))  '  ' num2str(round(Err(iter))) ' (' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                        text(Peaks(iter)+xAnnotateOffset,0.24*ylimit(2)+f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    else
                        Text = str2mat([sprintf('%1.0f',Peaks(iter)) ' (' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                        text(Peaks(iter)+xAnnotateOffset,0.24*ylimit(2)+f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                    end
                end
            end
            text(0.8,0.8,['N = ' num2str(length(CurrentVel))],'FontSize',textFont,'fontweight','b','Units','normalized');
            xlim(xLimitVelHist);
            ylim(ylimit);
            ylabel('Count');
            set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
            xlabel('Velocity (nm/s)');
            title('Bulk Microtubule Velocity Histogram');
        end
        % Save plot
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\BulkMicVelocity.jpg');
        savefig('Aggregate Images\BulkMicVelocity.fig');
        end
        
        % Plot velocity vs total kinesin
        figure;
        bar(1:max(TotalKin),VelTotMeanStdevSEM(:,1),'FaceColor',HistogramColor,'LineWidth',1.5); hold on;
        errorbar(1:max(TotalKin),VelTotMeanStdevSEM(:,1),VelTotMeanStdevSEM(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
        title('Mic. Velocity vs Total Kin.');
        ylabel('Mic. Velocity');
        xlabel('Total Kinesin');
        set(gca,'FontWeight','b','FontSize',16);
        TotalCount = sum(TotTotalCounts);
        text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
        ylim([0 max(VelTotMeanStdevSEM(:,1))*1.2]);
        % Save plot
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Velocity vs Total.jpg');
        savefig('Aggregate Images\Velocity vs Total.fig');
        
        % Plot velocity vs driving kinesin
        figure;
        bar(1:max(DrivingKin),VelDriveMeanStdevSEM(:,1),'FaceColor',HistogramColor,'LineWidth',1.5); hold on;
        errorbar(1:max(DrivingKin),VelDriveMeanStdevSEM(:,1),VelDriveMeanStdevSEM(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
        title('Mic. Velocity vs Driving Kin.');
        ylabel('Mic. Velocity');
        xlabel('Driving Kinesin');
        set(gca,'FontWeight','b','FontSize',16);
        TotalCount = sum(DriveTotalCounts);
        text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
        ylim([0 max(VelDriveMeanStdevSEM(:,1))*1.2]);
        % Save plot
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Velocity vs Driving.jpg');
        savefig('Aggregate Images\Velocity vs Driving.fig');
        
        % Plot velocity vs hindering kinesin
        figure;
        bar(1:max(HinderingKin),VelHinderMeanStdevSEM(:,1),'FaceColor',HistogramColor,'LineWidth',1.5); hold on;
        errorbar(1:max(HinderingKin),VelHinderMeanStdevSEM(:,1),VelHinderMeanStdevSEM(:,3),'LineStyle','none','Color','k','LineWidth',1.5);
        title('Mic. Velocity vs Hindering Kin.');
        ylabel('Mic. Velocity');
        xlabel('Hindering Kinesin');
        set(gca,'FontWeight','b','FontSize',16);
        TotalCount = sum(HinderTotalCounts);
        text(xText,yText,{['N = ' num2str(TotalCount)]},'FontSize',16,'fontweight','b','Units','normalized');
        ylim([0 max(VelHinderMeanStdevSEM(:,1))*1.2]);
        % Save plot
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Velocity vs Hindering.jpg');
        savefig('Aggregate Images\Velocity vs Hindering.fig');
        
        % Plot TotTotalCounts
        figure;
        bar(1:max(TotalKin),TotTotalCounts,'FaceColor',HistogramColor,'LineWidth',1.5); hold on;
        %plot(1:max(TotalKin),TotalCounts,'LineWidth',1.5);
        title('No. of samples vs Total Kin.');
        set(gca,'FontWeight','b','FontSize',16);
        % Fit exponential
        f = fit((1:max(TotalKin))',TotTotalCounts,'exp1');
        CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
        Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
        h = plot(f,(1:max(TotalKin))',TotTotalCounts);
        set(h,'LineWidth',1.2,'Color','r');
        legend('off');
        TotalCount = sum(TotTotalCounts);
        text(0.25,yText-0.05,{['Kin. const.: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' kin'],...
            ['N = ' num2str(TotalCount)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
        text((1:max(TotalKin)),TotTotalCounts+max(TotTotalCounts)*0.05,num2str(TotTotalCounts),'FontSize',FontSize,'fontweight','b','HorizontalAlignment','center');
        ylabel('No. of samples');
        xlabel('Total Kinesin');
        ylim([0 max(TotTotalCounts)*1.2]);
        % Save TotTotalCounts histogram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\No of Samples vs Total (Velocity).jpg');
        savefig('Aggregate Images\No of Samples vs Total (Velocity).fig');
        
        % Plot DriveTotalCounts
        figure;
        bar(1:max(DrivingKin),DriveTotalCounts,'FaceColor',HistogramColor,'LineWidth',1.5); hold on;
        %plot(1:max(TotalKin),TotalCounts,'LineWidth',1.5);
        title('No. of samples vs Driving Kin.');
        set(gca,'FontWeight','b','FontSize',16);
        % Fit exponential
        f = fit((1:max(DrivingKin))',DriveTotalCounts,'exp1');
        CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
        Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
        h = plot(f,(1:max(DrivingKin))',DriveTotalCounts);
        set(h,'LineWidth',1.2,'Color','r');
        legend('off');
        TotalCount = sum(DriveTotalCounts);
        text(0.25,yText-0.05,{['Kin. const.: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' kin'],...
            ['N = ' num2str(TotalCount)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
        text((1:max(DrivingKin)),DriveTotalCounts+max(DriveTotalCounts)*0.05,num2str(DriveTotalCounts),'FontSize',FontSize,'fontweight','b','HorizontalAlignment','center');
        ylabel('No. of samples');
        xlabel('Driving Kinesin');
        ylim([0 max(DriveTotalCounts)*1.2]);
        % Save DriveTotalCounts histogram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\No of Samples vs Drive (Velocity).jpg');
        savefig('Aggregate Images\No of Samples vs Drive (Velocity).fig');
        
        % Plot HinderTotalCounts
        figure;
        bar(1:max(HinderingKin),HinderTotalCounts,'FaceColor',HistogramColor,'LineWidth',1.5); hold on;
        %plot(1:max(TotalKin),TotalCounts,'LineWidth',1.5);
        title('No. of samples vs Hindering Kin.');
        set(gca,'FontWeight','b','FontSize',16);
        % Fit exponential
        f = fit((1:max(HinderingKin))',HinderTotalCounts,'exp1');
        CI = confint(f,ConfidenceInterval); CI = CI(:,2); CI = -1./CI;
        Error = abs(CI(1)-CI(2))/2; CIMean = mean(CI);
        h = plot(f,(1:max(HinderingKin))',HinderTotalCounts);
        set(h,'LineWidth',1.2,'Color','r');
        legend('off');
        TotalCount = sum(HinderTotalCounts);
        text(0.25,yText-0.05,{['Kin. const.: ' num2str(round(mean(CIMean)*1000)/1000) '  ' num2str(round(Error*1000)/1000) ' kin'],...
            ['N = ' num2str(TotalCount)]},'FontSize',FontSize,'fontweight','b','Units','normalized');
        text((1:max(HinderingKin)),HinderTotalCounts+max(HinderTotalCounts)*0.05,num2str(HinderTotalCounts),'FontSize',FontSize,'fontweight','b','HorizontalAlignment','center');
        ylabel('No. of samples');
        xlabel('Hindering Kinesin');
        ylim([0 max(HinderTotalCounts)*1.2]);
        % Save HinderTotalCounts histogram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\No of Samples vs Hinder (Velocity).jpg');
        savefig('Aggregate Images\No of Samples vs Hinder (Velocity).fig');
            %% Plot velocity of certain total kinesin at varying surface conc
            % Want all microtubule present?
            WantAllMicPresent = 0;      % 1 if we want only points with all microtubule present. 0 otherwise
            
            % Use kinesin availability or density?
            UseAvailability = 1;    % 1 if use kinesin availability. 0 to use just density

            % Plot red line or red curve for center of Gaussian or Gaussian
            % components?
            PlotRedLine = 0;    % 1 if want to plot red line to mark center of Gaussian. 0 otherwise
            PlotRedCurve = 0;    % 1 if want to plot red curve to mark Gaussian components. 0 otherwise
            
            % Get parameters
            AvailabilityGroup = 1;      % Use 1, 2, 3, etc to split graphs based on availability
            xText = 0.12;
            yText = 0.95;
            FontSize = 16;
            binSpacing = 75;
            xLimitVelHist = [-300 1600];
            textFont = 14;
%             HistogramColor = [0.3 0.3 0.8];
%             HistogramColor = [0.3 0.8 0.3];
%             HistogramColor = [0.8 0.8 0.3];
%             HistogramColor = [0.5 0.3 0.6];
            HistogramColor = [0.6 0.5 0.3];
            UseGaussianFit = 1;     % 1 to use Gaussian fit, 0 for none, 2 for using maximum likelihood estimation
            UseStdev = 1;       % 1 to use standard deviation, other values to use SEM
            centerGauss = [0;800];  % Center of Gaussian if UseGaussianFit is chosen
            stdGauss = [20;100];   % Std of Gaussian if UseGaussianFit is chosen
            
            % Get MicLength, AllMicPresent, Velocity, and KinDensity parameter
            MicLength = TimePoints(:,4);
            AllMicPresent = TimePoints(:,5);
            Velocity = TimePoints(:,6);
            KinDensity = TimePoints(:,7);
            
            % Initialize MeanStdevSEM and FractionHinder
            VelTotMeanStdevSEM = zeros(max(TotalKin),3*AvailabilityGroup);
            VelDriveMeanStdevSEM = zeros(max(DrivingKin),3*AvailabilityGroup);
            VelHinderMeanStdevSEM = zeros(max(HinderingKin),3*AvailabilityGroup);
            TotTotalCounts = zeros(max(TotalKin),AvailabilityGroup);
            DriveTotalCounts = zeros(max(DrivingKin),AvailabilityGroup);
            HinderTotalCounts = zeros(max(HinderingKin),AvailabilityGroup);

            if WantAllMicPresent == 1
                if AvailabilityGroup == 1
                    hFig = figure('units','normalized','outerposition',[0.05 0.2 0.4 0.7]);
                else
                    hFig = figure('units','normalized','outerposition',[0.05 0 0.95 0.7]);
                end
                TotalRowNo = max(TotalKin(AllMicPresent==1))-double(length(find(TotalKin(AllMicPresent==1)==max(TotalKin(AllMicPresent==1))))<AvailabilityGroup*2);
            else
                if AvailabilityGroup == 1
                    hFig = figure('units','normalized','outerposition',[0.05 0.2 0.4 0.7]);
                else
                    hFig = figure('units','normalized','outerposition',[0.05 0 0.95 0.7]);
                end
                TotalRowNo = max(TotalKin)-double(length(find(TotalKin==max(TotalKin)))<AvailabilityGroup*2);
%                 ha1 = tight_subplot(TotalRowNo,AvailabilityGroup,[.003 .003],[.08 .05],[.1 .03]);
%                 edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
            end
            
            % Ask for Duplicates to keep
            TitleDialog = cell(1,TotalRowNo);
            StartingValues = cell(1,TotalRowNo);
            for i = 1:TotalRowNo
                TitleDialog(i) = {['Row (' num2str(i) ') (e.g. 1,2,3):']};
                StartingValues(i) = {num2str([])};
            end
            
            if exist('RowGrouping','var')
                StartingValues = RowGrouping;
            end

            % Specify options for inputdlg
            options.Resize = 'off'; options.WindowStyle = 'normal'; options.Interpreter = 'none';

            if TotalRowNo > 10
                % Determine how many times we need to loop
                RowGrouping = cell(1,TotalRowNo);
                for i = 1:ceil(TotalRowNo/10)
                    if i == ceil(TotalRowNo/10)
                        RowGrouping((i-1)*10+1:TotalRowNo) = inputdlg(TitleDialog((i-1)*10+1:TotalRowNo),'Parameters Input',1,StartingValues((i-1)*10+1:TotalRowNo),options);
                    else
                        RowGrouping((i-1)*10+1:i*10) = inputdlg(TitleDialog((i-1)*10+1:i*10),'Parameters Input',1,StartingValues((i-1)*10+1:i*10),options);
                    end
                end
            else
                RowGrouping = inputdlg(TitleDialog,'Parameters Input',1,StartingValues,options);
            end

            RowGroup = RowGrouping(~cellfun('isempty',RowGrouping));
            TotalRowNo = length(RowGroup);
            
            % Start tight subplot
            ha1 = tight_subplot(TotalRowNo,AvailabilityGroup,[.003 .003],[.12 .05],[.1 .03]);
            edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
            
            for i = 1:TotalRowNo
                KinNoGroup = cellfun(@str2num,strsplit(RowGroup{i},','));
                
                if length(KinNoGroup) > 1
                    CurrentVel = []; CurrentMicLength = []; CurrentKinDensity = [];
                    for ii = 1:length(KinNoGroup)
                        if WantAllMicPresent == 0
                            CurVel = Velocity(TotalKin==KinNoGroup(ii));
                        else
                            CurVel = Velocity(TotalKin==KinNoGroup(ii) & AllMicPresent==1);
                        end
                        
                        NotNaNInd = ~isnan(CurVel);
                        CurVel = CurVel(NotNaNInd);
                        CurMicLength = MicLength(TotalKin==KinNoGroup(ii));
                        CurMicLength = CurMicLength(NotNaNInd);
                        CurKinDensity = KinDensity(TotalKin==KinNoGroup(ii));
                        CurKinDensity = CurKinDensity(NotNaNInd);

                        CurrentVel = [CurrentVel; CurVel];
                        CurrentMicLength = [CurrentMicLength; CurMicLength];
                        CurrentKinDensity = [CurrentKinDensity; CurKinDensity];
                    end
                else
                    if WantAllMicPresent == 0
                        CurrentVel = Velocity(TotalKin==KinNoGroup);
                    else
                        CurrentVel = Velocity(TotalKin==KinNoGroup & AllMicPresent==1);
                    end

                    NotNaNInd = ~isnan(CurrentVel);
                    CurrentVel = CurrentVel(NotNaNInd);
                    CurrentMicLength = MicLength(TotalKin==KinNoGroup);
                    CurrentMicLength = CurrentMicLength(NotNaNInd);
                    CurrentKinDensity = KinDensity(TotalKin==KinNoGroup);
                    CurrentKinDensity = CurrentKinDensity(NotNaNInd);
                end

                if UseAvailability == 1
                    Availability = CurrentMicLength.*CurrentKinDensity;
                else
                    Availability = CurrentKinDensity;
                end
                [Availability,AvailabilityInd] = sort(Availability);
                CurrentVel = CurrentVel(AvailabilityInd);
                GroupLength = floor(length(Availability)/AvailabilityGroup);
                GroupInd = 1;

                for j = 1:AvailabilityGroup
                    VelNow = CurrentVel(GroupInd:GroupInd+GroupLength-1);
                    MeanNow = mean(VelNow);
                    StdNow = std(VelNow);
                    SEMNow = std(VelNow)/sqrt(length(VelNow));
                    VelTotMeanStdevSEM(i,(j-1)*AvailabilityGroup+1) = MeanNow;
                    VelTotMeanStdevSEM(i,(j-1)*AvailabilityGroup+2) = StdNow;
                    VelTotMeanStdevSEM(i,(j-1)*AvailabilityGroup+3) = SEMNow;
                    TotTotalCounts(i,j) = length(VelNow);
%                         axes(ha1(AvailabilityGroup*(i-1)+j));
                    axes(ha1(AvailabilityGroup*(TotalRowNo-i)+j));
                    hHist = histogram(VelNow,edges,'FaceColor',HistogramColor); hold on;
                    binCenter = (edges(2:end)+edges(1:end-1))/2;
                    HistData = hHist.Values;
                    
                    if UseGaussianFit == 1
%                         centerGauss = [0;800];
%                         stdGauss = [100;100];
                        [~,IGauss] = min(abs(centerGauss-binCenter),[],2);
                        peakGauss = HistData(IGauss)';

                        p1 = [peakGauss centerGauss stdGauss];
                        f = @(p,x)sum(p(:,1).*exp(-((x-p(:,2))./(sqrt(2)*p(:,3))).^2),1);
                        pfit = lsqcurvefit(f,p1,binCenter,HistData); 
                        f1gauss = @(p,x)p(1)*exp(-((x-p(2))/(sqrt(2)*p(3))).^2);
                        xfit = floor(edges(1)):0.5:ceil(edges(end));
                        yfit = zeros(size(p1,1),length(xfit));
                        AreaGauss = zeros(size(p1,1),1);
                        for ii = 1:size(p1,1)
                            yfit(ii,:) = f1gauss(pfit(ii,:),xfit);
                            plot(xfit,yfit(ii,:),'--k','LineWidth',2);
                            AreaGauss(ii) = sum(f1gauss(pfit(ii,:),binCenter));
                        end
                        
                        sem = pfit(:,3)./sqrt(AreaGauss);
                        if UseStdev == 1; Err = round(pfit(:,3)*10)/10; else; Err = round(sem*10)/10; end; 

                        ylimit = ylim; ylimit = [ylimit(1) ylimit(2)*1.3];
                        
                        xAnnotateOffset = 0;       % x Offset for texts in step-size histogram
                        yAnnotateOffset = 0;       % y Offset for texts in step-size histogram
                        ReportStdev = 1;    % 1 to report Stdev. Any other to report all peaks but no Stdev
                        UseStdev = 1;       % 1 if we want to use Stdev. 0 if we want to use Standard Error of Mean (SEM)
                        AnnotationSize = 14;
                        for iter = 1:size(p1,1)
                            if ReportStdev == 1
                                Text = str2mat([sprintf('%1.0f',pfit(iter,2))  '  ' num2str(round(Err(iter))) ' (' num2str(round(AreaGauss(iter)/sum(AreaGauss)*100)) '%)']);
                                text(pfit(iter,2)+xAnnotateOffset,0.1*ylimit(2)+pfit(iter,1)+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                            else
                                Text = str2mat([sprintf('%1.0f',pfit(iter,2)) ' (' num2str(round(AreaGauss(iter)/sum(AreaGauss)*100)) '%)']);
                                text(pfit(iter,2)+xAnnotateOffset,0.1*ylimit(2)+pfit(iter,1)+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                            end
                        end   
                    elseif UseGaussianFit == 2
                        % Maximum likelihood fit (With repeated fit for one parameter number and choose lowest BIC)
                        OverridePeakNo = 2;     % 0 to let code find the number of peaks. > 0  to define a fixed number of peaks
                        MinPeakNo = 2;      % Decide the minimum peak number if there is no override
                        MaxPeakNo = 4;      % Decide the minimum peak number if there is no override
                        RepeatFit = 4;      % Number of repetition for each peak values for MLE
                        DifferenceThreshold = 1;    % The difference between center of first and second peak below which the fit is rejected. This is to prevent peaks with the same center
                        xAnnotateOffset = 0;       % x Offset for texts in step-size histogram
                        yAnnotateOffset = 0;       % y Offset for texts in step-size histogram
                        ReportStdev = 1;    % 1 to report Stdev. Any other to report all peaks but no Stdev
                        UseStdev = 1;       % 1 if we want to use Stdev. 0 if we want to use Standard Error of Mean (SEM)
                        AnnotationSize = 14;
                        xfit = edges(1):edges(end);
                        Length = length(VelNow');

                        if OverridePeakNo == 0
                            FitLengths = (MaxPeakNo-MinPeakNo+1)*RepeatFit;   % Total number of fits
                            BIC = zeros(FitLengths,2);
                            obj2 = cell(1,FitLengths);
                            FitIndex = 0;
                            for PeakNo = MinPeakNo:MaxPeakNo
                                for Repeat = 1:RepeatFit
                                    try
                                        FitIndex = FitIndex + 1;
                                        obj2{FitIndex} = gmdistribution.fit(VelNow,PeakNo);
                                        BIC(FitIndex,:)= [obj2{FitIndex}.BIC PeakNo];
                                    catch
                                        obj2{FitIndex} = [];
                                        BIC(FitIndex,:)= [Inf PeakNo];
                                    end
                                end
                            end
                            [BICSorted,SortIndex] = sort(BIC(:,1));
                            objSorted = obj2(SortIndex);
                            PeakSorted = BIC(SortIndex,2);
                        else
                            BIC = zeros(RepeatFit,2);
                            obj2 = cell(1,RepeatFit);
                            FitIndex = 0;
                            for Repeat = 1:RepeatFit
                                try
                                    FitIndex = FitIndex + 1;
                                    obj2{FitIndex} = gmdistribution.fit(VelNow,OverridePeakNo);
                                    BIC(FitIndex,:)= [obj2{FitIndex}.BIC OverridePeakNo];
                                catch
                                    obj2{FitIndex} = [];
                                    BIC(FitIndex,:)= [Inf OverridePeakNo];
                                end
                            end
                            [BICSorted,SortIndex] = sort(BIC(:,1));
                            objSorted = obj2(SortIndex);
                            PeakSorted = BIC(SortIndex,2);
                        end

                        if min(BIC(:,1))==Inf
                            BIC = zeros(RepeatFit,2);
                            obj2 = cell(1,RepeatFit);
                            FitIndex = 0;
                            for Repeat = 1:RepeatFit
                                try
                                    FitIndex = FitIndex + 1;
                                    obj2{FitIndex} = gmdistribution.fit(VelNow,1);
                                    BIC(FitIndex,:)= [obj2{FitIndex}.BIC 1];
                                catch
                                    obj2{FitIndex} = [];
                                    BIC(FitIndex,:)= [Inf 1];
                                end
                            end
                            [BICSorted,SortIndex] = sort(BIC(:,1));
                            objSorted = obj2(SortIndex);
                            PeakSorted = BIC(SortIndex,2);
                        end

                        PeakToTry = 1;
                        NoError = 0;
                        while NoError == 0
                            try
                                obj = objSorted{PeakToTry};
                                Amplitude = zeros(PeakSorted(PeakToTry),1);
                                for iter = 1:PeakSorted(PeakToTry)
                                    Amplitude(iter) = pdf(obj,obj.mu(iter))*Length;
                                end

                                [~,PeakMax] = max(Amplitude);
                                [Amplitude,AmpInd]=sort(Amplitude,'descend');
                                Peaks = obj.mu; Peaks = Peaks(AmpInd);
                                Stdev = permute(obj.Sigma,[3 2 1]); Stdev = Stdev(AmpInd);

                                if PeakSorted(PeakToTry) > 1
                                    if abs(Peaks(2)-Peaks(1)) > DifferenceThreshold
                                        NoError = 1;
                                    else
                                        PeakToTry = PeakToTry + 1;
                                    end
                                else
                                    NoError = 1;
                                end
                            catch
                                PeakToTry = PeakToTry + 1;
                            end
                        end

                        Peaks = obj.mu;
                        Stdev = permute(sqrt(obj.Sigma),[3 2 1]);
                        Amplitude = obj.PComponents*Length./(Stdev'*sqrt(2*pi))*binSpacing;
                        [Amplitude,AmpInd]=sort(Amplitude,'descend');
                        Peaks = Peaks(AmpInd);
                        Stdev = Stdev(AmpInd);

                        yfit = (pdf(obj,xfit')*Length)';
                        f1gauss = @(p,x)p(1)*exp(-((x-p(2))/(sqrt(2)*p(3))).^2);
                        NoOfPeaks = length(Peaks);
                        nGauss = Amplitude'.*Stdev*sqrt(2*pi);
                        sem = Stdev./sqrt(nGauss);

                        % Append with zeros if there are less than 3 peaks
                        if NoOfPeaks < 3
                            Peaks = [Peaks; zeros(3-NoOfPeaks,1)];
                            nGauss = [nGauss; zeros(3-NoOfPeaks,1)];
                            Stdev = [Stdev; zeros(3-NoOfPeaks,1)];
                            sem = [sem; zeros(3-NoOfPeaks,1)];
                        end

                        plot(xfit,yfit*binSpacing,'--k','LineWidth',1);
                        yfitgauss = zeros(length(xfit),NoOfPeaks);
                        if UseStdev == 1; Err = round(Stdev*10)/10; else; Err = round(sem*10)/10; end; 

                        ylimit = ylim; ylimit = [ylimit(1) ylimit(2)*1.3];
                        for iter = 1:NoOfPeaks
                            yfitgauss(:,iter) = f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],xfit);
                            if PlotRedCurve == 1
                                plot(xfit,yfitgauss(:,iter),'--r','LineWidth',1);
                            end
                            if PlotRedLine == 1
                                plot([Peaks(iter) Peaks(iter)],ylimit,'r','LineWidth',1);
                            end
                            if ReportStdev == 1
                                Text = str2mat([sprintf('%1.0f',Peaks(iter))  '  ' num2str(round(Err(iter))) ' (' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                                text(Peaks(iter)+xAnnotateOffset,0.2*ylimit(2)+f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                            else
                                Text = str2mat([sprintf('%1.0f',Peaks(iter)) ' (' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                                text(Peaks(iter)+xAnnotateOffset,0.2*ylimit(2)+f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                            end
                        end   
                    end
                    
                    if UseAvailability == 1
                        text(0.05,0.8,{['Av. = ' num2str(round(mean(Availability(GroupInd:GroupInd+GroupLength-1)))) ' nm^-^1'],...
                            ['N = ' num2str(length(VelNow))]},'FontSize',textFont,'fontweight','b','Units','normalized');
                    else
                        text(0.05,0.8,{['Dns. = ' num2str(round(mean(Availability(GroupInd:GroupInd+GroupLength-1))*100)/100) ' nm^-^2'],...
                            ['N = ' num2str(length(VelNow))]},'FontSize',textFont,'fontweight','b','Units','normalized');
                    end
                    xlim(xLimitVelHist);

                    ylim(ylimit);
                    if j == 1
                        if length(KinNoGroup) > 1
                            ylabel([num2str(KinNoGroup(1)) ' - ' num2str(KinNoGroup(end))]);
                        else
                            ylabel(KinNoGroup);
                        end
                    end

                    if i == TotalRowNo
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                        if j == round(AvailabilityGroup/2)
                            title('Vel. Hist. at Varying Total Kin.');
                        end
                    elseif i == 1
                        set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                        if j == round(AvailabilityGroup/2)
                            xlabel('Velocity (nm/s)');
                        end
                    else
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    end

                    GroupInd = GroupInd + GroupLength;
                    
                end
            end

            % Save ha1, ha2 and ha3
            axes(ha1);
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Total Kin (1 group).jpg');
            savefig('Aggregate Images\Velocity histogram at Varying Total Kin (1 group).fig');
%             print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Total Kin.jpg');
%             savefig('Aggregate Images\Velocity histogram at Varying Total Kin.fig');
            %% Plot velocity vs drive-hinder number, distanceForce and force
            % Want all microtubule present?
            WantAllMicPresent = 0;      % 1 if we want only points with all microtubule present. 0 otherwise

            % Get parameters
            xText = 0.1;
            yText = 0.95;
            FontSize = 16;
            binSpacing = 50;
            xLimitVelHist = [-200 1600];
            HistogramColor = [0.3 0.8 0.3];

            % Get Velocity and AllMicPresent parameter
            Velocity = TimePoints(:,6);
            AllMicPresent = TimePoints(:,5);
            
            % Get DriveMinusHinderNo, DriveMinusHinderDistForce, and DriveMinusHinderForce
            DriveMinusHinderNo = DrivingKin - HinderingKin;
            DriveMinusHinderNo = [DriveMinusHinderNo Velocity];
            DriveMinusHinderNo(isnan(Velocity),:) = [];
            DistanceForces = TimePoints(:,8:size(TimePoints,2));
            DriveMinusHinderDistForce = nansum(DistanceForces,2);
            DriveMinusHinderDistForce = [DriveMinusHinderDistForce Velocity];
            DriveMinusHinderDistForce(isnan(Velocity),:) = [];
            DriveMinusHinderForce = nansum(Forces,2);
            DriveMinusHinderForce = [DriveMinusHinderForce Velocity];
            DriveMinusHinderForce(isnan(Velocity),:) = [];
            
%             plot(DriveMinusHinderNo(:,1),DriveMinusHinderNo(:,2),'*');
%             plot(DriveMinusHinderDistForce(:,1),DriveMinusHinderDistForce(:,2),'*');
%             plot(DriveMinusHinderForce(:,1),DriveMinusHinderForce(:,2),'*');
            
%             % Try sorting
%             MeanGrouping = 10;
%             SortedDriveMinusHinderDistForce = sortrows(DriveMinusHinderDistForce,1);
%             SortedDriveMinusHinderDistForce = SortedDriveMinusHinderDistForce(1:floor(length(SortedDriveMinusHinderDistForce)/MeanGrouping)*MeanGrouping,:);
%             SortedDriveMinusHinderDistForce = [mean(reshape(SortedDriveMinusHinderDistForce(:,1),MeanGrouping,[]),1)' mean(reshape(SortedDriveMinusHinderDistForce(:,2),MeanGrouping,[]),1)'];
%             figure; plot(SortedDriveMinusHinderDistForce(:,1),SortedDriveMinusHinderDistForce(:,2),'*');
%             
%             [SortedDriveMinusHinderForce,SortInd] = sortrows(DriveMinusHinderForce,1);
%             SortedDriveMinusHinderForce = SortedDriveMinusHinderForce(1:floor(length(SortedDriveMinusHinderForce)/MeanGrouping)*MeanGrouping,:);
%             SortedDriveMinusHinderForce = [mean(reshape(SortedDriveMinusHinderForce(:,1),MeanGrouping,[]),1)' mean(reshape(SortedDriveMinusHinderForce(:,2),MeanGrouping,[]),1)'];
%             figure; plot(SortedDriveMinusHinderForce(:,1),SortedDriveMinusHinderForce(:,2),'*');
            
            % Get Velocity Histogram at Varying Net Kin No.
            xBinSize = 2;
            yBinSize = 80;
            StuckThreshold = 250;       % Velocity in nm/s below which microtubule is considered stuck
            
            xCenters = floor(min(DriveMinusHinderNo(:,1))/xBinSize)*xBinSize:xBinSize:ceil(max(DriveMinusHinderNo(:,1))/xBinSize)*xBinSize;
            yCenters = floor(min(DriveMinusHinderNo(:,2))/yBinSize)*yBinSize:yBinSize:ceil(max(DriveMinusHinderNo(:,2))/yBinSize)*yBinSize;
            StuckIndices = find(yCenters<StuckThreshold);
            n = hist3([DriveMinusHinderNo(:,1) round(DriveMinusHinderNo(:,2))],{xCenters yCenters});
            HistMax = max(max(n))*1.05;
            
            figure('units','normalized','outerposition',[0.2 0 0.5 1]);
            ha = tight_subplot(length(xCenters),1,[.02 .05],[.14 .06],[.10 .03]);
            
            for i = 1:length(xCenters)
                axes(ha(i));
                bar(yCenters,n(i,:));
%                 ylim([0 HistMax]);
                if i == length(xCenters)
                    set(gca,'FontSize',13,'fontweight','b','YTick',[]);
                    xlabel('Velocity (nm/s)');
                elseif i == 1
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                    title('Velocity Histogram at Varying Net Kinesin No.');
                elseif i == round(length(xCenters)/2)
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                    hText = text(-0.09,0.1,'Net Kinesin No.','Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                    set(hText,'Rotation',90)
                else
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                end
                
                
                StuckPercentage = round(sum(n(i,StuckIndices))/sum(n(i,:))*1000)/10;
                text(0.2,0.7,['Stuck = ' num2str(StuckPercentage) '%'],'Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                text(0.87,0.7,['N = ' num2str(sum(n(i,:)))],'Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                ylabel(num2str(xCenters(i)));
            end

            % Save figure
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\Velocity Hist at Varying Net Kin No.jpg');
            savefig('Aggregate Images\Velocity Hist at Varying Net Kin No.fig');

            
            % Get Velocity Histogram at Varying Net Distance Force
            xBinSize = 700;
            yBinSize = 80;
            
            xCenters = floor(min(DriveMinusHinderDistForce(:,1))/xBinSize)*xBinSize:xBinSize:ceil(max(DriveMinusHinderDistForce(:,1))/xBinSize)*xBinSize;
            yCenters = floor(min(DriveMinusHinderDistForce(:,2))/yBinSize)*yBinSize:yBinSize:ceil(max(DriveMinusHinderDistForce(:,2))/yBinSize)*yBinSize;
            StuckIndices = find(yCenters<StuckThreshold);
            n = hist3([DriveMinusHinderDistForce(:,1) round(DriveMinusHinderDistForce(:,2))],{xCenters yCenters});
            HistMax = max(max(n))*1.05;
            
            figure('units','normalized','outerposition',[0.2 0 0.5 1]);
            ha = tight_subplot(length(xCenters)-2,1,[.02 .05],[.14 .06],[.10 .03]);
            
            for i = 1:length(xCenters)-2
                axes(ha(i));
                bar(yCenters,n(i+1,:));
%                 ylim([0 HistMax]);
                if i == length(xCenters)-2
                    set(gca,'FontSize',13,'fontweight','b','YTick',[]);
                    xlabel('Velocity (nm/s)');
                elseif i == 1
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                    title('Velocity Histogram at Varying Net Distance Force');
                elseif i == round((length(xCenters)-1)/2)
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                    hText = text(-0.09,0.1,'Net Distance Force (nm)','Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                    set(hText,'Rotation',90)
                else
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                end
                
                StuckPercentage = round(sum(n(i+1,StuckIndices))/sum(n(i+1,:))*1000)/10;
                text(0.2,0.7,['Stuck = ' num2str(StuckPercentage) '%'],'Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                text(0.87,0.7,['N = ' num2str(sum(n(i+1,:)))],'Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                ylabel(num2str(xCenters(i+1)));
            end
            
            % Save figure
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\Velocity Hist at Varying Net Distance Force.jpg');
            savefig('Aggregate Images\Velocity Hist at Varying Net Distance Force.fig');

            
%             figure('units','normalized','outerposition',[0.2 0.2 0.5 0.7]);
%             n = hist3([DriveMinusHinderDistForce(:,1) round(DriveMinusHinderDistForce(:,2))],[xBin yBin]);
%             n1 = n';
%             n1(size(n,2) + 1, size(n,1) + 1) = 0;
%             xb = linspace(min(DriveMinusHinderDistForce(:,1)),max(DriveMinusHinderDistForce(:,1)),size(n,1)+1);
%             yb = linspace(min(round(DriveMinusHinderDistForce(:,2))),max(round(DriveMinusHinderDistForce(:,2))),size(n,2)+1);
%             h = pcolor(xb,yb,n1);
%             colormap(hot) % heat map
%             ylabel('Velocity (nm/s)');
%             title('Velocity vs Net Distance Force');
%             set(gca,'FontSize',13,'fontweight','b');
%             caxis([min(min(n1)) max(max(n1))/20]);


            % Get Velocity Histogram at Varying Net Force
            xBinSize = 7;
            yBinSize = 80;
            
            xCenters = floor(min(DriveMinusHinderForce(:,1))/xBinSize)*xBinSize:xBinSize:ceil(max(DriveMinusHinderForce(:,1))/xBinSize)*xBinSize;
            yCenters = floor(min(DriveMinusHinderForce(:,2))/yBinSize)*yBinSize:yBinSize:ceil(max(DriveMinusHinderForce(:,2))/yBinSize)*yBinSize;
            StuckIndices = find(yCenters<StuckThreshold);
            n = hist3([DriveMinusHinderForce(:,1) round(DriveMinusHinderForce(:,2))],{xCenters yCenters});
            HistMax = max(max(n))*1.05;
            
            figure('units','normalized','outerposition',[0.2 0 0.5 1]);
            ha = tight_subplot(length(xCenters)-2,1,[.02 .05],[.14 .06],[.10 .03]);
            
            for i = 1:length(xCenters)-2
                axes(ha(i));
                bar(yCenters,n(i+1,:));
%                 ylim([0 HistMax]);
                if i == length(xCenters)-2
                    set(gca,'FontSize',13,'fontweight','b','YTick',[]);
                    xlabel('Velocity (nm/s)');
                elseif i == 1
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                    title('Velocity Histogram at Varying Net Force');
                elseif i == round((length(xCenters)-1)/2)
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                    hText = text(-0.09,0.1,'Net Force (pN)','Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                    set(hText,'Rotation',90)
                else
                    set(gca,'FontSize',13,'fontweight','b','XTick',[],'YTick',[]);
                end
                
                StuckPercentage = round(sum(n(i+1,StuckIndices))/sum(n(i+1,:))*1000)/10;
                text(0.2,0.7,['Stuck = ' num2str(StuckPercentage) '%'],'Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                text(0.87,0.7,['N = ' num2str(sum(n(i+1,:)))],'Color','black','FontSize',15,'fontweight','b','Units','normalized','HorizontalAlignment','center','VerticalAlignment','middle'); hold on;
                ylabel(num2str(xCenters(i+1)));
            end
            
            % Save figure
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\Velocity Hist at Varying Net Force.jpg');
            savefig('Aggregate Images\Velocity Hist at Varying Net Force.fig');

            
            
            
            
            
            
%             % Initialize MeanStdevSEM and FractionHinder
%             VelTotMeanStdevSEM = zeros(max(TotalKin),3);
%             VelDriveMeanStdevSEM = zeros(max(DrivingKin),3);
%             VelHinderMeanStdevSEM = zeros(max(HinderingKin),3);
%             TotTotalCounts = zeros(max(TotalKin),1);
%             DriveTotalCounts = zeros(max(DrivingKin),1);
%             HinderTotalCounts = zeros(max(HinderingKin),1);
% 
%             if WantAllMicPresent == 1
%                 figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
%                 ha1 = tight_subplot(max(TotalKin(AllMicPresent==1)),1,[.003 .003],[.08 .05],[.1 .03]);
%                 edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
%                 figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
%                 ha2 = tight_subplot(max(DrivingKin(AllMicPresent==1)),1,[.003 .003],[.08 .05],[.1 .03]);
%                 figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
%                 ha3 = tight_subplot(max(HinderingKin(AllMicPresent==1)),1,[.003 .003],[.08 .05],[.1 .03]);
%             else
%                 figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
%                 ha1 = tight_subplot(max(TotalKin),1,[.003 .003],[.08 .05],[.1 .03]);
%                 edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
%                 figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
%                 ha2 = tight_subplot(max(DrivingKin),1,[.003 .003],[.08 .05],[.1 .03]);
%                 figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
%                 ha3 = tight_subplot(max(HinderingKin),1,[.003 .003],[.08 .05],[.1 .03]);
%             end
% 
%             for i = 1:max(TotalKin)
%                 if WantAllMicPresent == 0
%                     CurrentVel = Velocity(TotalKin==i);
%                     CurrentVel = CurrentVel(~isnan(CurrentVel));
%                     VelTotMeanStdevSEM(i,1) = mean(CurrentVel);
%                     VelTotMeanStdevSEM(i,2) = std(CurrentVel);
%                     VelTotMeanStdevSEM(i,3) = std(CurrentVel)/sqrt(length(CurrentVel));
%                     TotTotalCounts(i) = length(CurrentVel);
%                     axes(ha1(i));
%                     histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
%                     text(0.2,0.7,{['Mean: ' num2str(round(mean(VelTotMeanStdevSEM(i,1)))) '  ' num2str(round(VelTotMeanStdevSEM(i,3))) ' nm/s'],...
%                         ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
%                     xlim(xLimitVelHist);
%                     ylimit = ylim; plot([VelTotMeanStdevSEM(i,1) VelTotMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
%                     ylim(ylimit);
%                     ylabel(i);
%                     if i == max(TotalKin)
%                         set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
%                         xlabel('Velocity (nm/s)');
%                     elseif i == 1
%                         set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
%                         title('Vel. Hist. at Varying Total Kin.');
%                     else
%                         set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
%                     end
%                 else
%                     CurrentVel = Velocity(TotalKin==i & AllMicPresent==1);
%                     CurrentVel = CurrentVel(~isnan(CurrentVel));
%                     VelTotMeanStdevSEM(i,1) = mean(CurrentVel);
%                     VelTotMeanStdevSEM(i,2) = std(CurrentVel);
%                     VelTotMeanStdevSEM(i,3) = std(CurrentVel)/sqrt(length(CurrentVel));
%                     TotTotalCounts(i) = length(CurrentVel);
% 
%                     axes(ha1(i));
%                     histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
%                     text(0.2,0.7,{['Mean: ' num2str(round(mean(VelTotMeanStdevSEM(i,1)))) '  ' num2str(round(VelTotMeanStdevSEM(i,3))) ' nm/s'],...
%                         ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
%                     xlim(xLimitVelHist);
%                     ylimit = ylim; plot([VelTotMeanStdevSEM(i,1) VelTotMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
%                     ylim(ylimit);
%                     ylabel(i);
%                     if i == max(TotalKin(AllMicPresent==1))
%                         set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
%                         xlabel('Velocity (nm/s)');
%                     elseif i == 1
%                         set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
%                         title('Vel. Hist. at Varying Total Kin.');
%                     else
%                         set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
%                     end
%                 end
%             end
% 
%             % Save ha1, ha2 and ha3
%             axes(ha1);
%             if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
%             print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Total Kin.jpg');
%             savefig('Aggregate Images\Velocity histogram at Varying Total Kin.fig');
%             axes(ha2);
%             if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
%             print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Driving Kin.jpg');
%             savefig('Aggregate Images\Velocity histogram at Varying Driving Kin.fig');
%             axes(ha3);
%             if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
%             print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Hindering Kin.jpg');
%             savefig('Aggregate Images\Velocity histogram at Varying Hindering Kin.fig');
            %% Simulate kinesin lost
            % Want all microtubule present?
            WantAllMicPresent = 0;      % 1 if we want only points with all microtubule present. 0 otherwise
            
            % Use kinesin availability or density?
            UseAvailability = 1;    % 1 if use kinesin availability. 0 to use just density

            % Get parameters
            AvailabilityGroup = 1;
            xText = 0.1;
            yText = 0.95;
            FontSize = 16;
            binSpacing = 50;
            xLimitVelHist = [-200 1600];
            textFont = 13;
            HistogramColor = [0.3 0.8 0.3];

            % Get MicLength, AllMicPresent, Velocity, and KinDensity parameter
            TotalKin = TimePoints(:,1);
            MicLength = TimePoints(:,4);
            AllMicPresent = TimePoints(:,5);
            Velocity = TimePoints(:,6);
            KinDensity = TimePoints(:,7);
            
            % Simulate data loss
            LostFraction = 0.7;
            for i = 1:length(TotalKin)
                RandIndex = rand(TotalKin(i),1)>LostFraction; 
                TotalKin(i) = sum(RandIndex);
            end
            RandIndex = find(TotalKin>0);
            TotalKin = TotalKin(RandIndex);
            MicLength = MicLength(RandIndex);
            AllMicPresent = AllMicPresent(RandIndex);
            Velocity = Velocity(RandIndex);
            KinDensity = KinDensity(RandIndex);

            % Initialize MeanStdevSEM and FractionHinder
            VelTotMeanStdevSEM = zeros(max(TotalKin),3*AvailabilityGroup);
            VelDriveMeanStdevSEM = zeros(max(DrivingKin),3*AvailabilityGroup);
            VelHinderMeanStdevSEM = zeros(max(HinderingKin),3*AvailabilityGroup);
            TotTotalCounts = zeros(max(TotalKin),AvailabilityGroup);
            DriveTotalCounts = zeros(max(DrivingKin),AvailabilityGroup);
            HinderTotalCounts = zeros(max(HinderingKin),AvailabilityGroup);

            if WantAllMicPresent == 1
                if AvailabilityGroup == 1
                    figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
                else
                    figure('units','normalized','outerposition',[0.05 0 0.95 0.95]);
                end
                TotalRowNo = max(TotalKin(AllMicPresent==1))-double(length(find(TotalKin(AllMicPresent==1)==max(DrivingKin(AllMicPresent==1))))<AvailabilityGroup*2);
                ha1 = tight_subplot(TotalRowNo,AvailabilityGroup,[.003 .003],[.08 .05],[.1 .03]);
                edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
            else
                if AvailabilityGroup == 1
                    figure('units','normalized','outerposition',[0.05 0 0.4 0.95]);
                else
                    figure('units','normalized','outerposition',[0.05 0 0.95 0.95]);
                end
                TotalRowNo = max(TotalKin)-double(length(find(TotalKin==max(TotalKin)))<AvailabilityGroup*2);
                ha1 = tight_subplot(TotalRowNo,AvailabilityGroup,[.003 .003],[.08 .05],[.1 .03]);
                edges = min(Velocity):binSpacing:max(Velocity)+binSpacing;
            end

            for i = 1:TotalRowNo
                if WantAllMicPresent == 0
                    CurrentVel = Velocity(TotalKin==i);
                    NotNaNInd = ~isnan(CurrentVel);
                    CurrentVel = CurrentVel(NotNaNInd);
                    CurrentMicLength = MicLength(TotalKin==i);
                    CurrentMicLength = CurrentMicLength(NotNaNInd);
                    CurrentKinDensity = KinDensity(TotalKin==i);
                    CurrentKinDensity = CurrentKinDensity(NotNaNInd);
                    if UseAvailability == 1
                        Availability = CurrentMicLength.*CurrentKinDensity;
                    else
                        Availability = CurrentKinDensity;
                    end
                    [Availability,AvailabilityInd] = sort(Availability);
                    CurrentVel = CurrentVel(AvailabilityInd);
                    GroupLength = floor(length(Availability)/AvailabilityGroup);
                    GroupInd = 1;
                    
                    for j = 1:AvailabilityGroup
                        VelNow = CurrentVel(GroupInd:GroupInd+GroupLength-1);
                        MeanNow = mean(VelNow);
                        StdNow = std(VelNow);
                        SEMNow = std(VelNow)/sqrt(length(VelNow));
                        VelTotMeanStdevSEM(i,(j-1)*AvailabilityGroup+1) = MeanNow;
                        VelTotMeanStdevSEM(i,(j-1)*AvailabilityGroup+2) = StdNow;
                        VelTotMeanStdevSEM(i,(j-1)*AvailabilityGroup+3) = SEMNow;
                        TotTotalCounts(i,j) = length(VelNow);
                        axes(ha1(AvailabilityGroup*(i-1)+j));
                        histogram(VelNow,edges,'FaceColor',HistogramColor); hold on;
                        
                        % Maximum likelihood fit (With repeated fit for one parameter number and choose lowest BIC)
                        OverridePeakNo = 0;     % 0 to let code find the number of peaks. > 0  to define a fixed number of peaks
                        MinPeakNo = 1;      % Decide the minimum peak number if there is no override
                        MaxPeakNo = 2;      % Decide the minimum peak number if there is no override
                        RepeatFit = 4;      % Number of repetition for each peak values for MLE
                        DifferenceThreshold = 100;    % The difference between center of first and second peak below which the fit is rejected. This is to prevent peaks with the same center
                        xAnnotateOffset = 0;       % x Offset for texts in step-size histogram
                        yAnnotateOffset = 0;       % y Offset for texts in step-size histogram
                        ReportStdev = 1;    % 1 to report Stdev. Any other to report all peaks but no Stdev
                        UseStdev = 1;       % 1 if we want to use Stdev. 0 if we want to use Standard Error of Mean (SEM)
                        AnnotationSize = 13;
                        xfit = edges(1):edges(end);
                        Length = length(VelNow');

                        if OverridePeakNo == 0
                            FitLengths = (MaxPeakNo-MinPeakNo+1)*RepeatFit;   % Total number of fits
                            BIC = zeros(FitLengths,2);
                            obj2 = cell(1,FitLengths);
                            FitIndex = 0;
                            for PeakNo = MinPeakNo:MaxPeakNo
                                for Repeat = 1:RepeatFit
                                    try
                                        FitIndex = FitIndex + 1;
                                        obj2{FitIndex} = gmdistribution.fit(VelNow,PeakNo);
                                        BIC(FitIndex,:)= [obj2{FitIndex}.BIC PeakNo];
                                    catch
                                        obj2{FitIndex} = [];
                                        BIC(FitIndex,:)= [Inf PeakNo];
                                    end
                                end
                            end
                            [BICSorted,SortIndex] = sort(BIC(:,1));
                            objSorted = obj2(SortIndex);
                            PeakSorted = BIC(SortIndex,2);
                        else
                            BIC = zeros(RepeatFit,2);
                            obj2 = cell(1,RepeatFit);
                            FitIndex = 0;
                            for Repeat = 1:RepeatFit
                                try
                                    FitIndex = FitIndex + 1;
                                    obj2{FitIndex} = gmdistribution.fit(VelNow,OverridePeakNo);
                                    BIC(FitIndex,:)= [obj2{FitIndex}.BIC OverridePeakNo];
                                catch
                                    obj2{FitIndex} = [];
                                    BIC(FitIndex,:)= [Inf OverridePeakNo];
                                end
                            end
                            [BICSorted,SortIndex] = sort(BIC(:,1));
                            objSorted = obj2(SortIndex);
                            PeakSorted = BIC(SortIndex,2);
                        end

                        if min(BIC(:,1))==Inf
                            BIC = zeros(RepeatFit,2);
                            obj2 = cell(1,RepeatFit);
                            FitIndex = 0;
                            for Repeat = 1:RepeatFit
                                try
                                    FitIndex = FitIndex + 1;
                                    obj2{FitIndex} = gmdistribution.fit(VelNow,1);
                                    BIC(FitIndex,:)= [obj2{FitIndex}.BIC 1];
                                catch
                                    obj2{FitIndex} = [];
                                    BIC(FitIndex,:)= [Inf 1];
                                end
                            end
                            [BICSorted,SortIndex] = sort(BIC(:,1));
                            objSorted = obj2(SortIndex);
                            PeakSorted = BIC(SortIndex,2);
                        end

                        PeakToTry = 1;
                        NoError = 0;
                        while NoError == 0
                            try
                                obj = objSorted{PeakToTry};
                                Amplitude = zeros(PeakSorted(PeakToTry),1);
                                for iter = 1:PeakSorted(PeakToTry)
                                    Amplitude(iter) = pdf(obj,obj.mu(iter))*Length;
                                end

                                [~,PeakMax] = max(Amplitude);
                                [Amplitude,AmpInd]=sort(Amplitude,'descend');
                                Peaks = obj.mu; Peaks = Peaks(AmpInd);
                                Stdev = permute(obj.Sigma,[3 2 1]); Stdev = Stdev(AmpInd);

                                if PeakSorted(PeakToTry) > 1
                                    if abs(Peaks(2)-Peaks(1)) > DifferenceThreshold
                                        NoError = 1;
                                    else
                                        PeakToTry = PeakToTry + 1;
                                    end
                                else
                                    NoError = 1;
                                end
                            catch
                                PeakToTry = PeakToTry + 1;
                            end
                        end

                        Peaks = obj.mu;
                        Stdev = permute(sqrt(obj.Sigma),[3 2 1]);
                        Amplitude = obj.PComponents*Length./(Stdev'*sqrt(2*pi))*binSpacing;
                        [Amplitude,AmpInd]=sort(Amplitude,'descend');
                        Peaks = Peaks(AmpInd);
                        Stdev = Stdev(AmpInd);

                        yfit = (pdf(obj,xfit')*Length)';
                        f1gauss = @(p,x)p(1)*exp(-((x-p(2))/(sqrt(2)*p(3))).^2);
                        NoOfPeaks = length(Peaks);
                        nGauss = Amplitude'.*Stdev*sqrt(2*pi);
                        sem = Stdev./sqrt(nGauss);

                        % Append with zeros if there are less than 3 peaks
                        if NoOfPeaks < 3
                            Peaks = [Peaks; zeros(3-NoOfPeaks,1)];
                            nGauss = [nGauss; zeros(3-NoOfPeaks,1)];
                            Stdev = [Stdev; zeros(3-NoOfPeaks,1)];
                            sem = [sem; zeros(3-NoOfPeaks,1)];
                        end

                        plot(xfit,yfit*binSpacing,'-b','LineWidth',2);
                        yfitgauss = zeros(length(xfit),NoOfPeaks);
                        if UseStdev == 1; Err = round(Stdev*10)/10; else; Err = round(sem*10)/10; end; 
                        
                        ylimit = ylim; ylimit = [ylimit(1) ylimit(2)*1.3];
                        for iter = 1:NoOfPeaks
                            yfitgauss(:,iter) = f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],xfit);
                            plot(xfit,yfitgauss(:,iter),'--r','LineWidth',1);
                            plot([Peaks(iter) Peaks(iter)],ylimit,'r','LineWidth',1);
                            if ReportStdev == 1
                                Text = str2mat([sprintf('%1.0f',Peaks(iter))  '' num2str(round(Err(iter))) '(' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                                text(Peaks(iter)+xAnnotateOffset,1.3*f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                            else
                                Text = str2mat([sprintf('%1.0f',Peaks(iter)) '(' num2str(round(nGauss(iter)/sum(nGauss)*100)) '%)']);
                                text(Peaks(iter)+xAnnotateOffset,1.3*f1gauss([Amplitude(iter) Peaks(iter) Stdev(iter)],Peaks(iter))+yAnnotateOffset,Text,'FontSize',AnnotationSize,'fontweight','b','HorizontalAlignment','center','VerticalAlignment','middle');
                            end
                        end
                        
                        if UseAvailability == 1
                            text(0.68,0.7,{['Av. = ' num2str(round(mean(Availability(GroupInd:GroupInd+GroupLength-1)))) ' nm^-^1'],...
                                ['N = ' num2str(length(VelNow))]},'FontSize',textFont,'fontweight','b','Units','normalized');
                        else
                            text(0.68,0.7,{['Dns. = ' num2str(round(mean(Availability(GroupInd:GroupInd+GroupLength-1))*100)/100) ' nm^-^2'],...
                                ['N = ' num2str(length(VelNow))]},'FontSize',textFont,'fontweight','b','Units','normalized');
                        end
                        xlim(xLimitVelHist);
                        
                        ylim(ylimit);
                        if j == 1
                            ylabel(i);
                        end
                        
                        if i == TotalRowNo
                            set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                            if j == round(AvailabilityGroup/2)
                                xlabel('Velocity (nm/s)');
                            end
                        elseif i == 1
                            set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                            if j == round(AvailabilityGroup/2)
                                title('Vel. Hist. at Varying Total Kin.');
                            end
                        else
                            set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                        end
                        
                        GroupInd = GroupInd + GroupLength;
                    end
                else
                    CurrentVel = Velocity(TotalKin==i & AllMicPresent==1);
                    CurrentVel = CurrentVel(~isnan(CurrentVel));
                    VelTotMeanStdevSEM(i,1) = mean(CurrentVel);
                    VelTotMeanStdevSEM(i,2) = std(CurrentVel);
                    VelTotMeanStdevSEM(i,3) = std(CurrentVel)/sqrt(length(CurrentVel));
                    TotTotalCounts(i) = length(CurrentVel);

                    axes(ha1(i));
                    histogram(CurrentVel,edges,'FaceColor',HistogramColor); hold on;
                    text(0.62,0.7,{['Mean: ' num2str(round(mean(VelTotMeanStdevSEM(i,1)))) '  ' num2str(round(VelTotMeanStdevSEM(i,3))) ' s'],...
                        ['N = ' num2str(length(CurrentVel))]},'FontSize',14,'fontweight','b','Units','normalized');
                    xlim(xLimitVelHist);
                    ylimit = ylim; plot([VelTotMeanStdevSEM(i,1) VelTotMeanStdevSEM(i,1)],ylim,'r','LineWidth',2);
                    ylim(ylimit);
                    ylabel(i);
                    if i == max(TotalKin(AllMicPresent==1))
                        set(gca,'YTick',[],'FontWeight','b','FontSize',FontSize,'TickDir','out');
                        xlabel('Velocity (nm/s)');
                    elseif i == 1
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                        title('Vel. Hist. at Varying Total Kin.');
                    else
                        set(gca,'XTick',[],'YTick',[],'FontWeight','b','FontSize',FontSize);
                    end
                end
            end

            % Save ha1, ha2 and ha3
            axes(ha1);
            if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
            print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Total Kin (70% Loss).jpg');
            savefig('Aggregate Images\Velocity histogram at Varying Total Kin (70% Loss).fig');
%             print('-djpeg','-r300','Aggregate Images\Velocity histogram at Varying Total Kin.jpg');
%             savefig('Aggregate Images\Velocity histogram at Varying Total Kin.fig');

            % Return real TotalKin
            TotalKin = TimePoints(:,1);
        %% i) Plot run length vs total kinesins
        GroupNo = 2;
        
        % Get parameters
        xText = 0.1;
        yText = 0.9;
        FontSize = 16;
        
        % Get KinNo and DisplacementsMat
        KinNo = cell2mat(Displacements(:,2));
        DisplacementsMat = cell2mat(Displacements(:,3))/1000;
        
        % Initialize MeanStdevSEM and FractionHinder
        DispMeanStdSEM = zeros(ceil(max(KinNo)/GroupNo),5);
        
        for i = 1:ceil(max(KinNo)/GroupNo)
            if GroupNo == 1
                CurrentDisp = DisplacementsMat(KinNo==i);
                DispMeanStdSEM(i,1) = i;
                DispMeanStdSEM(i,2) = mean(CurrentDisp);
                DispMeanStdSEM(i,3) = std(CurrentDisp);
                DispMeanStdSEM(i,4) = std(CurrentDisp)/sqrt(length(CurrentDisp));
                DispMeanStdSEM(i,5) = length(CurrentDisp);
            else
                CurrentDisp = DisplacementsMat(KinNo>=1+(i-1)*GroupNo & KinNo<i*GroupNo);
                DispMeanStdSEM(i,1) = (1+(2*i-1)*GroupNo)/2;
                DispMeanStdSEM(i,2) = mean(CurrentDisp);
                DispMeanStdSEM(i,3) = std(CurrentDisp);
                DispMeanStdSEM(i,4) = std(CurrentDisp)/sqrt(length(CurrentDisp));
                DispMeanStdSEM(i,5) = length(CurrentDisp);
            end
        end
        
        % Delete NaN from DispMeanStdSEM
        DispMeanStdSEM = DispMeanStdSEM(~isnan(DispMeanStdSEM(:,2)),:);
        
        % Plot Displacements vs No. of Kinesins
        figure;
        bar(DispMeanStdSEM(:,1),DispMeanStdSEM(:,2),'FaceColor',[0.5 0.5 1],'LineWidth',1.5); hold on;
        errorbar(DispMeanStdSEM(:,1),DispMeanStdSEM(:,2),DispMeanStdSEM(:,4),'LineStyle','none','Color','k','LineWidth',1.5);
        title('Run Length vs No. of Kin.');
        ylabel('Run Length (um)');
        xlabel('No. of Kinesin');
        set(gca,'FontWeight','b','FontSize',16);
        text(xText,yText,{['N = ' num2str(length(DisplacementsMat))]},'FontSize',16,'fontweight','b','Units','normalized');
        % Save figures for number of hindering kinesin vs total kinesin
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Run Length vs Kin No.jpg');
        savefig('Aggregate Images\Run Length vs Kin No.fig');
        %% j) Find durations of each kinesin and microtubule length
        [~,TraceFirstInd,~] = unique(DurDistForces(:,[1 2 3]),'rows','first');
        [~,TraceLastInd,~] = unique(DurDistForces(:,[1 2 3]),'rows','last');
%         TimeFirstInd = DurDistForces(TraceFirstInd,5);
%         TimeLastInd = DurDistForces(TraceLastInd,5);
        
        TraceDurations = zeros(length(TraceFirstInd),1);
        for i = 1:length(TraceFirstInd)
            TraceDuration = DurDistForces(TraceFirstInd(i):TraceLastInd(i),5);
            TraceDurations(i) = max(TraceDuration) - min(TraceDuration);
        end
        
        figure; histogram(TraceDurations);
        xlabel('Time (s)');
        ylabel('Frequency');
        title('Kinesin Attachment Time to Microtubule');
        set(gca,'FontSize',16,'FontWeight','b');
        
%         ylim([0 280]);
        text(0.4,0.8,{['Mean: ' num2str(round(mean(TraceDurations)*10)/10) '  ' num2str(round(std(TraceDurations)*10)/10) ' s'],['N = ' num2str(length(TraceDurations))]},'Units','Normalized','FontSize',16,'FontWeight','b');
        
        % Save figures for kin attachment time histogram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Kin Attachment Time.jpg');
        savefig('Aggregate Images\Kin Attachment Time.fig');
        
        % Plot Microtubule Length
        figure; histogram(TimePoints(:,4));
        xlabel('Microtubule Length (um)');
        ylabel('Frequency');
        title('Microtubule Length Histogram');
        set(gca,'FontSize',16,'FontWeight','b');
        text(0.45,0.8,{['Mean: ' num2str(round(mean(TimePoints(:,4))*10)/10) '  ' num2str(round(std(TimePoints(:,4))*10)/10) ' um'],['N = ' num2str(length(TimePoints(:,4)))]},'Units','Normalized','FontSize',16,'FontWeight','b');
        
        % Save figures for microtubule length histogram
        if exist('Aggregate Images','dir')~=7; mkdir('Aggregate Images'); end
        print('-djpeg','-r300','Aggregate Images\Mic Length Histogram.jpg');
        savefig('Aggregate Images\Mic Length Histogram.fig');
        %% k) No. of driving/ hindering vs microtubule velocity
        Velocity = TimePoints(:,6);
        [SortedVelocity,iVel] = sort(Velocity);
        SortedFractionHinder = HinderingKin./(HinderingKin+DrivingKin);
        SortedFractionHinder = SortedFractionHinder(iVel);
        
        AverageNo = 100;
        MeanVelHinderDrive = zeros(floor(length(Velocity)/AverageNo),2);
        for i = 1:floor(length(Velocity)/AverageNo)
            MeanVelHinderDrive(i,:) = [mean(SortedVelocity((i-1)*AverageNo+1:i*AverageNo)) mean(SortedFractionHinder((i-1)*AverageNo+1:i*AverageNo))]; 
        end
        
        figure; plot(MeanVelHinderDrive(:,1),MeanVelHinderDrive(:,2),'-*','LineWidth',2);
        title('Fraction Hindering vs Velocity');
        ylabel('Fraction Hinder (%)');
        xlabel('Velocity (nm/s)');
        set(gca,'FontSize',14,'FontWeight','bold');
    %% Just plotting
    % Read EXCEL file
    TextSize = 12;

    ResultName = uigetfile('*.xlsx');
    FinalResult = xlsread(ResultName);
    RegionNo = floor((size(FinalResult,2)-3)/3);
    ExposureTime = FinalResult(2,1)-FinalResult(1,1);
    EquilibriumPos = FinalResult(1:RegionNo,end);
    EquilibriumPos = [EquilibriumPos; FinalResult(RegionNo+2:2*RegionNo+1,end)];
    StartEndMatrix = FinalResult(2*RegionNo+3:4*RegionNo+2,end);

    figure('units','normalized','outerposition',[0 0 1 1]);
    subplot(9,1,1);
    plot([0 size(FinalResult,1)*ExposureTime],[0 0],'red','LineWidth',2); hold on;
    plot(FinalResult(:,1),FinalResult(:,3),'black','LineWidth',2);
    xlim([0 size(FinalResult,1)*ExposureTime]);
    if min(FinalResult(:,3))<0
        ylim([min(FinalResult(:,3)) max(FinalResult(:,3))]);
    else
        ylim([0 max(FinalResult(:,3))]);
    end
    ylabel('V (nm/s)');
    title('Microtubule Velocity');
    set(gca,'FontSize',TextSize,'XTick',[],'FontWeight','b');

    % Add certain magnitude to On-Axis and Off-Axis to bring them closer together
    subplot(9,1,3:6);

    for i = 1:RegionNo
        if i == 1
            MaxOnAxisDiff = max(FinalResult(:,3*i+2))-min(FinalResult(:,3*i+2));
            MaxOffAxisDiff = max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3));
        else
            MaxOnAxisDiff = max(MaxOnAxisDiff,max(FinalResult(:,3*i+2))-min(FinalResult(:,3*i+2)));
            MaxOffAxisDiff = max(MaxOffAxisDiff,max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3)));
        end
    end

    for i = 1:RegionNo
        FinalResult(:,3*i+2) = FinalResult(:,3*i+2)-min(FinalResult(:,3*i+2))+((i-1)*MaxOnAxisDiff)+MaxOnAxisDiff*0.05;
        FinalResult(:,3*i+3) = FinalResult(:,3*i+3)-min(FinalResult(:,3*i+3))+((i-1)*MaxOffAxisDiff)+MaxOffAxisDiff*0.05;
    end

    % plot On-Axis Displacement
    cc=hsv(RegionNo)';
    subplot(9,1,3:6);
    LegendName = cell(RegionNo,1);
    for i = 1:RegionNo
        % Plot equilibrium positions
        plot([0 size(FinalResult,1)],[EquilibriumPos(i) EquilibriumPos(i)],':black','LineWidth',2); hold on;

        % Plot trace
        plot(FinalResult(:,3*i+1),FinalResult(:,3*i+2),'Color',cc(:,i),'LineWidth',2,'LineStyle','-');

        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),3*i+1),FinalResult(StartEndMatrix(2*i-1),3*i+2),'pk','LineWidth',2,'MarkerFaceColor','black'); hold on;
        plot(FinalResult(StartEndMatrix(2*i),3*i+1),FinalResult(StartEndMatrix(2*i),3*i+2),'ok','LineWidth',2,'MarkerFaceColor','black'); hold on;
    end

    % Add label for points
    yHeight = ylim; yHeight = yHeight(2)-yHeight(1);
    for i = 1:RegionNo
        LastIndex = find(~isnan(FinalResult(:,3*i+1)),1,'last');
        text(FinalResult(LastIndex,3*i+1)*1.02,FinalResult(LastIndex,3*i+2)+(yHeight/25),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b'); hold on;
%         if i == 1
%             MaxOnAxisDiff = max(FinalResult(:,3*i+1))-min(FinalResult(:,3*i+1));
%             MaxOffAxisDiff = max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3));
%         else
%             MaxOnAxisDiff = max(MaxOnAxisDiff,max(FinalResult(:,3*i+1))-min(FinalResult(:,3*i+1)));
%             MaxOffAxisDiff = max(MaxOffAxisDiff,max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3)));
%         end
    end

    xlim([0 size(FinalResult,1)*ExposureTime]);
    ylim([0 MaxOnAxisDiff*RegionNo*1.05]);
    ylabel('On-Axis Displacement (nm)');
    title('Kinesin On-Axis Displacement');
    set(gca,'XTick',[],'FontSize',TextSize,'FontWeight','b');

%     % Add certain magnitude to Off-Axis so that the order on plot is the same as On-Axis 
%     if FinalResult(1,3*RegionNo+2)<FinalResult(1,5)
%         for i = 1:RegionNo
%             FinalResult(:,3*i+3) = FinalResult(:,3*i+3)-min(FinalResult(:,3*i+3))+((3-i)*MaxOffAxisDiff)+20;
%         end
%     else
%         for i = 1:RegionNo
%             FinalResult(:,3*i+3) = FinalResult(:,3*i+3)-min(FinalResult(:,3*i+3))+((i-1)*MaxOffAxisDiff)+20;
%         end
%     end

%     % Adjust magnitude of the Off-Axis so that it is well-spread apart
%     subplot(9,1,8:9);
%     for i = 1:RegionNo
%         FinalResult(:,3*i+3) = FinalResult(:,3*i+3)-min(FinalResult(:,3*i+3))+((i-1)*MaxOffAxisDiff)+20;
%     end

    subplot(9,1,8:9);
    % Plot Off-Axis Displacement
    for i = 1:RegionNo
        % Plot equilibrium positions
        plot([0 size(FinalResult,1)],[EquilibriumPos(RegionNo+i) EquilibriumPos(RegionNo+i)],':black','LineWidth',2); hold on;

        % Plot trace
        plot(FinalResult(:,3*i+1),FinalResult(:,3*i+3),'Color',cc(:,i),'LineWidth',2,'LineStyle','-');

        % Plot the start and end points
        plot(FinalResult(StartEndMatrix(2*i-1),3*i+1),FinalResult(StartEndMatrix(2*i-1),3*i+3),'pk','LineWidth',2,'MarkerFaceColor','black'); hold on;
        plot(FinalResult(StartEndMatrix(2*i),3*i+1),FinalResult(StartEndMatrix(2*i),3*i+3),'ok','LineWidth',2,'MarkerFaceColor','black'); hold on;
    end

    % Add label for points
    yHeight = MaxOffAxisDiff*3;
    for i = 1:RegionNo
        LastIndex = find(~isnan(FinalResult(:,3*i+1)),1,'last');
        text(FinalResult(LastIndex,3*i+1)*1.02,FinalResult(LastIndex,3*i+3)+(yHeight/25),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b'); hold on;
    end

    xlim([0 size(FinalResult,1)*ExposureTime]);
    ylim([0 MaxOffAxisDiff*RegionNo*1.05]);
    ylabel('Off-Axis Displacement (nm)');
    xlabel('Time (s)');
    title('Kinesin Off-Axis Displacement');
    set(gca,'FontSize',TextSize,'FontWeight','b');
    %% Plot final graph with on-axis zoomed in further
    ResultName = uigetfile('*.xlsx');
    FinalResult = xlsread(ResultName);
    RegionNo = floor((size(FinalResult,2)-3)/3);
    ExposureTime = FinalResult(2,1)-FinalResult(1,1);
    EquilibriumPos = FinalResult(1:RegionNo,end);
    EquilibriumPos = [EquilibriumPos; FinalResult(RegionNo+2:2*RegionNo+1,end)];
    TextSize = 12;

    figure('units','normalized','outerposition',[0 0 1 1]);
    subplot(9,1,1);
    plot(FinalResult(:,1),FinalResult(:,3),'black','LineWidth',2);
    xlim([0 size(FinalResult,1)*ExposureTime]);
    ylim([min(FinalResult(:,3)) max(FinalResult(:,3))]);
    ylabel('V (nm/s)');
    title('Microtubule Velocity');
    set(gca,'FontSize',TextSize,'XTick',[],'FontWeight','b');

    % Add certain magnitude to On-Axis to bring them closer together
    subplot(9,1,3:6);

    for i = 1:RegionNo
        if i == 1
            MaxOnAxisDiff = max(FinalResult(:,3*i+2))-min(FinalResult(:,3*i+2));
            MaxOffAxisDiff = max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3));
        else
            MaxOnAxisDiff = max(MaxOnAxisDiff,max(FinalResult(:,3*i+2))-min(FinalResult(:,3*i+2)));
            MaxOffAxisDiff = max(MaxOffAxisDiff,max(FinalResult(:,3*i+3))-min(FinalResult(:,3*i+3)));
        end
    end

    for i = 1:RegionNo
        EquilibriumPos(i) = EquilibriumPos(i)-min(FinalResult(:,3*i+2))+((i-1)*MaxOnAxisDiff)+20;
        FinalResult(:,3*i+2) = FinalResult(:,3*i+2)-min(FinalResult(:,3*i+2))+((i-1)*MaxOnAxisDiff)+20;
    end

    % plot On-Axis Displacement
    cc=hsv(RegionNo)';
    subplot(9,1,3:6);
    for i = 1:RegionNo
        plot([0 size(FinalResult,1)*ExposureTime],[EquilibriumPos(i) EquilibriumPos(i)],':black','LineWidth',2); hold on;
        plot(FinalResult(:,3*i+1),FinalResult(:,3*i+2),'Color',cc(:,i),'LineWidth',2,'LineStyle','-');
    end

    % Add label for points and find maximum displacement for On and Off-Axis
    yHeight = ylim; yHeight = yHeight(2)-yHeight(1);
    for i = 1:RegionNo
        LastIndex = find(~isnan(FinalResult(:,3*i+1)),1,'last');
        text(FinalResult(LastIndex,3*i+1)*1.02,FinalResult(LastIndex,3*i+2)+(yHeight/25),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b'); hold on;
    end
    xlim([0 size(FinalResult,1)*ExposureTime]);
    ylim([0 MaxOnAxisDiff*RegionNo*1.05]);
    ylabel('On-Axis Displacement (nm)');
    title('Kinesin On-Axis Displacement');
    set(gca,'XTick',[],'FontSize',TextSize,'FontWeight','b');

    subplot(9,1,8:9);
    % Plot Off-Axis Displacement
    for i = 1:RegionNo
        plot([0 size(FinalResult,1)*ExposureTime],[EquilibriumPos(RegionNo+i) EquilibriumPos(RegionNo+i)],':black','LineWidth',2); hold on;
        plot(FinalResult(:,3*i+1),FinalResult(:,3*i+3),'Color',cc(:,i),'LineWidth',2,'LineStyle','-');
    end

    % Add label for points
    yHeight = MaxOffAxisDiff*3;
    for i = 1:RegionNo
        LastIndex = find(~isnan(FinalResult(:,3*i+1)),1,'last');
        text(FinalResult(LastIndex,3*i+1)*1.02,FinalResult(LastIndex,3*i+3)+(yHeight/25),num2str(i),'Color',cc(:,i),'FontSize',18,'fontweight','b'); hold on;
    end

    xlim([0 size(FinalResult,1)*ExposureTime]);
    ylim([0 MaxOffAxisDiff*RegionNo*1.05]);
    ylabel('Off-Axis Displacement (nm)');
    xlabel('Time (s)');
    title('Kinesin Off-Axis Displacement');
    set(gca,'FontSize',TextSize,'FontWeight','b');
    %% Roadblock analysis
    TextSize = 16;
    
    %Loading the kinesin x-y data
    load('TraceMatrix');
    
    disp('Please select the trace of Microtubule quantum dot.');
    MicInputName = uigetfile('*.csv');
    MicInput = xlsread(MicInputName,'D:I');
    
    % Ask for pixel size
    %PixelTime = inputdlg({'Pixel size (nm):','Exposure Time (s):'},'Parameters Input',1);
    %PixelSize = str2num(PixelTime{1}); ExposureTime = str2num(PixelTime{2});
    PixelSize = 160; ExposureTime = 0.2;
    
    %Assign positions
    KinxPos = TraceMatrix{1}(:,2)*PixelSize;
    KinyPos = TraceMatrix{1}(:,3)*PixelSize;
    KintPos= TraceMatrix{1}(:,4);
    
    MicxPos = MicInput(:,3)*PixelSize;
    MicyPos = MicInput(:,4)*PixelSize;
    MictPos = MicInput(:,6)*ExposureTime;
    
    %Taking care of missing values in the position matrix.
    EndFrame=max([KintPos(end)/ExposureTime MictPos(end)/ExposureTime]);
    RefMat=[0:EndFrame];
    KinDiff=setdiff(RefMat,uint8(KintPos/ExposureTime));
    MicDiff=setdiff(RefMat,uint8(MictPos/ExposureTime));
    
    for i=1:length(MicDiff)
        MictPos=[MictPos(1:MicDiff(i));double(MicDiff(i)*ExposureTime);MictPos(MicDiff(i)+1:end)]; 
        MicxPos=[MicxPos(1:MicDiff(i));0;MicxPos(MicDiff(i)+1:end)];
        MicyPos=[MicyPos(1:MicDiff(i));0;MicyPos(MicDiff(i)+1:end)];     
    end 
    
    for i=1:length(KinDiff)
        KintPos=[KintPos(1:KinDiff(i));double(KinDiff(i)*ExposureTime);KintPos(KinDiff(i)+1:end)]; 
        KinxPos=[KinxPos(1:KinDiff(i));0;KinxPos(KinDiff(i)+1:end)];
        KinyPos=[KinyPos(1:KinDiff(i));0;KinyPos(KinDiff(i)+1:end)];     
    end 
    
    
    %Ask for start and end of timepoints for plotting.
    StartEnd = inputdlg({'Start frame:','End frame (s):'},'Parameters Input',1);
    StartT = str2num(StartEnd{1}); EndT = str2num(StartEnd{2});
    
    %Defining the starting and ending frame for the plot.
    KinStart=StartT- (KintPos(1)/ExposureTime);
    KinEnd= EndT- (KintPos(1)/ExposureTime);
    
    MicStart=StartT- (MictPos(1)/ExposureTime);
    MicEnd= EndT- (MictPos(1)/ExposureTime);
    
    %Plotting the animation
    figure;
     
    xlabel('x position','FontSize',12);
    ylabel('y position','FontSize',12);
    h1= animatedline('Color','r','LineStyle','--');
    h2= animatedline('Color','b','LineStyle','--');
    x1=MicxPos(MicStart:MicEnd);
    y1=MicyPos(MicStart:MicEnd);
    x2=KinxPos(KinStart:KinEnd);
    y2=KinyPos(KinStart:KinEnd);
    
    
    
    %Estimating the limit of the axis.
    xmax=max([max(x1) max(x2)]);
    xmin=min([min(x1(find(x1~=0))) min(x2(find(x2~=0)))]);
    ymax=max([max(y1) max(y2)]);
    ymin=min([min(y1(find(y1~=0))) min(y2(find(y2~=0)))]);
    
    set(gca,'xlim',[xmin,xmax],'XLimMode','Manual');
    set(gca,'ylim',[ymin,ymax],'YLimMode','Manual');  
    
    
    %# create AVI object
    nFrames = length(x1);
    vidObj = VideoWriter('RoadBlockClash.avi');
    vidObj.Quality = 100;
    vidObj.FrameRate = 5;
    open(vidObj);
    
    
    
    for k = 1:length(x1)
        if k~=1
            delete(p1)
            delete(p2)
        end
        if x1(k)~=0
            addpoints(h1,x1(k),y1(k));
        end
        if x2(k)~=0
            addpoints(h2,x2(k),y2(k));
        end
        drawnow
        hold on
        if x1(k)~=0
            p1=plot(gca,x1(k),y1(k),'or','MarkerSize',15,'MarkerFaceColor','r'); 
        else 
            p1=plot(gca,x1(k),y1(k),'LineStyle','none');
        end
        hold on
        if x2(k)~=0
            p2=plot(gca,x2(k),y2(k),'ob','MarkerSize',15,'MarkerFaceColor','b');
        else
            p2=plot(gca,x2(k),y2(k),'LineStyle','none');
        end     
        pause(0.2)
        

        drawnow
        writeVideo(vidObj, getframe(gca));
        if k==length(x1)
            saveas(gcf,'KinesinRoadblockTrace.png');
            saveas(gcf,'KinesinRoadblockTrace.fig');
        end 
    end 
    legend([p1 p2],'Roadblock','Kinesin','Orientation','horizontal');
    close(gcf)
%     %Making the movie
    close(vidObj);
    winopen('RoadBlockClash.avi')  % Opening the movie
        

%% Codes from Three Kinesin Gliding
    %% Open tiff files and plot positions
    FileInputName = 'Three dots.tiff';
    info = imfinfo(FileInputName);
    data=imread(FileInputName, 1, 'Info', info);
    num_images = numel(info);       
    dim = [info(1).Height info(1).Width num_images];
    PixelSize = 106.7;  % PixelSize in nm
    PointNo = 3;
    %imshow(imadjust(data)); hold on;

    % Import positions
    Input = xlsread('Spot positions.xlsx');
    Input(:,2:3) = Input(:,2:3)*PixelSize;
    Tracks = cell(PointNo,1);
    for i = 1:PointNo
        Tracks{i}=Input(Input(:,1)==(i-1),:); %Tracks{i}(:,2:3)=Tracks{i}(:,2:3)-Tracks{i}(1,2:3);
        plot(Tracks{i}(:,2),Tracks{i}(:,3)); hold on;
    end

    xline=[(min(Input(:,2))-((max(Input(:,2))-min(Input(:,2)))/10)) max(Input(:,2))];
    dcm_obj = datacursormode;
    %% Calculate Rotated Trace
    Cursor = getCursorInfo(dcm_obj);
    xPoints = [Cursor(1).Position(1) Cursor(2).Position(1)];
    yPoints = [Cursor(1).Position(2) Cursor(2).Position(2)];
    gradient = (yPoints(2)-yPoints(1))/(xPoints(2)-xPoints(1));
    intercept = yPoints(1)-gradient*xPoints(1);
    yline=gradient*xline+intercept;
    plot(xline,yline,'red','LineWidth',2);
    TracksRotated = Tracks;
    Distance = Tracks;

    figure;
    % Find out rotated trace
    for i = 1:PointNo
        x=Tracks{i}(:,2)-xline(1);
        y=Tracks{i}(:,3)-yline(1);
        radius=x.*x+y.*y;
        theta = atan(gradient);
        tanNew = tan(atan(y./x)-theta);
        TracksRotated{i}(:,2) = sqrt(radius./(1+tanNew.*tanNew));
        TracksRotated{i}(:,3) = tanNew.*TracksRotated{i}(:,2);
        Distance{i}(:,2:3)=TracksRotated{i}(:,2:3)-TracksRotated{i}(1,2:3);
        plot(TracksRotated{i}(:,2),TracksRotated{i}(:,3)); hold on;
        %plot(Distance{i}(:,2),Distance{i}(:,3)); hold on;
        dcm_obj = datacursormode;
    end
    %% Plot distances on graph for TracksRotated
    figure;
    MaxIndex = 0;
    LineSize = 3;
    for i = PointNo:-1:1
        plot(TracksRotated{i}(:,4),TracksRotated{i}(:,2),'LineWidth',LineSize); hold on;
        MaxIndex = max([MaxIndex max(TracksRotated{i}(:,4))]);
    end
    xlim([0 MaxIndex]);
    set(gca,'FontSize',20,'FontWeight','b');
    xlabel('Frame'); ylabel('Displacement (nm)');
    title('Motor Displacement over Time');
    % Add legend
    TrackName = cell(PointNo,1);
    for i = 1:PointNo
        %TrackName{i} = ['Spot ' num2str(i)];
        TrackName{i} = [num2str(i)];
    end
    legend(TrackName{3},TrackName{2},TrackName{1});
    dcm_obj = datacursormode;
    %% Draw three lines based on average of six cursors. One line using two cursors
    Cursor = getCursorInfo(dcm_obj);
    for i = 1:PointNo
        % Find out the cursors for the particular track
        CursorData = zeros(2,1);
        CursorIndex = 1;
        for j = 1:PointNo*2
            if Cursor(j).Target.DisplayName==TrackName{i}
                    CursorData(CursorIndex) = [Cursor(j).DataIndex];
                    CursorIndex = CursorIndex + 1;
            end
        end
        CursorData = sort(CursorData);
        AverageLine = mean(TracksRotated{i}(CursorData(1):CursorData(2),2));
        % Plot line
        plot([TracksRotated{i}(1,4) TracksRotated{i}(end,4)],[AverageLine AverageLine],'-k'); hold on;
    end

    %  Save figure
    delete(findall(gcf,'Type','hggroup'));
    print('-djpeg','-r300','Three traces distances.jpg');
    %% Plot Distances
    figure;
    for i = 1:PointNo
        plot(Distance{i}(:,2),Distance{i}(:,3)); hold on;
        dcm_obj = datacursormode;
    end
    %% Adjust center for Distance vector if necessary
    i = 3;
    Cursor = getCursorInfo(dcm_obj);
    Distance{i}(:,2:3)=Distance{i}(:,2:3)-[Cursor(1).Position(1) Cursor(1).Position(2)];
    figure;
    for i = 1:PointNo
        plot(Distance{i}(:,2),Distance{i}(:,3)); hold on;
    end
    %% Plot overlapping distances on graph for Distance vector
    figure;
    for i = 1:3
        plot(Distance{i}(:,4),Distance{i}(:,2)); hold on;
    end
    % Add legend
    TrackName = cell(PointNo,1);
    for i = 1:PointNo
        TrackName{i} = ['Spot ' num2str(i)];
    end
    legend(TrackName{1},TrackName{2},TrackName{3});
    print('-djpeg','-r300','Three traces distances.jpg');
    %% Calculate and plot force
    Force = Distance;
    ForceSign = Distance;
    L0 = 516;   % Contour length in nm
    kT = 4.114; % Boltzmann constant time temperature in pN nm
    P = 45;     % Persistence length in nm
    figure;
    Offset = 2; % Graph offset in pN
    % Convert distance to force
    for i=1:PointNo
        ForceSign{i}(:,2:3) = sign(Force{i}(:,2:3));
        Force{i}(:,2:3) = abs(Force{i}(:,2:3));
        [row,col] = find(Force{i}(:,2:3)>L0*0.90);
        Force{i}(row,col+1)=L0*0.90;

        Force{i}(:,2:3)=(1/4*(1-(Force{i}(:,2:3)/L0)).^(-2)-1/4+(Force{i}(:,2:3)/L0)-0.8*(Force{i}(:,2:3)/L0).^2.15)*kT/P;
        Force{i}(:,2:3)=Force{i}(:,2:3).*ForceSign{i}(:,2:3);
        plot(Force{i}(:,2),Force{i}(:,3)); hold on;
    end

    % Plot forces on graph
    figure;
    for i = 1:PointNo
        plot(Force{i}(:,4),Force{i}(:,2)+Offset*(i-1)); hold on;
    end
    TrackName = cell(PointNo,1);
    MaxIndex = 0;
    for i = 1:PointNo
        TrackName{i} = ['Spot ' num2str(i)];
        MaxIndex = max([MaxIndex max(TracksRotated{i}(:,4))]);
    end
    xlim([0 MaxIndex]);
    legend(TrackName{1},TrackName{2},TrackName{3});
    title('Force vs Time Graph');
    xlabel('Time (frame)');
    ylabel('Force (pN)');

    %  Save figure
    print('-djpeg','-r300','Forces Offset.jpg');
    
    
    
%% Legacy code
    %% Extract duration, mean and maximum forces
    % Import DistanceForces and FinalResult2
    FileName = dir('*(Final Result2).xlsx');
    FileName = FileName.name;
    FinalResult2 = xlsread(FileName);
    FileInitial = strrep(FileName,' (Final Result2).xlsx','');
    
    % Get PixelSize and ExposureTime
    PixelSize = xlsread([FileInitial ' (Variables).xlsx']);
    ExposureTime = PixelSize(2);    % Exposure time in sec
    PixelSize = PixelSize(1);       % PixelSize in nm
    
    RegionNo = round((size(FinalResult2,2)-4)/7);
    FinalResultLength = size(FinalResult2,1);
    
    % Extract DistanceForces
    DistanceForces = zeros(FinalResultLength,RegionNo*2);
    for i = 1:RegionNo
        DistanceForces(:,i) = FinalResult2(:,7+7*(i-1));
        DistanceForces(:,RegionNo+i) = FinalResult2(:,10+7*(i-1));
    end

    % Initialize indexForces
    indexForces = zeros(FinalResultLength,4);
    ForceCount = 0;
    
    for i = 1:RegionNo
        DistanceForce = DistanceForces(:,i);
        indexForce = DistanceForce~=0 & ~isnan(DistanceForce);
        indexForces(:,i) = bwlabel(indexForce);
        ForceCount = ForceCount + max(indexForces(:,i));
    end
    
    % Initialize Duration-Mean-Max-Distance Force
    DurMeanMax = zeros(ForceCount,3);
    ForceCount = 1;
    
    for i = 1:RegionNo
        DistanceForce = sqrt(DistanceForces(:,i).^2+DistanceForces(:,RegionNo+i).^2);
        
        for ii = 1:max(indexForces(:,i))
            indexForce = find(indexForces(:,i)==ii);
            Duration = FinalResult2(indexForce,4+7*(i-1));
            Duration = max(Duration)-min(Duration);
            
            MeanDistanceForce = mean(DistanceForce(indexForce))*mode(sign(DistanceForces(indexForce,i)));
            MaxDistanceForce = max(DistanceForce(indexForce))*mode(sign(DistanceForces(indexForce,i)));
            
            DurMeanMax(ForceCount,:) = [Duration MeanDistanceForce MaxDistanceForce];
            ForceCount = ForceCount + 1;
        end
    end
    
    % Prepare cells for EXCEL spreadsheet
    FirstLine = cell(1,3);
    FirstLine(1,:) = {'Duration (s)','Mean Distance Force (nm)','Max Distance Force (nm)'};
    DurMeanMax = [FirstLine; num2cell(DurMeanMax)];
    
    % Write DurMeanMax to EXCEL file
    xlswrite([FileInitial ' (Dur-Mean-Max).xlsx'],DurMeanMax);
    
    
    %%
    Stop = 0;
    while Stop == 0
        Pixels = rp(i).PixelList;
        [N,edges,bin]=histcounts(Pixels(:,1),'BinMethod','integers');
        Centers = (edges(2:end)+edges(1:end-1))/2;
        DoubledIndex = find(N>1);
        DiffMatrix = zeros(length(DoubledIndex),1);
        DiffCount = 1;
        for ii = DoubledIndex
            yPix = Pixels(Pixels(:,1)==Centers(ii),2);
            yDiff = max(yPix) - min(yPix);
            DiffMatrix(DiffCount) = yDiff;
            DiffCount = DiffCount + 1;
        end
        
        DoubledIndex = DoubledIndex(DiffMatrix>1);
        DiffMatrix = DiffMatrix(DiffMatrix>1);
        
%         yPix1Real = Pixels(Pixels(:,1)==Centers(DoubledIndex(1)-1),2);
%         yPix1 = Pixels(Pixels(:,1)==Centers(DoubledIndex(1)),2);
%         yPix1 = yPix1(1)+1:yPix1(2)-1;
%         yPix2Real = Pixels(Pixels(:,1)==Centers(DoubledIndex(end)+1),2);
%         yPix2 = Pixels(Pixels(:,1)==Centers(DoubledIndex(end)),2);
%         yPix2 = yPix2(1)+1:yPix2(2)-1;
        
%         if ismember(yPix1,yPix1Real)
%             rp = rp(abs([rp.Area])<LengthThreshold);
%             BW(vertcat(rp.PixelIdxList)) = false;
%         else
%             
%         end
        
        if max(DiffMatrix)>1
            Stop = 1;
        end
        i=i+1;
    end
    figure; plot(Pixels(:,1),Pixels(:,2),'*');
    %% Find No. of TimePoints at various number of DrivingKin and Hindering DistanceForce
    DrivingKinThreshold = 1:5;    % Find no. of timepoints with DrivingKin equal or more than this DrivingKinThreshold
    HinderingDistanceThreshold = 100:100:500;   % Find HinderingKin DistanceForce that is equal or more than this HinderingDistanceThreshold
    
    NoOfEvents = zeros(length(DrivingKinThreshold)*length(HinderingDistanceThreshold),1);
    
    for i = 1:length(DrivingKinThreshold)
        for j = 1:length(HinderingDistanceThreshold)
            NoOfEvents(5*(i-1)+j) = length(find(min(TimePoints(DrivingKin>=DrivingKinThreshold(i),8:end),[],2)<=-HinderingDistanceThreshold(j)));
        end
    end
    %% Calculate force for 9 detachment kinesins using xWLC
    load('SortedOnAxisDNAExtensions.mat');
    DNAExtensions = SortedOnAxisDNAExtensions;
    
    ExtraExtension = 60;
    
    for i=1:length(DNAExtensions)
        DNAExtensions{i} = max(DNAExtensions{i}-ExtraExtension,0);
    end
    
    MinForces = cell(length(DNAExtensions),1);
    Forces = cell(length(DNAExtensions),1);
    MaxForces = cell(length(DNAExtensions),1);
    AllMaxForces = zeros(length(DNAExtensions),3);
    AllLastForces = zeros(length(DNAExtensions),3);
    ExposureTime = 0.2;
    Transparency = 0;
    ExtensionUncertainty = 20;      % Extension uncertainty in +/- uncertainty (nm)
    L0 = 532;       % Contour length in nm
    PlotFigure = 1;
    
    % Calculate using XWLC
    for i = 1:length(DNAExtensions)
        TempMinForces = zeros(length(DNAExtensions{i}),1);
        TempMaxForces = zeros(length(DNAExtensions{i}),1);
        TempForces = zeros(length(DNAExtensions{i}),1);
   
        for j = 1:length(DNAExtensions{i})
            TempForces(j) = fzero(@(x)(DNAExtensions{i}(j)/L0-XWLCContour(x)),1);
            TempMinForces(j) = fzero(@(x)(max(DNAExtensions{i}(j)-ExtensionUncertainty,0)/L0-XWLCContour(x)),1);
            TempMaxForces(j) = fzero(@(x)((DNAExtensions{i}(j)+ExtensionUncertainty)/L0-XWLCContour(x)),1);
        end
        
        MinForces{i} = TempMinForces;
        Forces{i} = TempForces;
        MaxForces{i} = TempMaxForces;
        AllMaxForces(i,:) = [max(TempMinForces) max(TempForces) max(TempMaxForces)];
        AllLastForces(i,:) = [TempMinForces(end) TempForces(end) TempMaxForces(end)];
        
        if PlotFigure == 1
            figure;
            shadedErrorBar((1:length(TempForces))*ExposureTime,TempForces',[TempMaxForces TempMinForces]','b',Transparency); hold on;
            xlabel('Time (s)');
            ylabel('Force (pN)');
            title(['Force of Detaching kinesin - ' num2str(i)]);
            set(gca,'FontSize',14,'FontWeight','b');
            xlim([0 ceil(length(TempForces)*1.1*ExposureTime)]);
            print('-djpeg','-r300',['Force of Detaching kinesin - ' num2str(i) '.jpg']);
        end
    end
    %% Extract DistanceForces and compile them then save
    SampleTitle = '3-2 nM';
    
    % Get UniqueID for each  kinesin
    UniqueID = unique(DurDistForces(FirstInd,1:3),'rows');
    
    % Calculate durations and MeanDistForces for all drive/hinder
    MeanDistForces = zeros(length(FirstInd),1);
    MaxDistForces = zeros(length(FirstInd),1);      % Calculate max force of smoothed DistForces
    MeanRunLength = zeros(length(FirstInd),1);      % Mean run length in um
    UniqueIndices = zeros(length(FirstInd),1);
    DistForceDuration = zeros(length(FirstInd),1);
    CompiledDistances = cell(length(FirstInd),5);
    for i = 1:length(MeanDistForces)
        [~,UniqueIndex,~] = intersect(UniqueID,DurDistForces(FirstInd(i),1:3),'rows');
        MeanDistForces(i) = mean(DurDistForces(FirstInd(i):LastInd(i),6));
        SmoothedDistForces = smooth(DurDistForces(FirstInd(i):LastInd(i),6));
        [MaxDistForces(i),MaxDistForceInd] = max(abs(SmoothedDistForces));
        MaxDistForces(i) = sign(SmoothedDistForces(MaxDistForceInd))*MaxDistForces(i);
        TimeInterval = DurDistForces(FirstInd(i):LastInd(i),5);
        TimeInterval = TimeInterval(2:end) - TimeInterval(1:end-1);
        MeanRunLength(i) = sum(TimeInterval.*MicVelocity(FirstInd(i)+1:LastInd(i)))/1000;
        DistForceDuration(i) = DurDistForces(LastInd(i),5) - DurDistForces(FirstInd(i),5);
        UniqueIndices(i) = UniqueIndex;
        CompiledDistances{i,1} = DurDistForces(FirstInd(i):LastInd(i),5:8);
        CompiledDistances{i,2} = MeanDistForces(i);
        CompiledDistances{i,3} = MaxDistForces(i);
        CompiledDistances{i,4} = DistForceDuration(i);
        CompiledDistances{i,5} = UniqueIndex;
    end
    
    DistanceThreshold = 500;
    SelectedIndices = find(MaxDistForces>DistanceThreshold);
    SelectedDistances = cell(length(find(MaxDistForces>DistanceThreshold)),5);
    for i = 1:length(SelectedIndices)
        SelectedDistances{i,1} = CompiledDistances{SelectedIndices(i),1};
        SelectedDistances{i,2} = CompiledDistances{SelectedIndices(i),2};
        SelectedDistances{i,3} = CompiledDistances{SelectedIndices(i),3};
        SelectedDistances{i,4} = CompiledDistances{SelectedIndices(i),4};
        SelectedDistances{i,5} = CompiledDistances{SelectedIndices(i),5};
    end
    
    DistForceVariables = [MeanDistForces MaxDistForces DistForceDuration UniqueIndices];
    xlswrite(['DistForceVariables (' SampleTitle ').xlsx'],[{'MeanDistForce (nm)' 'MaxDistForce (nm)' 'Duration (s)' 'Kinesin Number'};num2cell(DistForceVariables)]);
    save(['CompiledDistances(' SampleTitle ')'],'CompiledDistances');
    save(['SelectedDistances(' SampleTitle ')'],'SelectedDistances');
    %% Compile all DistanceForces
    InputName = dir('*.xlsx');
    dataTemp = [];
    for i = 1:length(InputName)
        % Open EXCEL file
        CurrentData = xlsread(InputName(i).name);
        CurrentData = [CurrentData repmat(i,length(CurrentData),1)];
        dataTemp = [dataTemp;CurrentData];
    end
    
    xlswrite('DistForceVariables (all).xlsx',[{'MeanDistForce (nm)' 'MaxDistForce (nm)' 'Duration (s)' 'Kinesin No.' 'Experiment No.'};num2cell(dataTemp)]);
    %% Analyze DistanceForce Statistics
    load('SortedOnAxisDNAExtensions.mat');
    DNAExtensions = SortedOnAxisDNAExtensions;
    MaxDNAExtensions = cellfun(@max,DNAExtensions);
    
    DistanceForceStats = xlsread('DistForceVariables (all).xlsx');  
    MaxDistanceForceMag = abs(DistanceForceStats(:,2));
    HinderingDistances = DistanceForceStats(DistanceForceStats(:,2)>0,2);
    
    ExtensionToTry = 100:50:700;
    NoOfEvents = zeros(length(ExtensionToTry),6);
    
    for i = 1:length(ExtensionToTry)
        TotalEvent = length(find(MaxDistanceForceMag>ExtensionToTry(i)));
        TotalKinesin = length(unique(DistanceForceStats(find(MaxDistanceForceMag>ExtensionToTry(i)),4:5),'rows'));
        HinderingEvent = length(find(HinderingDistances>ExtensionToTry(i)));
        HinderingKinesin = length(unique(DistanceForceStats(find(HinderingDistances>ExtensionToTry(i)),4:5),'rows'));
        NoOfEvents(i,:) = [ExtensionToTry(i) TotalEvent TotalKinesin HinderingEvent HinderingKinesin length(find(MaxDNAExtensions>ExtensionToTry(i)))];
    end
    
    PercentOverHinderingEvent = NoOfEvents(:,6)./NoOfEvents(:,4)*100;
    PercentOverHinderingKinesin = NoOfEvents(:,6)./NoOfEvents(:,5)*100;
    PercentOverTotalEvent = NoOfEvents(:,6)./NoOfEvents(:,2)*100;
    PercentOverTotalKinesin = NoOfEvents(:,6)./NoOfEvents(:,3)*100;
    
    xlswrite('Summary.xlsx',[{'Extension (>nm)' 'No of Driving & Hindering Event' 'No of Driving & Hindering Kinesin' 'No. of Hindering Event' 'No. of Hindering Kinesin' 'No. of Detached Kinesins' 'Percent over Total Event' 'Percent over Total Kinesin' 'Percent over Hindering Event' 'Percent over Hindering Kinesin'};...
        num2cell([NoOfEvents,PercentOverTotalEvent,PercentOverTotalKinesin,PercentOverHinderingEvent,PercentOverHinderingKinesin])]);
    %% Get SelectedDistances with certain threshold
    DistanceThreshold = 500;
    SelectedIndices = find(MaxDistForces>DistanceThreshold);
    SelectedDistances = cell(length(find(MaxDistForces>DistanceThreshold)),4);
    for i = 1:length(SelectedIndices)
        SelectedDistances{i,1} = CompiledDistances{SelectedIndices(i),1};
        SelectedDistances{i,2} = CompiledDistances{SelectedIndices(i),2};
        SelectedDistances{i,3} = CompiledDistances{SelectedIndices(i),3};
        SelectedDistances{i,4} = CompiledDistances{SelectedIndices(i),4};
    end
    %% Plot SelectedDistances then select two points to assess loading rate
    TraceNo = 26;
    figure; plot(SelectedDistances{TraceNo,1}(:,1),SelectedDistances{TraceNo,1}(:,2),'LineWidth',3);
    datacursormode on;
        %% Label axis and note loading rate. Save image
        dcm_obj=datacursormode;
        Cursor = getCursorInfo(dcm_obj);
        LoadRate = (Cursor(1).Position(2)-Cursor(2).Position(2))/(Cursor(1).Position(1)-Cursor(2).Position(1));
        xlabel('Time (s)');
        ylabel('Distance (nm)');
        title(['Trace: ' num2str(TraceNo) '. Load Rate: ' num2str(round(LoadRate)) ' nm/s']);
        set(gca,'FontSize',14,'FontWeight','b','LineWidth',2);
        print('-djpeg','-r300',['Trace ' num2str(TraceNo) '.jpg']);
        %% Loop through plots and getting loading rate
        GraphTitle = 'SelectedDistances(100 nM)';
        load([GraphTitle '.mat']);
        LoadRates = zeros(length(SelectedDistances),1);
        
        for i = 1:length(SelectedDistances)
            TraceNo = i;
            figure; plot(SelectedDistances{TraceNo,1}(:,1),SelectedDistances{TraceNo,1}(:,2),'LineWidth',3);
%             datacursormode on;
            dcm_obj=datacursormode;
            h = msgbox('Select two points to get loading rate. Click OK when data point is selected.','Get Loading Rate');
            set(h, 'Position', [50 250 280 70]);
            uiwait(h);

            Cursor = getCursorInfo(dcm_obj);
            LoadRate = (Cursor(1).Position(2)-Cursor(2).Position(2))/(Cursor(1).Position(1)-Cursor(2).Position(1));
            xlabel('Time (s)');
            ylabel('Distance (nm)');
            title([GraphTitle ' Trace: ' num2str(TraceNo) '. Load Rate: ' num2str(round(LoadRate)) ' nm/s']);
            set(gca,'FontSize',14,'FontWeight','b','LineWidth',2);
            print('-djpeg','-r300',['Trace ' num2str(TraceNo) '.jpg']);
            LoadRates(i) = LoadRate;
            close all;
        end
        
        save([GraphTitle 'LoadRates'],'LoadRates');
        %% Count number of LoadRates within a certain threshold
        LoadRateThreshold = 450;
        LoadRateCount = length(find(LoadRatesAll < LoadRateThreshold & LoadRatesAll > -LoadRateThreshold));
        